import { toast } from '@/hooks/use-toast';

export interface BackupData {
  version: string;
  timestamp: string;
  user: {
    id: string;
    email: string;
    name: string;
    preferences: any;
  };
  dashboard: {
    widgets: any[];
    layout: any;
    settings: any;
  };
  analytics: {
    data: any[];
    reports: any[];
    insights: any[];
  };
  integrations: {
    connections: any[];
    settings: any[];
  };
  templates: any[];
  customizations: any[];
}

export interface BackupMetadata {
  id: string;
  name: string;
  description?: string;
  timestamp: string;
  size: number;
  version: string;
  type: 'manual' | 'auto' | 'scheduled';
  status: 'completed' | 'failed' | 'in_progress';
}

class BackupService {
  private readonly STORAGE_KEY = 'flashcore_backups';
  private readonly MAX_BACKUPS = 10;
  private readonly BACKUP_VERSION = '1.0.0';

  /**
   * Create a new backup of user data
   */
  async createBackup(name: string, description?: string): Promise<BackupMetadata> {
    try {
      const backupData: BackupData = {
        version: this.BACKUP_VERSION,
        timestamp: new Date().toISOString(),
        user: await this.getUserData(),
        dashboard: await this.getDashboardData(),
        analytics: await this.getAnalyticsData(),
        integrations: await this.getIntegrationsData(),
        templates: await this.getTemplatesData(),
        customizations: await this.getCustomizationsData(),
      };

      const backup: BackupMetadata = {
        id: this.generateBackupId(),
        name,
        description,
        timestamp: backupData.timestamp,
        size: JSON.stringify(backupData).length,
        version: this.BACKUP_VERSION,
        type: 'manual',
        status: 'completed',
      };

      // Store backup data
      await this.storeBackup(backup.id, backupData);
      
      // Update metadata
      await this.addBackupMetadata(backup);

      toast({
        title: "Backup Created",
        description: `Backup "${name}" has been created successfully.`,
      });

      return backup;
    } catch (error) {
      console.error('Backup creation failed:', error);
      toast({
        title: "Backup Failed",
        description: "Failed to create backup. Please try again.",
        variant: "destructive",
      });
      throw error;
    }
  }

  /**
   * Restore data from a backup
   */
  async restoreBackup(backupId: string): Promise<void> {
    try {
      const backupData = await this.getBackupData(backupId);
      if (!backupData) {
        throw new Error('Backup not found');
      }

      // Validate backup data
      this.validateBackupData(backupData);

      // Restore data in order
      await this.restoreUserData(backupData.user);
      await this.restoreDashboardData(backupData.dashboard);
      await this.restoreAnalyticsData(backupData.analytics);
      await this.restoreIntegrationsData(backupData.integrations);
      await this.restoreTemplatesData(backupData.templates);
      await this.restoreCustomizationsData(backupData.customizations);

      toast({
        title: "Restore Completed",
        description: "Your data has been restored successfully.",
      });
    } catch (error) {
      console.error('Restore failed:', error);
      toast({
        title: "Restore Failed",
        description: "Failed to restore backup. Please try again.",
        variant: "destructive",
      });
      throw error;
    }
  }

  /**
   * Export backup to file
   */
  async exportBackup(backupId: string): Promise<void> {
    try {
      const backupData = await this.getBackupData(backupId);
      const metadata = await this.getBackupMetadata(backupId);
      
      if (!backupData || !metadata) {
        throw new Error('Backup not found');
      }

      const exportData = {
        metadata,
        data: backupData,
      };

      const blob = new Blob([JSON.stringify(exportData, null, 2)], {
        type: 'application/json',
      });

      const url = URL.createObjectURL(blob);
      const a = document.createElement('a');
      a.href = url;
      a.download = `flashcore-backup-${metadata.name}-${new Date().toISOString().split('T')[0]}.json`;
      document.body.appendChild(a);
      a.click();
      document.body.removeChild(a);
      URL.revokeObjectURL(url);

      toast({
        title: "Export Successful",
        description: "Backup has been exported to your device.",
      });
    } catch (error) {
      console.error('Export failed:', error);
      toast({
        title: "Export Failed",
        description: "Failed to export backup. Please try again.",
        variant: "destructive",
      });
      throw error;
    }
  }

  /**
   * Import backup from file
   */
  async importBackup(file: File): Promise<BackupMetadata> {
    try {
      const content = await this.readFileAsText(file);
      const importData = JSON.parse(content);

      // Validate import data
      if (!importData.metadata || !importData.data) {
        throw new Error('Invalid backup file format');
      }

      this.validateBackupData(importData.data);

      // Generate new backup ID for imported data
      const newBackupId = this.generateBackupId();
      const metadata: BackupMetadata = {
        ...importData.metadata,
        id: newBackupId,
        timestamp: new Date().toISOString(),
        type: 'manual',
        status: 'completed',
      };

      // Store imported backup
      await this.storeBackup(newBackupId, importData.data);
      await this.addBackupMetadata(metadata);

      toast({
        title: "Import Successful",
        description: `Backup "${metadata.name}" has been imported successfully.`,
      });

      return metadata;
    } catch (error) {
      console.error('Import failed:', error);
      toast({
        title: "Import Failed",
        description: "Failed to import backup. Please check the file format.",
        variant: "destructive",
      });
      throw error;
    }
  }

  /**
   * Get all backup metadata
   */
  async getBackups(): Promise<BackupMetadata[]> {
    try {
      const backups = localStorage.getItem(this.STORAGE_KEY);
      return backups ? JSON.parse(backups) : [];
    } catch (error) {
      console.error('Failed to get backups:', error);
      return [];
    }
  }

  /**
   * Delete a backup
   */
  async deleteBackup(backupId: string): Promise<void> {
    try {
      // Remove backup data
      localStorage.removeItem(`${this.STORAGE_KEY}_${backupId}`);
      
      // Remove from metadata
      const backups = await this.getBackups();
      const filteredBackups = backups.filter(b => b.id !== backupId);
      localStorage.setItem(this.STORAGE_KEY, JSON.stringify(filteredBackups));

      toast({
        title: "Backup Deleted",
        description: "Backup has been deleted successfully.",
      });
    } catch (error) {
      console.error('Delete failed:', error);
      toast({
        title: "Delete Failed",
        description: "Failed to delete backup. Please try again.",
        variant: "destructive",
      });
      throw error;
    }
  }

  /**
   * Schedule automatic backups
   */
  async scheduleBackup(frequency: 'daily' | 'weekly' | 'monthly'): Promise<void> {
    try {
      const schedule = {
        frequency,
        lastBackup: null,
        enabled: true,
      };

      localStorage.setItem('flashcore_backup_schedule', JSON.stringify(schedule));

      toast({
        title: "Backup Scheduled",
        description: `Automatic backups have been scheduled for ${frequency} intervals.`,
      });
    } catch (error) {
      console.error('Schedule backup failed:', error);
      toast({
        title: "Schedule Failed",
        description: "Failed to schedule automatic backups.",
        variant: "destructive",
      });
      throw error;
    }
  }

  // Private helper methods

  private generateBackupId(): string {
    return `backup_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;
  }

  private async getUserData() {
    // Get user data from localStorage or API
    const userData = localStorage.getItem('flashcore_user');
    return userData ? JSON.parse(userData) : {};
  }

  private async getDashboardData() {
    // Get dashboard configuration
    const dashboardData = localStorage.getItem('flashcore_dashboard');
    return dashboardData ? JSON.parse(dashboardData) : {};
  }

  private async getAnalyticsData() {
    // Get analytics data
    const analyticsData = localStorage.getItem('flashcore_analytics');
    return analyticsData ? JSON.parse(analyticsData) : {};
  }

  private async getIntegrationsData() {
    // Get integrations data
    const integrationsData = localStorage.getItem('flashcore_integrations');
    return integrationsData ? JSON.parse(integrationsData) : {};
  }

  private async getTemplatesData() {
    // Get templates data
    const templatesData = localStorage.getItem('flashcore_templates');
    return templatesData ? JSON.parse(templatesData) : [];
  }

  private async getCustomizationsData() {
    // Get customizations data
    const customizationsData = localStorage.getItem('flashcore_customizations');
    return customizationsData ? JSON.parse(customizationsData) : [];
  }

  private async storeBackup(backupId: string, data: BackupData): Promise<void> {
    localStorage.setItem(`${this.STORAGE_KEY}_${backupId}`, JSON.stringify(data));
  }

  private async getBackupData(backupId: string): Promise<BackupData | null> {
    const data = localStorage.getItem(`${this.STORAGE_KEY}_${backupId}`);
    return data ? JSON.parse(data) : null;
  }

  private async addBackupMetadata(metadata: BackupMetadata): Promise<void> {
    const backups = await this.getBackups();
    backups.unshift(metadata);
    
    // Keep only the latest MAX_BACKUPS
    if (backups.length > this.MAX_BACKUPS) {
      backups.splice(this.MAX_BACKUPS);
    }
    
    localStorage.setItem(this.STORAGE_KEY, JSON.stringify(backups));
  }

  private async getBackupMetadata(backupId: string): Promise<BackupMetadata | null> {
    const backups = await this.getBackups();
    return backups.find(b => b.id === backupId) || null;
  }

  private validateBackupData(data: BackupData): void {
    if (!data.version || !data.timestamp || !data.user) {
      throw new Error('Invalid backup data structure');
    }
  }

  private async restoreUserData(userData: any): Promise<void> {
    localStorage.setItem('flashcore_user', JSON.stringify(userData));
  }

  private async restoreDashboardData(dashboardData: any): Promise<void> {
    localStorage.setItem('flashcore_dashboard', JSON.stringify(dashboardData));
  }

  private async restoreAnalyticsData(analyticsData: any): Promise<void> {
    localStorage.setItem('flashcore_analytics', JSON.stringify(analyticsData));
  }

  private async restoreIntegrationsData(integrationsData: any): Promise<void> {
    localStorage.setItem('flashcore_integrations', JSON.stringify(integrationsData));
  }

  private async restoreTemplatesData(templatesData: any[]): Promise<void> {
    localStorage.setItem('flashcore_templates', JSON.stringify(templatesData));
  }

  private async restoreCustomizationsData(customizationsData: any[]): Promise<void> {
    localStorage.setItem('flashcore_customizations', JSON.stringify(customizationsData));
  }

  private readFileAsText(file: File): Promise<string> {
    return new Promise((resolve, reject) => {
      const reader = new FileReader();
      reader.onload = (e) => resolve(e.target?.result as string);
      reader.onerror = reject;
      reader.readAsText(file);
    });
  }
}

export const backupService = new BackupService(); 