const API_BASE_URL = import.meta.env.DEV ? '/backend/api' : '/fcorex/backend/api';

interface ApiResponse<T> {
  data?: T;
  error?: string;
}

class ApiService {
  private token: string | null = null;

  setToken(token: string | null) {
    this.token = token;
  }

  private async request<T>(endpoint: string, options: RequestInit = {}): Promise<ApiResponse<T>> {
    const url = `${API_BASE_URL}${endpoint}`;

    const headers = {
      'Content-Type': 'application/json',
      ...(this.token && { 'Authorization': `Bearer ${this.token}` }),
      ...options.headers,
    };

    try {
      const response = await fetch(url, {
        ...options,
        headers,
      });

      const contentType = response.headers.get('content-type');
      let data: any;

      if (contentType && contentType.includes('application/json')) {
        data = await response.json();
      } else {
        const text = await response.text();
        if (!response.ok) {
          return { error: `Server error (${response.status}): ${text.substring(0, 100)}` };
        }
        return { error: 'Invalid response format: expected JSON' };
      }

      if (!response.ok) {
        return { error: data.error || data.message || `Request failed with status ${response.status}` };
      }

      return { data };
    } catch (error) {
      console.error(`API Request Error [${endpoint}]:`, error);
      return { error: error instanceof Error ? error.message : 'A network error occurred. Please check your connection.' };
    }
  }

  // Auth methods
  async login(email: string, password: string) {
    return this.request<{ user: any; profile: any; session_token: string }>('/auth.php/login', {
      method: 'POST',
      body: JSON.stringify({ email, password }),
    });
  }

  async register(email: string, password: string, firstName?: string, lastName?: string) {
    return this.request<{ user: any; profile: any; session_token: string }>('/auth.php/register', {
      method: 'POST',
      body: JSON.stringify({ email, password, firstName, lastName }),
    });
  }

  async logout() {
    return this.request<{ message: string }>('/auth.php/logout', {
      method: 'POST',
    });
  }

  // Dashboard methods
  async getDashboardData() {
    return this.request<any>('/dashboard-data.php');
  }

  async getSubscriptionStatus() {
    return this.request<any>('/subscription-status.php');
  }

  async logActivity(action: string, details?: any) {
    return this.request<any>('/dashboard-logs.php', {
      method: 'POST',
      body: JSON.stringify({ action, details }),
    });
  }

  // Admin methods
  async adminManageLicenses(action: string, payload: any = {}) {
    return this.request<any>('/admin-manage-licenses.php', {
      method: 'POST',
      body: JSON.stringify({ action, ...payload }),
    });
  }

  async adminManageUsers(action: string, payload: any = {}) {
    return this.request<any>('/admin-manage-users.php', {
      method: 'POST',
      body: JSON.stringify({ action, ...payload }),
    });
  }

  async adminManageSmtp(action: string, payload: any = {}) {
    return this.request<any>('/smtp-settings.php', {
      method: 'POST',
      body: JSON.stringify({ action, ...payload }),
    });
  }

  async adminManageEmailTemplates(action: string, payload: any = {}) {
    return this.request<any>('/email-templates.php', {
      method: 'POST',
      body: JSON.stringify({ action, ...payload }),
    });
  }

  // Subscription methods
  async checkSubscription() {
    return this.request<any>('/check-subscription.php');
  }

  async createCheckout(plan: string) {
    return this.request<any>('/create-checkout.php', {
      method: 'POST',
      body: JSON.stringify({ plan }),
    });
  }

  // Session management
  async getSessions() {
    return this.request<any>('/sessions.php', {
      method: 'GET',
    });
  }

  async createSession() {
    return this.request<any>('/sessions.php', {
      method: 'POST',
      body: JSON.stringify({ action: 'create' }),
    });
  }

  async refreshSession(sessionId: string) {
    return this.request<any>('/sessions.php', {
      method: 'POST',
      body: JSON.stringify({ action: 'refresh', session_id: sessionId }),
    });
  }

  async deleteSession(sessionId: string) {
    return this.request<any>('/sessions.php', {
      method: 'DELETE',
      body: JSON.stringify({ session_id: sessionId }),
    });
  }

  // Email methods
  async sendEmail(payload: any) {
    return this.request<any>('/send-email.php', {
      method: 'POST',
      body: JSON.stringify(payload),
    });
  }

  // Backup methods
  async listBackups() {
    return this.request<any[]>('/backups.php?action=list');
  }

  async createBackup() {
    return this.request<any>('/backups.php?action=create', {
      method: 'POST',
    });
  }

  async deleteBackup(filename: string) {
    return this.request<any>(`/backups.php?action=delete&filename=${filename}`, {
      method: 'POST',
    });
  }

  // License activation
  async activateLicense(licenseKey: string) {
    return this.request<any>('/activate-license.php', {
      method: 'POST',
      body: JSON.stringify({ license_key: licenseKey }),
    });
  }
}

export const apiService = new ApiService();