import React, { useState } from "react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import { Avatar, AvatarFallback, AvatarImage } from "@/components/ui/avatar";
import { 
  MessageSquare, 
  Send, 
  Users, 
  Globe, 
  TrendingUp,
  Heart,
  Share2,
  MoreHorizontal,
  Smile,
  Paperclip,
  Mic,
  Search,
  Filter,
  Settings,
  Zap
} from "lucide-react";
import Layout from "@/components/Layout";

const WorldChatPage = () => {
  const [message, setMessage] = useState("");
  const [activeTab, setActiveTab] = useState("global");

  const chatData = {
    onlineUsers: 1247,
    totalMessages: 45620,
    activeChannels: 8,
    messages: [
      {
        id: 1,
        user: "Sarah Johnson",
        avatar: "SJ",
        message: "Just discovered this amazing platform! The features are incredible 🔥",
        timestamp: "2 min ago",
        likes: 12,
        isLiked: false,
        country: "🇺🇸"
      },
      {
        id: 2,
        user: "Alex Chen",
        avatar: "AC",
        message: "Anyone tried the new automation features? They're game-changing!",
        timestamp: "5 min ago",
        likes: 8,
        isLiked: true,
        country: "🇨🇦"
      },
      {
        id: 3,
        user: "Emma Davis",
        avatar: "ED",
        message: "The referral program is fantastic. Already earned $50 this month! 💰",
        timestamp: "8 min ago",
        likes: 15,
        isLiked: false,
        country: "🇬🇧"
      },
      {
        id: 4,
        user: "Mike Thompson",
        avatar: "MT",
        message: "Looking for tips on email marketing strategies. Any experts here?",
        timestamp: "12 min ago",
        likes: 6,
        isLiked: false,
        country: "🇦🇺"
      }
    ],
    channels: [
      { name: "General", users: 1247, active: true },
      { name: "Marketing", users: 456, active: false },
      { name: "Automation", users: 234, active: false },
      { name: "Tips & Tricks", users: 789, active: false }
    ]
  };

  const handleSendMessage = () => {
    if (message.trim()) {
      // Add message logic here
      setMessage("");
    }
  };

  const handleLike = (messageId: number) => {
    // Like logic here
  };

  return (
    <Layout>
      <div className="space-y-6">
        {/* Header with Logo */}
        <div className="flex items-center gap-3">
          <div className="w-8 h-8 bg-gradient-to-r from-flashcore-purple to-flashcore-green rounded-lg flex items-center justify-center">
            <Globe className="w-5 h-5 text-white" />
          </div>
          <div>
            <h1 className="text-3xl font-bold">World Chat</h1>
            <p className="text-muted-foreground">
              Connect with the global FlashCore community
            </p>
          </div>
        </div>

        {/* Stats Overview */}
        <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
          <Card className="hover:shadow-lg transition-shadow">
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-muted-foreground">Online Users</p>
                  <p className="text-2xl font-bold">{chatData.onlineUsers.toLocaleString()}</p>
                  <div className="flex items-center mt-2">
                    <TrendingUp className="w-4 h-4 text-green-500 mr-1" />
                    <span className="text-sm text-green-500">+23 today</span>
                  </div>
                </div>
                <Users className="w-8 h-8 text-flashcore-purple" />
              </div>
            </CardContent>
          </Card>

          <Card className="hover:shadow-lg transition-shadow">
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-muted-foreground">Total Messages</p>
                  <p className="text-2xl font-bold">{chatData.totalMessages.toLocaleString()}</p>
                  <p className="text-sm text-muted-foreground mt-2">This month</p>
                </div>
                <MessageSquare className="w-8 h-8 text-flashcore-green" />
              </div>
            </CardContent>
          </Card>

          <Card className="hover:shadow-lg transition-shadow">
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-muted-foreground">Active Channels</p>
                  <p className="text-2xl font-bold">{chatData.activeChannels}</p>
                  <p className="text-sm text-muted-foreground mt-2">Available now</p>
                </div>
                <Globe className="w-8 h-8 text-flashcore-orange" />
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Chat Interface */}
        <div className="grid grid-cols-1 lg:grid-cols-4 gap-6">
          {/* Channels Sidebar */}
          <Card className="lg:col-span-1">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <MessageSquare className="w-5 h-5" />
                Channels
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="space-y-2">
                {chatData.channels.map((channel) => (
                  <button
                    key={channel.name}
                    className={`w-full p-3 rounded-lg text-left transition-colors ${
                      channel.active 
                        ? 'bg-flashcore-purple text-white' 
                        : 'hover:bg-muted'
                    }`}
                  >
                    <div className="flex items-center justify-between">
                      <span className="font-medium">#{channel.name}</span>
                      <Badge variant="secondary" className="text-xs">
                        {channel.users}
                      </Badge>
                    </div>
                  </button>
                ))}
              </div>
            </CardContent>
          </Card>

          {/* Chat Messages */}
          <Card className="lg:col-span-3">
            <CardHeader>
              <div className="flex items-center justify-between">
                <div>
                  <CardTitle className="flex items-center gap-2">
                    <Globe className="w-5 h-5" />
                    Global Chat
                  </CardTitle>
                  <CardDescription>
                    {chatData.onlineUsers.toLocaleString()} users online
                  </CardDescription>
                </div>
                <div className="flex items-center gap-2">
                  <Button variant="outline" size="sm">
                    <Search className="w-4 h-4 mr-2" />
                    Search
                  </Button>
                  <Button variant="outline" size="sm">
                    <Filter className="w-4 h-4 mr-2" />
                    Filter
                  </Button>
                </div>
              </div>
            </CardHeader>
            <CardContent>
              <div className="space-y-4 h-96 overflow-y-auto">
                {chatData.messages.map((msg) => (
                  <div key={msg.id} className="flex gap-3 p-3 hover:bg-muted/50 rounded-lg">
                    <Avatar className="w-10 h-10">
                      <AvatarFallback className="bg-gradient-to-r from-flashcore-purple to-flashcore-green text-white">
                        {msg.avatar}
                      </AvatarFallback>
                    </Avatar>
                    <div className="flex-1">
                      <div className="flex items-center gap-2 mb-1">
                        <span className="font-medium">{msg.user}</span>
                        <span className="text-sm">{msg.country}</span>
                        <span className="text-xs text-muted-foreground">{msg.timestamp}</span>
                      </div>
                      <p className="text-sm mb-2">{msg.message}</p>
                      <div className="flex items-center gap-4">
                        <button 
                          onClick={() => handleLike(msg.id)}
                          className={`flex items-center gap-1 text-xs transition-colors ${
                            msg.isLiked ? 'text-red-500' : 'text-muted-foreground hover:text-red-500'
                          }`}
                        >
                          <Heart className={`w-3 h-3 ${msg.isLiked ? 'fill-current' : ''}`} />
                          {msg.likes}
                        </button>
                        <button className="flex items-center gap-1 text-xs text-muted-foreground hover:text-foreground">
                          <Share2 className="w-3 h-3" />
                          Share
                        </button>
                        <button className="flex items-center gap-1 text-xs text-muted-foreground hover:text-foreground">
                          <MoreHorizontal className="w-3 h-3" />
                        </button>
                      </div>
                    </div>
                  </div>
                ))}
              </div>

              {/* Message Input */}
              <div className="mt-4 p-4 border-t">
                <div className="flex items-center gap-2">
                  <Button variant="outline" size="sm">
                    <Paperclip className="w-4 h-4" />
                  </Button>
                  <Input
                    value={message}
                    onChange={(e) => setMessage(e.target.value)}
                    placeholder="Type your message..."
                    className="flex-1"
                    onKeyPress={(e) => e.key === 'Enter' && handleSendMessage()}
                  />
                  <Button variant="outline" size="sm">
                    <Smile className="w-4 h-4" />
                  </Button>
                  <Button variant="outline" size="sm">
                    <Mic className="w-4 h-4" />
                  </Button>
                  <Button 
                    onClick={handleSendMessage}
                    className="bg-gradient-to-r from-flashcore-purple to-flashcore-green"
                  >
                    <Send className="w-4 h-4" />
                  </Button>
                </div>
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Community Guidelines */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Globe className="w-5 h-5" />
              Community Guidelines
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
              <div className="text-center p-4 bg-green-50 rounded-lg">
                <div className="w-8 h-8 bg-green-500 rounded-full flex items-center justify-center mx-auto mb-2">
                  <span className="text-white text-sm">✓</span>
                </div>
                <h3 className="font-semibold mb-1">Be Respectful</h3>
                <p className="text-sm text-muted-foreground">
                  Treat everyone with kindness and respect
                </p>
              </div>
              <div className="text-center p-4 bg-blue-50 rounded-lg">
                <div className="w-8 h-8 bg-blue-500 rounded-full flex items-center justify-center mx-auto mb-2">
                  <span className="text-white text-sm">💡</span>
                </div>
                <h3 className="font-semibold mb-1">Share Knowledge</h3>
                <p className="text-sm text-muted-foreground">
                  Help others learn and grow together
                </p>
              </div>
              <div className="text-center p-4 bg-orange-50 rounded-lg">
                <div className="w-8 h-8 bg-orange-500 rounded-full flex items-center justify-center mx-auto mb-2">
                  <span className="text-white text-sm">🚫</span>
                </div>
                <h3 className="font-semibold mb-1">No Spam</h3>
                <p className="text-sm text-muted-foreground">
                  Keep conversations relevant and meaningful
                </p>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>
    </Layout>
  );
};

export default WorldChatPage;
