import React, { useState } from 'react';
import { Link } from 'react-router-dom';
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import { 
  Calendar, 
  Clock, 
  User, 
  Users, 
  Play, 
  ExternalLink, 
  Search,
  Filter,
  Video,
  BookOpen,
  Award,
  Zap,
  Target,
  TrendingUp,
  Building,
  ShoppingCart,
  Mail,
  BarChart3,
  CheckCircle,
  AlertCircle,
  Star
} from "lucide-react";
import Navigation from "@/components/Navigation";
import Footer from "@/components/Footer";

const WebinarsPage = () => {
  const [searchQuery, setSearchQuery] = useState('');
  const [selectedFilter, setSelectedFilter] = useState('all');

  const filters = [
    { id: 'all', name: 'All Webinars', icon: Video },
    { id: 'upcoming', name: 'Upcoming', icon: Calendar },
    { id: 'past', name: 'Past Webinars', icon: Play },
    { id: 'email-marketing', name: 'Email Marketing', icon: Mail },
    { id: 'automation', name: 'Automation', icon: Zap },
    { id: 'analytics', name: 'Analytics', icon: BarChart3 },
    { id: 'ecommerce', name: 'E-commerce', icon: ShoppingCart },
    { id: 'enterprise', name: 'Enterprise', icon: Building }
  ];

  const upcomingWebinars = [
    {
      id: 1,
      title: "Mastering Email Automation: From Welcome Series to Win-Back Campaigns",
      description: "Learn how to create sophisticated email automation workflows that nurture leads and drive conversions. We'll cover welcome series, abandoned cart recovery, win-back campaigns, and more.",
      speaker: "Sarah Johnson",
      speakerRole: "Email Marketing Expert",
      speakerCompany: "FlashCore",
      date: "March 25, 2024",
      time: "2:00 PM EST",
      duration: "60 minutes",
      category: "automation",
      attendees: 1247,
      maxAttendees: 2000,
      isLive: true,
      registrationRequired: true,
      tags: ["Automation", "Welcome Series", "Win-Back", "Advanced"]
    },
    {
      id: 2,
      title: "E-commerce Email Marketing: Holiday Season Success Strategies",
      description: "Prepare for the upcoming holiday season with proven email marketing strategies that drive sales and customer engagement. Learn from real case studies and expert insights.",
      speaker: "Michael Chen",
      speakerRole: "E-commerce Specialist",
      speakerCompany: "FlashCore",
      date: "March 28, 2024",
      time: "1:00 PM EST",
      duration: "45 minutes",
      category: "ecommerce",
      attendees: 892,
      maxAttendees: 1500,
      isLive: true,
      registrationRequired: true,
      tags: ["E-commerce", "Holiday", "Sales", "Strategy"]
    },
    {
      id: 3,
      title: "Email Analytics Deep Dive: Understanding Your Data for Better ROI",
      description: "Go beyond basic metrics and learn how to analyze email performance data to optimize campaigns and maximize your return on investment.",
      speaker: "Emily Rodriguez",
      speakerRole: "Data Analyst",
      speakerCompany: "FlashCore",
      date: "April 2, 2024",
      time: "3:00 PM EST",
      duration: "75 minutes",
      category: "analytics",
      attendees: 567,
      maxAttendees: 1000,
      isLive: true,
      registrationRequired: true,
      tags: ["Analytics", "ROI", "Data", "Optimization"]
    },
    {
      id: 4,
      title: "Enterprise Email Marketing: Scaling Success for Large Organizations",
      description: "Discover how enterprise companies are leveraging email marketing to scale their operations, improve customer relationships, and drive growth.",
      speaker: "David Thompson",
      speakerRole: "Enterprise Consultant",
      speakerCompany: "FlashCore",
      date: "April 5, 2024",
      time: "2:30 PM EST",
      duration: "90 minutes",
      category: "enterprise",
      attendees: 234,
      maxAttendees: 500,
      isLive: true,
      registrationRequired: true,
      tags: ["Enterprise", "Scaling", "Growth", "Strategy"]
    }
  ];

  const pastWebinars = [
    {
      id: 5,
      title: "Email Marketing Fundamentals: Building Your First Campaign",
      description: "Perfect for beginners! Learn the basics of email marketing, from list building to creating your first successful campaign.",
      speaker: "Lisa Chen",
      speakerRole: "Marketing Educator",
      speakerCompany: "FlashCore",
      date: "March 15, 2024",
      time: "1:00 PM EST",
      duration: "60 minutes",
      category: "email-marketing",
      attendees: 2156,
      maxAttendees: 2500,
      isLive: false,
      registrationRequired: false,
      recordingUrl: "/webinars/email-fundamentals-march-15",
      rating: 4.8,
      reviewCount: 156,
      tags: ["Fundamentals", "Beginners", "Campaigns"]
    },
    {
      id: 6,
      title: "Subject Line Mastery: Writing Headlines That Get Opened",
      description: "Master the art of writing compelling subject lines that increase open rates and drive engagement. Includes A/B testing strategies.",
      speaker: "Amanda Kim",
      speakerRole: "Copywriting Expert",
      speakerCompany: "FlashCore",
      date: "March 8, 2024",
      time: "2:00 PM EST",
      duration: "45 minutes",
      category: "email-marketing",
      attendees: 1892,
      maxAttendees: 2000,
      isLive: false,
      registrationRequired: false,
      recordingUrl: "/webinars/subject-lines-march-8",
      rating: 4.9,
      reviewCount: 203,
      tags: ["Subject Lines", "Open Rates", "Copywriting"]
    },
    {
      id: 7,
      title: "GDPR Compliance in Email Marketing: What You Need to Know",
      description: "Stay compliant with GDPR regulations. Learn best practices for data collection, consent management, and maintaining customer trust.",
      speaker: "Mark Johnson",
      speakerRole: "Legal Expert",
      speakerCompany: "FlashCore",
      date: "March 1, 2024",
      time: "3:00 PM EST",
      duration: "75 minutes",
      category: "email-marketing",
      attendees: 1347,
      maxAttendees: 1500,
      isLive: false,
      registrationRequired: false,
      recordingUrl: "/webinars/gdpr-compliance-march-1",
      rating: 4.7,
      reviewCount: 89,
      tags: ["GDPR", "Compliance", "Legal"]
    },
    {
      id: 8,
      title: "Advanced Segmentation: Targeting the Right Audience",
      description: "Learn advanced segmentation techniques to deliver personalized content that resonates with different audience segments.",
      speaker: "Robert Martinez",
      speakerRole: "Segmentation Specialist",
      speakerCompany: "FlashCore",
      date: "February 23, 2024",
      time: "2:30 PM EST",
      duration: "60 minutes",
      category: "automation",
      attendees: 987,
      maxAttendees: 1200,
      isLive: false,
      registrationRequired: false,
      recordingUrl: "/webinars/advanced-segmentation-feb-23",
      rating: 4.6,
      reviewCount: 134,
      tags: ["Segmentation", "Personalization", "Advanced"]
    }
  ];

  const allWebinars = [...upcomingWebinars, ...pastWebinars];

  const filteredWebinars = allWebinars.filter(webinar => {
    const matchesSearch = webinar.title.toLowerCase().includes(searchQuery.toLowerCase()) ||
                         webinar.description.toLowerCase().includes(searchQuery.toLowerCase()) ||
                         webinar.tags.some(tag => tag.toLowerCase().includes(searchQuery.toLowerCase()));
    const matchesFilter = selectedFilter === 'all' || 
                         (selectedFilter === 'upcoming' && webinar.isLive) ||
                         (selectedFilter === 'past' && !webinar.isLive) ||
                         webinar.category === selectedFilter;
    return matchesSearch && matchesFilter;
  });

  const upcomingFiltered = filteredWebinars.filter(w => w.isLive);
  const pastFiltered = filteredWebinars.filter(w => !w.isLive);

  return (
    <div className="min-h-screen bg-background">
      <Navigation />

      {/* Hero Section */}
      <section className="container mx-auto px-4 py-12 md:py-20">
        <div className="max-w-4xl mx-auto text-center">
          <Badge variant="secondary" className="mb-4">
            <Video className="w-3 h-3 mr-1" />
            Live & On-Demand Webinars
          </Badge>
          <h1 className="text-3xl md:text-4xl lg:text-5xl font-bold mb-6 bg-gradient-to-r from-flashcore-purple to-flashcore-green bg-clip-text text-transparent">
            Learn Email Marketing from Industry Experts
          </h1>
          <p className="text-lg md:text-xl text-muted-foreground mb-8 max-w-2xl mx-auto">
            Join our live webinars and access on-demand recordings to master email marketing 
            strategies, automation, analytics, and more.
          </p>
          
          {/* Search Bar */}
          <div className="max-w-2xl mx-auto mb-8">
            <div className="relative">
              <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-muted-foreground w-5 h-5" />
              <Input
                type="text"
                placeholder="Search webinars by topic, speaker, or keyword..."
                value={searchQuery}
                onChange={(e) => setSearchQuery(e.target.value)}
                className="pl-10 pr-4 py-3 text-lg"
              />
            </div>
          </div>
        </div>
      </section>

      {/* Filters */}
      <section className="container mx-auto px-4 py-8">
        <div className="max-w-6xl mx-auto">
          <div className="flex flex-wrap gap-2 justify-center">
            {filters.map((filter) => (
              <Button
                key={filter.id}
                variant={selectedFilter === filter.id ? "default" : "outline"}
                size="sm"
                onClick={() => setSelectedFilter(filter.id)}
                className="flex items-center space-x-2"
              >
                <filter.icon className="w-4 h-4" />
                <span>{filter.name}</span>
              </Button>
            ))}
          </div>
        </div>
      </section>

      {/* Upcoming Webinars Section */}
      {upcomingFiltered.length > 0 && (
        <section className="container mx-auto px-4 py-12 md:py-20">
          <div className="max-w-6xl mx-auto">
            <div className="text-center mb-12">
              <h2 className="text-2xl md:text-3xl font-bold mb-4">Upcoming Live Webinars</h2>
              <p className="text-muted-foreground">
                Register for our upcoming live sessions and learn from industry experts.
              </p>
            </div>
            
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-6 md:gap-8">
              {upcomingFiltered.map((webinar) => (
                <Card key={webinar.id} className="hover:shadow-lg transition-shadow group border-2 border-flashcore-purple/20">
                  <CardHeader>
                    <div className="flex items-center justify-between mb-4">
                      <Badge variant="destructive" className="flex items-center">
                        <Calendar className="w-3 h-3 mr-1" />
                        Live
                      </Badge>
                      <Badge variant="secondary" className="text-xs">
                        {filters.find(f => f.id === webinar.category)?.name}
                      </Badge>
                    </div>
                    <CardTitle className="text-xl group-hover:text-flashcore-purple transition-colors">
                      {webinar.title}
                    </CardTitle>
                    <CardDescription className="text-sm">
                      {webinar.description}
                    </CardDescription>
                  </CardHeader>
                  <CardContent>
                    <div className="space-y-4">
                      {/* Speaker Info */}
                      <div className="flex items-center space-x-3">
                        <div className="w-10 h-10 bg-gradient-to-r from-flashcore-purple to-flashcore-green rounded-full flex items-center justify-center">
                          <User className="w-5 h-5 text-white" />
                        </div>
                        <div>
                          <p className="font-semibold">{webinar.speaker}</p>
                          <p className="text-sm text-muted-foreground">{webinar.speakerRole} at {webinar.speakerCompany}</p>
                        </div>
                      </div>

                      {/* Date & Time */}
                      <div className="flex items-center justify-between text-sm">
                        <div className="flex items-center space-x-4">
                          <span className="flex items-center">
                            <Calendar className="w-4 h-4 mr-1" />
                            {webinar.date}
                          </span>
                          <span className="flex items-center">
                            <Clock className="w-4 h-4 mr-1" />
                            {webinar.time} ({webinar.duration})
                          </span>
                        </div>
                        <span className="flex items-center text-muted-foreground">
                          <Users className="w-4 h-4 mr-1" />
                          {webinar.attendees}/{webinar.maxAttendees}
                        </span>
                      </div>

                      {/* Tags */}
                      <div className="flex flex-wrap gap-1">
                        {webinar.tags.map((tag, index) => (
                          <Badge key={index} variant="outline" className="text-xs">
                            {tag}
                          </Badge>
                        ))}
                      </div>

                      {/* Registration Button */}
                      <Button className="w-full bg-flashcore-purple hover:bg-flashcore-purple/90">
                        Register Now
                        <ExternalLink className="w-4 h-4 ml-2" />
                      </Button>
                    </div>
                  </CardContent>
                </Card>
              ))}
            </div>
          </div>
        </section>
      )}

      {/* Past Webinars Section */}
      {pastFiltered.length > 0 && (
        <section className="container mx-auto px-4 py-12 md:py-20 bg-muted/20">
          <div className="max-w-6xl mx-auto">
            <div className="text-center mb-12">
              <h2 className="text-2xl md:text-3xl font-bold mb-4">On-Demand Webinars</h2>
              <p className="text-muted-foreground">
                Watch recorded webinars at your own pace and learn from past sessions.
              </p>
            </div>
            
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-6 md:gap-8">
              {pastFiltered.map((webinar) => (
                <Card key={webinar.id} className="hover:shadow-lg transition-shadow group">
                  <CardHeader>
                    <div className="flex items-center justify-between mb-4">
                      <Badge variant="secondary" className="flex items-center">
                        <Play className="w-3 h-3 mr-1" />
                        On-Demand
                      </Badge>
                      <Badge variant="outline" className="text-xs">
                        {filters.find(f => f.id === webinar.category)?.name}
                      </Badge>
                    </div>
                    <CardTitle className="text-xl group-hover:text-flashcore-purple transition-colors">
                      {webinar.title}
                    </CardTitle>
                    <CardDescription className="text-sm">
                      {webinar.description}
                    </CardDescription>
                  </CardHeader>
                  <CardContent>
                    <div className="space-y-4">
                      {/* Speaker Info */}
                      <div className="flex items-center space-x-3">
                        <div className="w-10 h-10 bg-gradient-to-r from-gray-400 to-gray-500 rounded-full flex items-center justify-center">
                          <User className="w-5 h-5 text-white" />
                        </div>
                        <div>
                          <p className="font-semibold">{webinar.speaker}</p>
                          <p className="text-sm text-muted-foreground">{webinar.speakerRole} at {webinar.speakerCompany}</p>
                        </div>
                      </div>

                      {/* Date & Rating */}
                      <div className="flex items-center justify-between text-sm">
                        <div className="flex items-center space-x-4">
                          <span className="flex items-center">
                            <Calendar className="w-4 h-4 mr-1" />
                            {webinar.date}
                          </span>
                          <span className="flex items-center">
                            <Clock className="w-4 h-4 mr-1" />
                            {webinar.duration}
                          </span>
                        </div>
                        <div className="flex items-center">
                          <Star className="w-4 h-4 text-yellow-500 mr-1" />
                          <span className="font-semibold">{webinar.rating}</span>
                          <span className="text-muted-foreground ml-1">({webinar.reviewCount})</span>
                        </div>
                      </div>

                      {/* Tags */}
                      <div className="flex flex-wrap gap-1">
                        {webinar.tags.map((tag, index) => (
                          <Badge key={index} variant="outline" className="text-xs">
                            {tag}
                          </Badge>
                        ))}
                      </div>

                      {/* Watch Button */}
                      <Button variant="outline" className="w-full group-hover:bg-flashcore-purple group-hover:text-white transition-colors">
                        <Play className="w-4 h-4 mr-2" />
                        Watch Recording
                      </Button>
                    </div>
                  </CardContent>
                </Card>
              ))}
            </div>
          </div>
        </section>
      )}

      {/* Newsletter Section */}
      <section className="container mx-auto px-4 py-12 md:py-20">
        <div className="max-w-4xl mx-auto">
          <Card className="bg-gradient-to-r from-flashcore-purple to-flashcore-green text-white">
            <CardContent className="p-8 md:p-12 text-center">
              <h2 className="text-2xl md:text-3xl font-bold mb-4">Never Miss a Webinar</h2>
              <p className="text-lg mb-6 opacity-90">
                Get notified about upcoming webinars and receive exclusive content from our email marketing experts.
              </p>
              <div className="flex flex-col sm:flex-row gap-4 max-w-md mx-auto">
                <Input
                  type="email"
                  placeholder="Enter your email address"
                  className="bg-white text-gray-900 placeholder-gray-500"
                />
                <Button variant="secondary" className="bg-white text-flashcore-purple hover:bg-gray-100">
                  Subscribe
                  <ExternalLink className="w-4 h-4 ml-2" />
                </Button>
              </div>
              <p className="text-sm opacity-75 mt-4">
                Get early access to webinar registrations and exclusive content.
              </p>
            </CardContent>
          </Card>
        </div>
      </section>

      {/* Additional Resources */}
      <section className="container mx-auto px-4 py-12 md:py-20 bg-muted/20">
        <div className="max-w-6xl mx-auto">
          <div className="text-center mb-12">
            <h2 className="text-2xl md:text-3xl font-bold mb-4">More Learning Resources</h2>
            <p className="text-muted-foreground">
              Explore additional resources to enhance your email marketing knowledge.
            </p>
          </div>
          
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 md:gap-8">
            <Card className="hover:shadow-lg transition-shadow">
              <CardContent className="p-6 text-center">
                <div className="w-12 h-12 bg-gradient-to-r from-flashcore-purple to-flashcore-green rounded-lg flex items-center justify-center mx-auto mb-4">
                  <BookOpen className="w-6 h-6 text-white" />
                </div>
                <h3 className="text-lg font-semibold mb-2">Blog</h3>
                <p className="text-sm text-muted-foreground mb-4">
                  Latest insights and tips from our experts.
                </p>
                <Link to="/blog">
                  <Button variant="outline" size="sm" className="w-full">
                    Read Articles
                  </Button>
                </Link>
              </CardContent>
            </Card>
            
            <Card className="hover:shadow-lg transition-shadow">
              <CardContent className="p-6 text-center">
                <div className="w-12 h-12 bg-gradient-to-r from-flashcore-purple to-flashcore-green rounded-lg flex items-center justify-center mx-auto mb-4">
                  <Award className="w-6 h-6 text-white" />
                </div>
                <h3 className="text-lg font-semibold mb-2">Certification</h3>
                <p className="text-sm text-muted-foreground mb-4">
                  Get certified in email marketing.
                </p>
                <Link to="/certification">
                  <Button variant="outline" size="sm" className="w-full">
                    Get Certified
                  </Button>
                </Link>
              </CardContent>
            </Card>
            
            <Card className="hover:shadow-lg transition-shadow">
              <CardContent className="p-6 text-center">
                <div className="w-12 h-12 bg-gradient-to-r from-flashcore-purple to-flashcore-green rounded-lg flex items-center justify-center mx-auto mb-4">
                  <Users className="w-6 h-6 text-white" />
                </div>
                <h3 className="text-lg font-semibold mb-2">Community</h3>
                <p className="text-sm text-muted-foreground mb-4">
                  Connect with other marketers.
                </p>
                <Link to="/community">
                  <Button variant="outline" size="sm" className="w-full">
                    Join Community
                  </Button>
                </Link>
              </CardContent>
            </Card>
            
            <Card className="hover:shadow-lg transition-shadow">
              <CardContent className="p-6 text-center">
                <div className="w-12 h-12 bg-gradient-to-r from-flashcore-purple to-flashcore-green rounded-lg flex items-center justify-center mx-auto mb-4">
                  <Target className="w-6 h-6 text-white" />
                </div>
                <h3 className="text-lg font-semibold mb-2">Help Center</h3>
                <p className="text-sm text-muted-foreground mb-4">
                  Find answers to your questions.
                </p>
                <Link to="/help">
                  <Button variant="outline" size="sm" className="w-full">
                    Get Help
                  </Button>
                </Link>
              </CardContent>
            </Card>
          </div>
        </div>
      </section>

      <Footer />
    </div>
  );
};

export default WebinarsPage; 