import React, { useState } from 'react';
import { Link } from 'react-router-dom';
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { ArrowRight, Search, Filter, Mail, CreditCard, Receipt, FileText, Users, Calendar, Star, Download, Eye, Copy } from "lucide-react";
import Navigation from "@/components/Navigation";
import Footer from "@/components/Footer";

const TemplatesPage = () => {
  const [searchTerm, setSearchTerm] = useState('');
  const [selectedCategory, setSelectedCategory] = useState('all');

  const categories = [
    { value: 'all', label: 'All Templates' },
    { value: 'welcome', label: 'Welcome Emails' },
    { value: 'billing', label: 'Billing & Invoices' },
    { value: 'receipts', label: 'Receipts' },
    { value: 'marketing', label: 'Marketing' },
    { value: 'support', label: 'Support' },
    { value: 'transactional', label: 'Transactional' }
  ];

  const templates = [
    {
      id: 1,
      name: "Welcome Email",
      category: "welcome",
      description: "Professional welcome email for new customers",
      preview: "Welcome to FlashCore! We're excited to have you on board...",
      rating: 4.8,
      downloads: 1247,
      tags: ["welcome", "onboarding", "professional"],
      color: "from-flashcore-purple to-flashcore-green"
    },
    {
      id: 2,
      name: "Invoice Template",
      category: "billing",
      description: "Clean and professional invoice email template",
      preview: "Your invoice for services rendered is now ready...",
      rating: 4.9,
      downloads: 2156,
      tags: ["billing", "invoice", "professional"],
      color: "from-flashcore-green to-flashcore-purple"
    },
    {
      id: 3,
      name: "Payment Receipt",
      category: "receipts",
      description: "Detailed payment receipt with transaction details",
      preview: "Thank you for your payment. Here's your receipt...",
      rating: 4.7,
      downloads: 1893,
      tags: ["receipt", "payment", "confirmation"],
      color: "from-flashcore-orange to-flashcore-purple"
    },
    {
      id: 4,
      name: "Newsletter Template",
      category: "marketing",
      description: "Engaging newsletter template for marketing campaigns",
      preview: "Stay updated with our latest news and insights...",
      rating: 4.6,
      downloads: 987,
      tags: ["newsletter", "marketing", "engagement"],
      color: "from-blue-500 to-purple-500"
    },
    {
      id: 5,
      name: "Support Ticket Confirmation",
      category: "support",
      description: "Professional support ticket confirmation email",
      preview: "We've received your support request and will respond shortly...",
      rating: 4.8,
      downloads: 756,
      tags: ["support", "confirmation", "professional"],
      color: "from-green-500 to-blue-500"
    },
    {
      id: 6,
      name: "Order Confirmation",
      category: "transactional",
      description: "Complete order confirmation with tracking details",
      preview: "Your order has been confirmed and is being processed...",
      rating: 4.9,
      downloads: 1432,
      tags: ["order", "confirmation", "tracking"],
      color: "from-purple-500 to-pink-500"
    },
    {
      id: 7,
      name: "Password Reset",
      category: "support",
      description: "Secure password reset email template",
      preview: "We received a request to reset your password...",
      rating: 4.7,
      downloads: 654,
      tags: ["security", "password", "reset"],
      color: "from-red-500 to-orange-500"
    },
    {
      id: 8,
      name: "Account Verification",
      category: "welcome",
      description: "Email verification template for new accounts",
      preview: "Please verify your email address to complete registration...",
      rating: 4.8,
      downloads: 892,
      tags: ["verification", "security", "onboarding"],
      color: "from-indigo-500 to-purple-500"
    }
  ];

  const filteredTemplates = templates.filter(template => {
    const matchesSearch = template.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
                         template.description.toLowerCase().includes(searchTerm.toLowerCase()) ||
                         template.tags.some(tag => tag.toLowerCase().includes(searchTerm.toLowerCase()));
    const matchesCategory = selectedCategory === 'all' || template.category === selectedCategory;
    return matchesSearch && matchesCategory;
  });

  return (
    <div className="min-h-screen bg-background">
      <Navigation />

      {/* Header */}
      <section className="container mx-auto px-4 py-12 md:py-20 text-center">
        <div className="max-w-4xl mx-auto">
          <Badge variant="secondary" className="mb-4">
            <Mail className="w-3 h-3 mr-1" />
            Professional Templates
          </Badge>
          <h1 className="text-3xl md:text-4xl lg:text-5xl font-bold mb-6 bg-gradient-to-r from-flashcore-purple to-flashcore-green bg-clip-text text-transparent">
            Email Templates
          </h1>
          <p className="text-lg md:text-xl text-muted-foreground mb-8 max-w-2xl mx-auto">
            Choose from our library of professional email templates designed to help you communicate effectively with your customers.
          </p>
        </div>
      </section>

      {/* Search and Filter */}
      <section className="container mx-auto px-4 py-8">
        <div className="max-w-4xl mx-auto">
          <div className="flex flex-col md:flex-row gap-4">
            <div className="flex-1 relative">
              <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 w-4 h-4 text-muted-foreground" />
              <Input
                placeholder="Search templates..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className="pl-10"
              />
            </div>
            <Select value={selectedCategory} onValueChange={setSelectedCategory}>
              <SelectTrigger className="w-full md:w-48">
                <Filter className="w-4 h-4 mr-2" />
                <SelectValue placeholder="Filter by category" />
              </SelectTrigger>
              <SelectContent>
                {categories.map((category) => (
                  <SelectItem key={category.value} value={category.value}>
                    {category.label}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>
        </div>
      </section>

      {/* Templates Grid */}
      <section className="container mx-auto px-4 py-12 md:py-20">
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6 md:gap-8">
          {filteredTemplates.map((template) => (
            <Card key={template.id} className="hover:shadow-lg transition-shadow group">
              <CardHeader>
                <div className="flex items-center justify-between mb-4">
                  <div className={`w-10 h-10 bg-gradient-to-r ${template.color} rounded-lg flex items-center justify-center`}>
                    <Mail className="w-5 h-5 text-white" />
                  </div>
                  <div className="flex items-center space-x-2">
                    <div className="flex items-center">
                      <Star className="w-4 h-4 text-yellow-400 fill-current" />
                      <span className="text-sm font-medium ml-1">{template.rating}</span>
                    </div>
                    <Badge variant="secondary" className="text-xs">
                      {template.downloads} downloads
                    </Badge>
                  </div>
                </div>
                <CardTitle className="text-lg md:text-xl">{template.name}</CardTitle>
                <CardDescription className="text-sm md:text-base">
                  {template.description}
                </CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="bg-muted/50 rounded-lg p-3">
                  <p className="text-sm text-muted-foreground italic">
                    "{template.preview}..."
                  </p>
                </div>
                
                <div className="flex flex-wrap gap-2">
                  {template.tags.map((tag, index) => (
                    <Badge key={index} variant="outline" className="text-xs">
                      {tag}
                    </Badge>
                  ))}
                </div>
                
                <div className="flex space-x-2">
                  <Button variant="outline" size="sm" className="flex-1">
                    <Eye className="w-4 h-4 mr-2" />
                    Preview
                  </Button>
                  <Button size="sm" className="flex-1 bg-gradient-to-r from-flashcore-purple to-flashcore-green">
                    <Download className="w-4 h-4 mr-2" />
                    Use Template
                  </Button>
                </div>
              </CardContent>
            </Card>
          ))}
        </div>
        
        {filteredTemplates.length === 0 && (
          <div className="text-center py-12">
            <Mail className="w-16 h-16 text-muted-foreground mx-auto mb-4" />
            <h3 className="text-xl font-semibold mb-2">No templates found</h3>
            <p className="text-muted-foreground">
              Try adjusting your search terms or filter criteria.
            </p>
          </div>
        )}
      </section>

      {/* Template Categories */}
      <section className="container mx-auto px-4 py-12 md:py-20 bg-muted/20">
        <div className="text-center mb-12 md:mb-16">
          <h2 className="text-2xl md:text-3xl font-bold mb-4">Template Categories</h2>
          <p className="text-muted-foreground max-w-2xl mx-auto">
            Find the perfect template for every type of communication.
          </p>
        </div>
        
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 md:gap-8 max-w-6xl mx-auto">
          <Card className="text-center hover:shadow-lg transition-shadow">
            <CardHeader>
              <div className="w-12 h-12 bg-gradient-to-r from-flashcore-purple to-flashcore-green rounded-lg flex items-center justify-center mx-auto mb-4">
                <Users className="w-6 h-6 text-white" />
              </div>
              <CardTitle className="text-lg">Welcome Emails</CardTitle>
            </CardHeader>
            <CardContent>
              <p className="text-sm text-muted-foreground mb-4">
                Onboard new customers with professional welcome emails.
              </p>
              <Badge variant="secondary">24 templates</Badge>
            </CardContent>
          </Card>
          
          <Card className="text-center hover:shadow-lg transition-shadow">
            <CardHeader>
              <div className="w-12 h-12 bg-gradient-to-r from-flashcore-green to-flashcore-purple rounded-lg flex items-center justify-center mx-auto mb-4">
                <CreditCard className="w-6 h-6 text-white" />
              </div>
              <CardTitle className="text-lg">Billing & Invoices</CardTitle>
            </CardHeader>
            <CardContent>
              <p className="text-sm text-muted-foreground mb-4">
                Professional billing and invoice email templates.
              </p>
              <Badge variant="secondary">18 templates</Badge>
            </CardContent>
          </Card>
          
          <Card className="text-center hover:shadow-lg transition-shadow">
            <CardHeader>
              <div className="w-12 h-12 bg-gradient-to-r from-flashcore-orange to-flashcore-purple rounded-lg flex items-center justify-center mx-auto mb-4">
                <Receipt className="w-6 h-6 text-white" />
              </div>
              <CardTitle className="text-lg">Receipts</CardTitle>
            </CardHeader>
            <CardContent>
              <p className="text-sm text-muted-foreground mb-4">
                Clear and detailed payment receipt templates.
              </p>
              <Badge variant="secondary">15 templates</Badge>
            </CardContent>
          </Card>
          
          <Card className="text-center hover:shadow-lg transition-shadow">
            <CardHeader>
              <div className="w-12 h-12 bg-gradient-to-r from-blue-500 to-purple-500 rounded-lg flex items-center justify-center mx-auto mb-4">
                <Calendar className="w-6 h-6 text-white" />
              </div>
              <CardTitle className="text-lg">Marketing</CardTitle>
            </CardHeader>
            <CardContent>
              <p className="text-sm text-muted-foreground mb-4">
                Engaging marketing and newsletter templates.
              </p>
              <Badge variant="secondary">32 templates</Badge>
            </CardContent>
          </Card>
        </div>
      </section>

      {/* CTA Section */}
      <section className="container mx-auto px-4 py-12 md:py-20">
        <div className="bg-gradient-to-r from-flashcore-purple to-flashcore-green rounded-2xl p-6 md:p-8 lg:p-12 text-center text-white">
          <h2 className="text-2xl md:text-3xl font-bold mb-4">Ready to Get Started?</h2>
          <p className="text-lg md:text-xl mb-6 md:mb-8 opacity-90">
            Start using professional email templates to improve your communications.
          </p>
          <div className="flex flex-col sm:flex-row gap-4 justify-center">
            <Link to="/register">
              <Button size="lg" variant="secondary" className="bg-white text-flashcore-purple hover:bg-gray-100">
                Start Free Trial
                <ArrowRight className="w-4 h-4 ml-2" />
              </Button>
            </Link>
            <Link to="/contact">
              <Button size="lg" variant="outline" className="border-white text-white hover:bg-white hover:text-flashcore-purple">
                Contact Sales
              </Button>
            </Link>
          </div>
        </div>
      </section>

      <Footer />
    </div>
  );
};

export default TemplatesPage; 