import React, { useState } from "react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import { Progress } from "@/components/ui/progress";
import { 
  Users, 
  DollarSign, 
  Share2, 
  Copy, 
  CheckCircle, 
  TrendingUp,
  Gift,
  Target,
  Award,
  Zap
} from "lucide-react";
import Layout from "@/components/Layout";

const ReferPage = () => {
  const [copied, setCopied] = useState(false);

  const referralData = {
    totalReferrals: 24,
    totalEarnings: 120,
    pendingReferrals: 3,
    referralCode: "FLASH24",
    referralLink: "https://flashcore.com/ref/FLASH24",
    nextMilestone: 30,
    currentProgress: 24
  };

  const handleCopyLink = async () => {
    try {
      await navigator.clipboard.writeText(referralData.referralLink);
      setCopied(true);
      setTimeout(() => setCopied(false), 2000);
    } catch (err) {
      console.error('Failed to copy: ', err);
    }
  };

  return (
    <Layout>
      <div className="space-y-6">
        {/* Header with Logo */}
        <div className="flex items-center gap-3">
          <div className="w-8 h-8 bg-gradient-to-r from-flashcore-purple to-flashcore-green rounded-lg flex items-center justify-center">
            <Zap className="w-5 h-5 text-white" />
          </div>
          <div>
            <h1 className="text-3xl font-bold">Refer & Earn</h1>
            <p className="text-muted-foreground">
              Invite friends and earn rewards together
            </p>
          </div>
        </div>

        {/* Stats Overview */}
        <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
          <Card className="hover:shadow-lg transition-shadow">
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-muted-foreground">Total Referrals</p>
                  <p className="text-2xl font-bold">{referralData.totalReferrals}</p>
                  <div className="flex items-center mt-2">
                    <TrendingUp className="w-4 h-4 text-green-500 mr-1" />
                    <span className="text-sm text-green-500">+3 this week</span>
                  </div>
                </div>
                <Users className="w-8 h-8 text-flashcore-purple" />
              </div>
            </CardContent>
          </Card>

          <Card className="hover:shadow-lg transition-shadow">
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-muted-foreground">Total Earnings</p>
                  <p className="text-2xl font-bold">${referralData.totalEarnings}</p>
                  <p className="text-sm text-muted-foreground mt-2">Lifetime rewards</p>
                </div>
                <DollarSign className="w-8 h-8 text-flashcore-green" />
              </div>
            </CardContent>
          </Card>

          <Card className="hover:shadow-lg transition-shadow">
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-muted-foreground">Next Milestone</p>
                  <p className="text-2xl font-bold">{referralData.nextMilestone - referralData.currentProgress} more</p>
                  <div className="flex items-center mt-2">
                    <Target className="w-4 h-4 text-flashcore-orange mr-1" />
                    <span className="text-sm text-flashcore-orange">$200 reward</span>
                  </div>
                </div>
                <Award className="w-8 h-8 text-flashcore-orange" />
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Referral Link Section */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Share2 className="w-5 h-5" />
              Your Referral Link
            </CardTitle>
            <CardDescription>
              Share this link with friends and earn $5 for each successful referral
            </CardDescription>
          </CardHeader>
          <CardContent className="space-y-4">
            <div className="flex gap-2">
              <Input 
                value={referralData.referralLink} 
                readOnly 
                className="flex-1"
              />
              <Button 
                onClick={handleCopyLink}
                variant={copied ? "default" : "outline"}
                className={copied ? "bg-green-500 hover:bg-green-600" : ""}
              >
                {copied ? (
                  <>
                    <CheckCircle className="w-4 h-4 mr-2" />
                    Copied!
                  </>
                ) : (
                  <>
                    <Copy className="w-4 h-4 mr-2" />
                    Copy
                  </>
                )}
              </Button>
            </div>
            
            <div className="p-4 bg-muted/50 rounded-lg">
              <div className="flex items-center justify-between mb-2">
                <span className="text-sm font-medium">Progress to next milestone</span>
                <span className="text-sm text-muted-foreground">
                  {referralData.currentProgress}/{referralData.nextMilestone}
                </span>
              </div>
              <Progress 
                value={(referralData.currentProgress / referralData.nextMilestone) * 100} 
                className="h-2"
              />
              <p className="text-xs text-muted-foreground mt-2">
                {referralData.nextMilestone - referralData.currentProgress} more referrals to unlock $200
              </p>
            </div>
          </CardContent>
        </Card>

        {/* How It Works */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Zap className="w-5 h-5" />
              How It Works
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
              <div className="text-center">
                <div className="w-12 h-12 bg-flashcore-purple rounded-full flex items-center justify-center mx-auto mb-4">
                  <span className="text-white font-bold">1</span>
                </div>
                <h3 className="font-semibold mb-2">Share Your Link</h3>
                <p className="text-sm text-muted-foreground">
                  Copy and share your unique referral link with friends and family
                </p>
              </div>
              <div className="text-center">
                <div className="w-12 h-12 bg-flashcore-green rounded-full flex items-center justify-center mx-auto mb-4">
                  <span className="text-white font-bold">2</span>
                </div>
                <h3 className="font-semibold mb-2">They Sign Up</h3>
                <p className="text-sm text-muted-foreground">
                  When they sign up using your link, they get started with FlashCore
                </p>
              </div>
              <div className="text-center">
                <div className="w-12 h-12 bg-flashcore-orange rounded-full flex items-center justify-center mx-auto mb-4">
                  <span className="text-white font-bold">3</span>
                </div>
                <h3 className="font-semibold mb-2">Earn Rewards</h3>
                <p className="text-sm text-muted-foreground">
                  You both earn $5 instantly when they complete their first action
                </p>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>
    </Layout>
  );
};

export default ReferPage; 