import React from 'react';
import { Link } from 'react-router-dom';
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Check, X, ArrowRight, Star, Zap, Shield, Users, BarChart3 } from "lucide-react";
import Navigation from "@/components/Navigation";
import Footer from "@/components/Footer";

const PricingPage = () => {
  const plans = [
    {
      name: "Basic",
      price: "$9.99",
      description: "Perfect for small businesses",
      features: [
        "1,000 emails per month",
        "Basic email templates",
        "Email analytics",
        "24/7 support",
        "Standard delivery",
        "Basic reporting"
      ],
      notIncluded: [
        "Advanced templates",
        "Custom branding",
        "API access",
        "Priority support"
      ],
      color: "from-flashcore-purple to-flashcore-green",
      popular: false
    },
    {
      name: "Pro",
      price: "$29.99",
      description: "For growing businesses",
      features: [
        "10,000 emails per month",
        "Advanced email templates",
        "Advanced analytics",
        "Priority support",
        "Custom branding",
        "Advanced reporting",
        "A/B testing",
        "Automation workflows"
      ],
      notIncluded: [
        "Unlimited emails",
        "Dedicated support",
        "Custom integrations"
      ],
      color: "from-flashcore-green to-flashcore-purple",
      popular: true
    },
    {
      name: "Enterprise",
      price: "$99.99",
      description: "For large organizations",
      features: [
        "Unlimited emails",
        "Custom email templates",
        "Advanced reporting",
        "Dedicated support",
        "API access",
        "Custom integrations",
        "White-label options",
        "Advanced security"
      ],
      notIncluded: [],
      color: "from-flashcore-orange to-flashcore-purple",
      popular: false
    }
  ];

  return (
    <div className="min-h-screen bg-background">
      <Navigation />

      {/* Header */}
      <section className="container mx-auto px-4 py-12 md:py-20 text-center">
        <div className="max-w-4xl mx-auto">
          <Badge variant="secondary" className="mb-4">
            <Zap className="w-3 h-3 mr-1" />
            Simple Pricing
          </Badge>
          <h1 className="text-3xl md:text-4xl lg:text-5xl font-bold mb-6 bg-gradient-to-r from-flashcore-purple to-flashcore-green bg-clip-text text-transparent">
            Choose Your Plan
          </h1>
          <p className="text-lg md:text-xl text-muted-foreground mb-8 max-w-2xl mx-auto">
            Start free and scale as you grow. All plans include our core features with no hidden fees.
        </p>
      </div>
      </section>

      {/* Pricing Cards */}
      <section className="container mx-auto px-4 py-12 md:py-20">
        <div className="grid grid-cols-1 md:grid-cols-3 gap-6 md:gap-8 max-w-6xl mx-auto">
          {plans.map((plan, index) => (
            <Card key={index} className={`relative ${plan.popular ? 'border-2 border-flashcore-purple scale-105' : ''}`}>
            {plan.popular && (
              <div className="absolute -top-3 left-1/2 transform -translate-x-1/2">
                  <Badge className="bg-flashcore-purple text-white">
                    <Star className="w-3 h-3 mr-1" />
                    Most Popular
                  </Badge>
              </div>
            )}
            <CardHeader className="text-center">
                <CardTitle className="text-xl md:text-2xl">{plan.name}</CardTitle>
                <div className="text-3xl md:text-4xl font-bold bg-gradient-to-r bg-clip-text text-transparent" style={{backgroundImage: `linear-gradient(to right, var(--tw-gradient-stops))`}}>
                  {plan.price}<span className="text-base md:text-lg text-muted-foreground">/month</span>
                </div>
              <CardDescription>{plan.description}</CardDescription>
              </CardHeader>
              <CardContent className="space-y-6">
                <div className="space-y-3">
                  <h4 className="font-semibold text-sm text-muted-foreground uppercase tracking-wide">What's included:</h4>
                  {plan.features.map((feature, featureIndex) => (
                    <div key={featureIndex} className="flex items-center">
                      <Check className="w-4 h-4 text-green-500 mr-3 flex-shrink-0" />
                      <span className="text-sm md:text-base">{feature}</span>
                    </div>
                  ))}
              </div>
                
                {plan.notIncluded.length > 0 && (
                  <div className="space-y-3">
                    <h4 className="font-semibold text-sm text-muted-foreground uppercase tracking-wide">Not included:</h4>
                    {plan.notIncluded.map((feature, featureIndex) => (
                      <div key={featureIndex} className="flex items-center">
                        <X className="w-4 h-4 text-red-500 mr-3 flex-shrink-0" />
                        <span className="text-sm md:text-base text-muted-foreground">{feature}</span>
                      </div>
                    ))}
                  </div>
                )}
                
                <Link to="/register">
                  <Button className={`w-full bg-gradient-to-r ${plan.color}`}>
                    Get Started
                    <ArrowRight className="w-4 h-4 ml-2" />
              </Button>
                </Link>
            </CardContent>
          </Card>
        ))}
      </div>
      </section>

      {/* Comparison Table */}
      <section className="container mx-auto px-4 py-12 md:py-20">
        <div className="text-center mb-12 md:mb-16">
          <h2 className="text-2xl md:text-3xl font-bold mb-4">Feature Comparison</h2>
          <p className="text-muted-foreground max-w-2xl mx-auto">
            Compare all features across our plans to find the perfect fit for your business.
          </p>
        </div>
        
        <div className="overflow-x-auto">
          <table className="w-full border-collapse">
            <thead>
              <tr className="border-b border-border">
                <th className="text-left p-4 font-semibold">Feature</th>
                <th className="text-center p-4 font-semibold">Basic</th>
                <th className="text-center p-4 font-semibold">Pro</th>
                <th className="text-center p-4 font-semibold">Enterprise</th>
              </tr>
            </thead>
            <tbody>
              <tr className="border-b border-border/50">
                <td className="p-4 font-medium">Monthly Emails</td>
                <td className="p-4 text-center">1,000</td>
                <td className="p-4 text-center">10,000</td>
                <td className="p-4 text-center">Unlimited</td>
              </tr>
              <tr className="border-b border-border/50">
                <td className="p-4 font-medium">Email Templates</td>
                <td className="p-4 text-center">Basic</td>
                <td className="p-4 text-center">Advanced</td>
                <td className="p-4 text-center">Custom</td>
              </tr>
              <tr className="border-b border-border/50">
                <td className="p-4 font-medium">Analytics</td>
                <td className="p-4 text-center">Basic</td>
                <td className="p-4 text-center">Advanced</td>
                <td className="p-4 text-center">Advanced</td>
              </tr>
              <tr className="border-b border-border/50">
                <td className="p-4 font-medium">Support</td>
                <td className="p-4 text-center">24/7</td>
                <td className="p-4 text-center">Priority</td>
                <td className="p-4 text-center">Dedicated</td>
              </tr>
              <tr className="border-b border-border/50">
                <td className="p-4 font-medium">Custom Branding</td>
                <td className="p-4 text-center">
                  <X className="w-4 h-4 text-red-500 mx-auto" />
                </td>
                <td className="p-4 text-center">
                  <Check className="w-4 h-4 text-green-500 mx-auto" />
                </td>
                <td className="p-4 text-center">
                  <Check className="w-4 h-4 text-green-500 mx-auto" />
                </td>
              </tr>
              <tr className="border-b border-border/50">
                <td className="p-4 font-medium">API Access</td>
                <td className="p-4 text-center">
                  <X className="w-4 h-4 text-red-500 mx-auto" />
                </td>
                <td className="p-4 text-center">
                  <X className="w-4 h-4 text-red-500 mx-auto" />
                </td>
                <td className="p-4 text-center">
                  <Check className="w-4 h-4 text-green-500 mx-auto" />
                </td>
              </tr>
              <tr className="border-b border-border/50">
                <td className="p-4 font-medium">A/B Testing</td>
                <td className="p-4 text-center">
                  <X className="w-4 h-4 text-red-500 mx-auto" />
                </td>
                <td className="p-4 text-center">
                  <Check className="w-4 h-4 text-green-500 mx-auto" />
                </td>
                <td className="p-4 text-center">
                  <Check className="w-4 h-4 text-green-500 mx-auto" />
                </td>
              </tr>
            </tbody>
          </table>
        </div>
      </section>

      {/* FAQ Section */}
      <section className="container mx-auto px-4 py-12 md:py-20 bg-muted/20">
        <div className="text-center mb-12 md:mb-16">
          <h2 className="text-2xl md:text-3xl font-bold mb-4">Frequently Asked Questions</h2>
          <p className="text-muted-foreground max-w-2xl mx-auto">
            Get answers to common questions about our pricing and plans.
        </p>
      </div>
        
        <div className="grid grid-cols-1 md:grid-cols-2 gap-6 md:gap-8 max-w-4xl mx-auto">
          <Card>
            <CardHeader>
              <CardTitle className="text-lg">Can I change plans anytime?</CardTitle>
            </CardHeader>
            <CardContent>
              <p className="text-muted-foreground">
                Yes, you can upgrade or downgrade your plan at any time. Changes take effect immediately.
              </p>
            </CardContent>
          </Card>
          
          <Card>
            <CardHeader>
              <CardTitle className="text-lg">Is there a free trial?</CardTitle>
            </CardHeader>
            <CardContent>
              <p className="text-muted-foreground">
                Yes, all plans come with a 14-day free trial. No credit card required to start.
              </p>
            </CardContent>
          </Card>
          
          <Card>
            <CardHeader>
              <CardTitle className="text-lg">What payment methods do you accept?</CardTitle>
            </CardHeader>
            <CardContent>
              <p className="text-muted-foreground">
                We accept all major credit cards, PayPal, and bank transfers for Enterprise plans.
              </p>
            </CardContent>
          </Card>
          
          <Card>
            <CardHeader>
              <CardTitle className="text-lg">Do you offer refunds?</CardTitle>
            </CardHeader>
            <CardContent>
              <p className="text-muted-foreground">
                We offer a 30-day money-back guarantee. If you're not satisfied, we'll refund your payment.
              </p>
            </CardContent>
          </Card>
        </div>
      </section>

      {/* CTA Section */}
      <section className="container mx-auto px-4 py-12 md:py-20">
        <div className="bg-gradient-to-r from-flashcore-purple to-flashcore-green rounded-2xl p-6 md:p-8 lg:p-12 text-center text-white">
          <h2 className="text-2xl md:text-3xl font-bold mb-4">Ready to Get Started?</h2>
          <p className="text-lg md:text-xl mb-6 md:mb-8 opacity-90">
            Join thousands of businesses using FlashCore to automate their communications.
          </p>
          <div className="flex flex-col sm:flex-row gap-4 justify-center">
            <Link to="/register">
              <Button size="lg" variant="secondary" className="bg-white text-flashcore-purple hover:bg-gray-100">
                Start Free Trial
                <ArrowRight className="w-4 h-4 ml-2" />
              </Button>
            </Link>
            <Link to="/contact">
              <Button size="lg" variant="outline" className="border-white text-white hover:bg-white hover:text-flashcore-purple">
                Contact Sales
              </Button>
            </Link>
          </div>
        </div>
      </section>

      <Footer />
    </div>
  );
};

export default PricingPage;
