import React, { useState } from 'react';
import { Link } from 'react-router-dom';
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import { 
  Search,
  Calendar,
  ExternalLink,
  Download,
  FileText,
  Image,
  Video,
  Users,
  TrendingUp,
  Award,
  Globe,
  Mail,
  Phone,
  MapPin,
  ArrowRight,
  Filter,
  Newspaper,
  Play,
  Star,
  Quote,
  Building,
  Zap,
  Target,
  BarChart3
} from "lucide-react";
import Navigation from "@/components/Navigation";
import Footer from "@/components/Footer";

const PressPage = () => {
  const [searchQuery, setSearchQuery] = useState('');
  const [selectedCategory, setSelectedCategory] = useState('all');

  const categories = [
    { id: 'all', name: 'All News', icon: Newspaper },
    { id: 'press-releases', name: 'Press Releases', icon: FileText },
    { id: 'media-coverage', name: 'Media Coverage', icon: Globe },
    { id: 'company-news', name: 'Company News', icon: Building },
    { id: 'awards', name: 'Awards', icon: Award },
    { id: 'product-updates', name: 'Product Updates', icon: Zap }
  ];

  const pressReleases = [
    {
      id: 1,
      title: "FlashCore Raises $25M Series B Funding to Accelerate Email Marketing Innovation",
      category: "press-releases",
      date: "March 15, 2024",
      summary: "FlashCore announces $25M Series B funding round led by TechVentures Capital to expand product development and global market presence.",
      content: "FlashCore, the leading email marketing automation platform, today announced it has raised $25 million in Series B funding led by TechVentures Capital, with participation from existing investors GrowthFund and Innovation Partners. This funding will accelerate product development, expand the company's global presence, and strengthen its position as the premier email marketing solution for businesses of all sizes.",
      tags: ["Funding", "Series B", "Growth", "Investment"],
      featured: true,
      readTime: "3 min read"
    },
    {
      id: 2,
      title: "FlashCore Launches Advanced AI-Powered Email Personalization Engine",
      category: "product-updates",
      date: "March 8, 2024",
      summary: "New AI engine delivers personalized email content in real-time, increasing engagement rates by up to 300%.",
      content: "FlashCore today announced the launch of its revolutionary AI-powered email personalization engine, designed to transform how businesses connect with their customers. The new technology uses advanced machine learning algorithms to analyze customer behavior and deliver highly personalized email content in real-time, resulting in engagement rate increases of up to 300%.",
      tags: ["AI", "Personalization", "Product Launch", "Machine Learning"],
      featured: true,
      readTime: "4 min read"
    },
    {
      id: 3,
      title: "FlashCore Named 'Best Email Marketing Platform' at 2024 SaaS Awards",
      category: "awards",
      date: "February 28, 2024",
      summary: "FlashCore recognized for excellence in email marketing innovation and customer success.",
      content: "FlashCore has been named 'Best Email Marketing Platform' at the prestigious 2024 SaaS Awards, recognizing the company's innovative approach to email marketing automation and exceptional customer success rates. The award highlights FlashCore's commitment to delivering cutting-edge technology that empowers businesses to build meaningful customer relationships.",
      tags: ["Award", "Recognition", "SaaS", "Excellence"],
      featured: false,
      readTime: "2 min read"
    },
    {
      id: 4,
      title: "FlashCore Expands European Operations with New London Office",
      category: "company-news",
      date: "February 15, 2024",
      summary: "Strategic expansion into European market with dedicated team and localized support.",
      content: "FlashCore today announced the opening of its new European headquarters in London, marking a significant milestone in the company's global expansion strategy. The new office will serve as the hub for European operations, providing localized support and expertise to customers across the region.",
      tags: ["Expansion", "Europe", "London", "Global Growth"],
      featured: false,
      readTime: "3 min read"
    },
    {
      id: 5,
      title: "FlashCore Partners with Leading E-commerce Platforms for Seamless Integration",
      category: "company-news",
      date: "February 1, 2024",
      summary: "Strategic partnerships with Shopify, WooCommerce, and Magento enhance customer experience.",
      content: "FlashCore has announced strategic partnerships with leading e-commerce platforms including Shopify, WooCommerce, and Magento. These partnerships will provide seamless integration capabilities, enabling merchants to leverage FlashCore's advanced email marketing features directly within their existing e-commerce workflows.",
      tags: ["Partnership", "E-commerce", "Integration", "Shopify"],
      featured: false,
      readTime: "3 min read"
    },
    {
      id: 6,
      title: "FlashCore Introduces Enterprise-Grade Security and Compliance Features",
      category: "product-updates",
      date: "January 20, 2024",
      summary: "New security features meet enterprise requirements for data protection and compliance.",
      content: "FlashCore has launched comprehensive enterprise-grade security and compliance features, including advanced data encryption, GDPR compliance tools, and SOC 2 Type II certification. These enhancements ensure that enterprise customers can confidently use FlashCore while meeting their strict security and regulatory requirements.",
      tags: ["Security", "Compliance", "Enterprise", "GDPR"],
      featured: false,
      readTime: "4 min read"
    }
  ];

  const mediaCoverage = [
    {
      id: 1,
      title: "How FlashCore is Revolutionizing Email Marketing with AI",
      source: "TechCrunch",
      date: "March 10, 2024",
      author: "Sarah Johnson",
      summary: "An in-depth look at FlashCore's AI-powered approach to email marketing and its impact on the industry.",
      url: "https://techcrunch.com/flashcore-ai-email-marketing",
      logo: "techcrunch",
      featured: true
    },
    {
      id: 2,
      title: "FlashCore's $25M Funding Round Signals Growing Demand for Email Automation",
      source: "VentureBeat",
      date: "March 16, 2024",
      author: "Michael Chen",
      summary: "Analysis of FlashCore's recent funding round and the broader email marketing automation market.",
      url: "https://venturebeat.com/flashcore-funding-email-automation",
      logo: "venturebeat",
      featured: true
    },
    {
      id: 3,
      title: "The Future of Email Marketing: Interview with FlashCore CEO",
      source: "Forbes",
      date: "February 25, 2024",
      author: "Emily Rodriguez",
      summary: "Exclusive interview with FlashCore's CEO discussing the future of email marketing and company vision.",
      url: "https://forbes.com/flashcore-ceo-interview",
      logo: "forbes",
      featured: false
    },
    {
      id: 4,
      title: "FlashCore Named Top Email Marketing Platform for Small Businesses",
      source: "Business Insider",
      date: "February 10, 2024",
      author: "David Thompson",
      summary: "Review and ranking of FlashCore as the leading email marketing solution for small business needs.",
      url: "https://businessinsider.com/flashcore-small-business",
      logo: "business-insider",
      featured: false
    }
  ];

  const companyStats = [
    {
      icon: Users,
      number: "50,000+",
      label: "Active Customers",
      description: "Businesses worldwide trust FlashCore"
    },
    {
      icon: TrendingUp,
      number: "2.5B+",
      label: "Emails Sent",
      description: "Through our platform monthly"
    },
    {
      icon: Star,
      number: "98%",
      label: "Customer Satisfaction",
      description: "Based on verified reviews"
    },
    {
      icon: Globe,
      number: "150+",
      label: "Countries Served",
      description: "Global reach and support"
    }
  ];

  const mediaResources = [
    {
      title: "Company Logo",
      description: "High-resolution FlashCore logos in various formats",
      icon: Image,
      downloads: ["PNG", "SVG", "EPS"]
    },
    {
      title: "Product Screenshots",
      description: "Latest screenshots of FlashCore platform features",
      icon: Image,
      downloads: ["PNG", "JPG"]
    },
    {
      title: "Brand Guidelines",
      description: "Complete brand guidelines and style guide",
      icon: FileText,
      downloads: ["PDF"]
    },
    {
      title: "Company Fact Sheet",
      description: "Key facts and figures about FlashCore",
      icon: FileText,
      downloads: ["PDF"]
    },
    {
      title: "Executive Bios",
      description: "Biographies and photos of FlashCore leadership",
      icon: Users,
      downloads: ["PDF", "JPG"]
    },
    {
      title: "Product Videos",
      description: "Demo videos and product overviews",
      icon: Video,
      downloads: ["MP4", "WebM"]
    }
  ];

  const filteredPress = pressReleases.filter(item => {
    const matchesSearch = item.title.toLowerCase().includes(searchQuery.toLowerCase()) ||
                         item.summary.toLowerCase().includes(searchQuery.toLowerCase()) ||
                         item.tags.some(tag => tag.toLowerCase().includes(searchQuery.toLowerCase()));
    const matchesCategory = selectedCategory === 'all' || item.category === selectedCategory;
    return matchesSearch && matchesCategory;
  });

  return (
    <div className="min-h-screen bg-background">
      <Navigation />

      {/* Hero Section */}
      <section className="container mx-auto px-4 py-12 md:py-20">
        <div className="max-w-4xl mx-auto text-center">
          <Badge variant="secondary" className="mb-4">
            <Newspaper className="w-3 h-3 mr-1" />
            Press & Media
          </Badge>
          <h1 className="text-3xl md:text-4xl lg:text-5xl font-bold mb-6 bg-gradient-to-r from-flashcore-purple to-flashcore-green bg-clip-text text-transparent">
            FlashCore in the News
          </h1>
          <p className="text-lg md:text-xl text-muted-foreground mb-8 max-w-2xl mx-auto">
            Stay updated with the latest news, press releases, and media coverage 
            about FlashCore's mission to revolutionize email marketing.
          </p>
          
          {/* Search Bar */}
          <div className="max-w-2xl mx-auto mb-8">
            <div className="relative">
              <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-muted-foreground w-5 h-5" />
              <Input
                type="text"
                placeholder="Search press releases, news, and media coverage..."
                value={searchQuery}
                onChange={(e) => setSearchQuery(e.target.value)}
                className="pl-10 pr-4 py-3 text-lg"
              />
            </div>
          </div>
        </div>
      </section>

      {/* Filters */}
      <section className="container mx-auto px-4 py-8">
        <div className="max-w-6xl mx-auto">
          <div className="flex flex-wrap gap-2 justify-center">
            {categories.map((category) => (
              <Button
                key={category.id}
                variant={selectedCategory === category.id ? "default" : "outline"}
                size="sm"
                onClick={() => setSelectedCategory(category.id)}
                className="flex items-center space-x-2"
              >
                <category.icon className="w-4 h-4" />
                <span>{category.name}</span>
              </Button>
            ))}
          </div>
        </div>
      </section>

      {/* Featured Press Release */}
      {filteredPress.filter(item => item.featured).length > 0 && (
        <section className="container mx-auto px-4 py-12 md:py-20">
          <div className="max-w-6xl mx-auto">
            <div className="text-center mb-12">
              <h2 className="text-2xl md:text-3xl font-bold mb-4">Featured News</h2>
              <p className="text-muted-foreground">
                Latest major announcements and updates from FlashCore
              </p>
            </div>
            
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
              {filteredPress.filter(item => item.featured).map((item) => (
                <Card key={item.id} className="hover:shadow-lg transition-shadow group border-2 border-flashcore-purple/20">
                  <CardHeader>
                    <div className="flex items-center justify-between mb-4">
                      <Badge variant="destructive" className="flex items-center">
                        <Star className="w-3 h-3 mr-1" />
                        Featured
                      </Badge>
                      <Badge variant="secondary" className="text-xs">
                        {categories.find(c => c.id === item.category)?.name}
                      </Badge>
                    </div>
                    <CardTitle className="text-xl group-hover:text-flashcore-purple transition-colors">
                      {item.title}
                    </CardTitle>
                    <CardDescription className="text-sm">
                      {item.summary}
                    </CardDescription>
                  </CardHeader>
                  <CardContent>
                    <div className="space-y-4">
                      <div className="flex items-center justify-between text-sm">
                        <span className="flex items-center">
                          <Calendar className="w-4 h-4 mr-1" />
                          {item.date}
                        </span>
                        <span className="text-muted-foreground">
                          {item.readTime}
                        </span>
                      </div>
                      
                      <div className="flex flex-wrap gap-1">
                        {item.tags.map((tag, index) => (
                          <Badge key={index} variant="outline" className="text-xs">
                            {tag}
                          </Badge>
                        ))}
                      </div>

                      <Button className="w-full bg-flashcore-purple hover:bg-flashcore-purple/90">
                        Read Full Release
                        <ArrowRight className="w-4 h-4 ml-2" />
                      </Button>
                    </div>
                  </CardContent>
                </Card>
              ))}
            </div>
          </div>
        </section>
      )}

      {/* Press Releases */}
      <section className="container mx-auto px-4 py-12 md:py-20 bg-muted/20">
        <div className="max-w-6xl mx-auto">
          <div className="text-center mb-12">
            <h2 className="text-2xl md:text-3xl font-bold mb-4">Press Releases</h2>
            <p className="text-muted-foreground">
              Official announcements and updates from FlashCore
            </p>
          </div>
          
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-6 md:gap-8">
            {filteredPress.filter(item => !item.featured).map((item) => (
              <Card key={item.id} className="hover:shadow-lg transition-shadow group">
                <CardHeader>
                  <div className="flex items-center justify-between mb-4">
                    <Badge variant="secondary" className="text-xs">
                      {categories.find(c => c.id === item.category)?.name}
                    </Badge>
                    <span className="text-xs text-muted-foreground">
                      {item.readTime}
                    </span>
                  </div>
                  <CardTitle className="text-lg group-hover:text-flashcore-purple transition-colors">
                    {item.title}
                  </CardTitle>
                  <CardDescription className="text-sm">
                    {item.summary}
                  </CardDescription>
                </CardHeader>
                <CardContent>
                  <div className="space-y-4">
                    <div className="flex items-center justify-between text-sm">
                      <span className="flex items-center">
                        <Calendar className="w-4 h-4 mr-1" />
                        {item.date}
                      </span>
                    </div>
                    
                    <div className="flex flex-wrap gap-1">
                      {item.tags.slice(0, 3).map((tag, index) => (
                        <Badge key={index} variant="outline" className="text-xs">
                          {tag}
                        </Badge>
                      ))}
                    </div>

                    <Button variant="outline" className="w-full group-hover:bg-flashcore-purple group-hover:text-white transition-colors">
                      Read More
                      <ArrowRight className="w-4 h-4 ml-2" />
                    </Button>
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>

          {filteredPress.length === 0 && (
            <div className="text-center py-12">
              <Newspaper className="w-16 h-16 text-muted-foreground mx-auto mb-4" />
              <h3 className="text-xl font-semibold mb-2">No press releases found</h3>
              <p className="text-muted-foreground mb-4">
                Try adjusting your search criteria or check back later for updates.
              </p>
              <Button 
                variant="outline" 
                onClick={() => {
                  setSearchQuery('');
                  setSelectedCategory('all');
                }}
              >
                Clear Filters
              </Button>
            </div>
          )}
        </div>
      </section>

      {/* Media Coverage */}
      <section className="container mx-auto px-4 py-12 md:py-20">
        <div className="max-w-6xl mx-auto">
          <div className="text-center mb-12">
            <h2 className="text-2xl md:text-3xl font-bold mb-4">Media Coverage</h2>
            <p className="text-muted-foreground">
              Recent coverage of FlashCore in leading publications
            </p>
          </div>
          
          <div className="grid grid-cols-1 md:grid-cols-2 gap-6 md:gap-8">
            {mediaCoverage.map((coverage) => (
              <Card key={coverage.id} className="hover:shadow-lg transition-shadow group">
                <CardHeader>
                  <div className="flex items-center justify-between mb-4">
                    <Badge variant="outline" className="text-xs">
                      {coverage.source}
                    </Badge>
                    {coverage.featured && (
                      <Badge variant="destructive" className="text-xs">
                        Featured
                      </Badge>
                    )}
                  </div>
                  <CardTitle className="text-lg group-hover:text-flashcore-purple transition-colors">
                    {coverage.title}
                  </CardTitle>
                  <CardDescription className="text-sm">
                    {coverage.summary}
                  </CardDescription>
                </CardHeader>
                <CardContent>
                  <div className="space-y-4">
                    <div className="flex items-center justify-between text-sm">
                      <span className="flex items-center">
                        <Calendar className="w-4 h-4 mr-1" />
                        {coverage.date}
                      </span>
                      <span className="text-muted-foreground">
                        by {coverage.author}
                      </span>
                    </div>

                    <Button variant="outline" className="w-full group-hover:bg-flashcore-purple group-hover:text-white transition-colors">
                      <ExternalLink className="w-4 h-4 mr-2" />
                      Read Article
                    </Button>
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>
        </div>
      </section>

      {/* Company Stats */}
      <section className="container mx-auto px-4 py-12 md:py-20 bg-muted/20">
        <div className="max-w-6xl mx-auto">
          <div className="text-center mb-12">
            <h2 className="text-2xl md:text-3xl font-bold mb-4">FlashCore by the Numbers</h2>
            <p className="text-muted-foreground">
              Key metrics and achievements that define our success
            </p>
          </div>
          
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 md:gap-8">
            {companyStats.map((stat, index) => (
              <Card key={index} className="text-center hover:shadow-lg transition-shadow">
                <CardContent className="p-6">
                  <div className="w-12 h-12 bg-gradient-to-r from-flashcore-purple to-flashcore-green rounded-lg flex items-center justify-center mx-auto mb-4">
                    <stat.icon className="w-6 h-6 text-white" />
                  </div>
                  <h3 className="text-2xl font-bold mb-2">{stat.number}</h3>
                  <p className="text-sm font-semibold mb-2">{stat.label}</p>
                  <p className="text-xs text-muted-foreground">{stat.description}</p>
                </CardContent>
              </Card>
            ))}
          </div>
        </div>
      </section>

      {/* Media Resources */}
      <section className="container mx-auto px-4 py-12 md:py-20">
        <div className="max-w-6xl mx-auto">
          <div className="text-center mb-12">
            <h2 className="text-2xl md:text-3xl font-bold mb-4">Media Resources</h2>
            <p className="text-muted-foreground">
              Download logos, screenshots, and other media assets
            </p>
          </div>
          
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6 md:gap-8">
            {mediaResources.map((resource, index) => (
              <Card key={index} className="hover:shadow-lg transition-shadow">
                <CardContent className="p-6">
                  <div className="w-12 h-12 bg-gradient-to-r from-flashcore-purple to-flashcore-green rounded-lg flex items-center justify-center mb-4">
                    <resource.icon className="w-6 h-6 text-white" />
                  </div>
                  <h3 className="text-lg font-semibold mb-2">{resource.title}</h3>
                  <p className="text-sm text-muted-foreground mb-4">{resource.description}</p>
                  <div className="flex flex-wrap gap-2 mb-4">
                    {resource.downloads.map((format, idx) => (
                      <Badge key={idx} variant="outline" className="text-xs">
                        {format}
                      </Badge>
                    ))}
                  </div>
                  <Button variant="outline" className="w-full">
                    <Download className="w-4 h-4 mr-2" />
                    Download
                  </Button>
                </CardContent>
              </Card>
            ))}
          </div>
        </div>
      </section>

      {/* Contact Section */}
      <section className="container mx-auto px-4 py-12 md:py-20 bg-muted/20">
        <div className="max-w-4xl mx-auto text-center">
          <h2 className="text-2xl md:text-3xl font-bold mb-4">Media Inquiries</h2>
          <p className="text-muted-foreground mb-8">
            For press inquiries, interview requests, or media partnerships, 
            please contact our communications team.
          </p>
          
          <div className="grid grid-cols-1 md:grid-cols-3 gap-6 mb-8">
            <div className="flex items-center justify-center space-x-3">
              <Mail className="w-5 h-5 text-flashcore-purple" />
              <div className="text-left">
                <p className="font-semibold">Email</p>
                <p className="text-sm text-muted-foreground">press@flashcore.com</p>
              </div>
            </div>
            <div className="flex items-center justify-center space-x-3">
              <Phone className="w-5 h-5 text-flashcore-purple" />
              <div className="text-left">
                <p className="font-semibold">Phone</p>
                <p className="text-sm text-muted-foreground">+1 (555) 123-4567</p>
              </div>
            </div>
            <div className="flex items-center justify-center space-x-3">
              <MapPin className="w-5 h-5 text-flashcore-purple" />
              <div className="text-left">
                <p className="font-semibold">Office</p>
                <p className="text-sm text-muted-foreground">San Francisco, CA</p>
              </div>
            </div>
          </div>
          
          <div className="flex flex-col sm:flex-row gap-4 justify-center">
            <Button className="bg-flashcore-purple hover:bg-flashcore-purple/90">
              <Mail className="w-4 h-4 mr-2" />
              Contact Press Team
            </Button>
            <Button variant="outline">
              <Download className="w-4 h-4 mr-2" />
              Download Press Kit
            </Button>
          </div>
        </div>
      </section>

      <Footer />
    </div>
  );
};

export default PressPage; 