import React, { useState } from 'react';
import { Link } from 'react-router-dom';
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { ArrowRight, Search, Filter, Zap, Database, Cloud, Shield, Users, Settings, Check, ExternalLink, Code, Lock } from "lucide-react";
import Navigation from "@/components/Navigation";
import Footer from "@/components/Footer";

const IntegrationsPage = () => {
  const [searchTerm, setSearchTerm] = useState('');
  const [selectedCategory, setSelectedCategory] = useState('all');

  const categories = [
    { value: 'all', label: 'All Integrations' },
    { value: 'crm', label: 'CRM & Sales' },
    { value: 'ecommerce', label: 'E-commerce' },
    { value: 'marketing', label: 'Marketing' },
    { value: 'payment', label: 'Payment' },
    { value: 'analytics', label: 'Analytics' },
    { value: 'communication', label: 'Communication' }
  ];

  const integrations = [
    {
      id: 1,
      name: "Salesforce",
      category: "crm",
      description: "Sync customer data and automate sales processes",
      status: "popular",
      setup: "5 minutes",
      features: ["Contact sync", "Lead management", "Sales automation"],
      color: "from-blue-500 to-blue-600",
      icon: "SF"
    },
    {
      id: 2,
      name: "Shopify",
      category: "ecommerce",
      description: "Connect your online store and automate order notifications",
      status: "popular",
      setup: "3 minutes",
      features: ["Order sync", "Inventory updates", "Customer data"],
      color: "from-green-500 to-green-600",
      icon: "SP"
    },
    {
      id: 3,
      name: "Stripe",
      category: "payment",
      description: "Process payments and send automated receipts",
      status: "popular",
      setup: "2 minutes",
      features: ["Payment processing", "Receipt generation", "Refund handling"],
      color: "from-purple-500 to-purple-600",
      icon: "ST"
    },
    {
      id: 4,
      name: "Mailchimp",
      category: "marketing",
      description: "Sync email lists and campaign data",
      status: "new",
      setup: "4 minutes",
      features: ["List sync", "Campaign tracking", "Audience segmentation"],
      color: "from-orange-500 to-orange-600",
      icon: "MC"
    },
    {
      id: 5,
      name: "HubSpot",
      category: "crm",
      description: "Integrate with your marketing and sales platform",
      status: "popular",
      setup: "6 minutes",
      features: ["Contact sync", "Deal tracking", "Marketing automation"],
      color: "from-red-500 to-red-600",
      icon: "HS"
    },
    {
      id: 6,
      name: "Google Analytics",
      category: "analytics",
      description: "Track email performance and user behavior",
      status: "new",
      setup: "3 minutes",
      features: ["Event tracking", "Conversion tracking", "Audience insights"],
      color: "from-yellow-500 to-yellow-600",
      icon: "GA"
    },
    {
      id: 7,
      name: "Slack",
      category: "communication",
      description: "Get notifications and alerts in your workspace",
      status: "popular",
      setup: "2 minutes",
      features: ["Notifications", "Channel integration", "Custom alerts"],
      color: "from-purple-500 to-pink-500",
      icon: "SL"
    },
    {
      id: 8,
      name: "Zapier",
      category: "automation",
      description: "Connect with 5000+ apps and automate workflows",
      status: "new",
      setup: "5 minutes",
      features: ["Workflow automation", "Multi-app sync", "Custom triggers"],
      color: "from-orange-500 to-red-500",
      icon: "ZP"
    }
  ];

  const filteredIntegrations = integrations.filter(integration => {
    const matchesSearch = integration.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
                         integration.description.toLowerCase().includes(searchTerm.toLowerCase());
    const matchesCategory = selectedCategory === 'all' || integration.category === selectedCategory;
    return matchesSearch && matchesCategory;
  });

  return (
    <div className="min-h-screen bg-background">
      <Navigation />

      {/* Header */}
      <section className="container mx-auto px-4 py-12 md:py-20 text-center">
        <div className="max-w-4xl mx-auto">
          <Badge variant="secondary" className="mb-4">
            <Zap className="w-3 h-3 mr-1" />
            Third-Party Integrations
          </Badge>
          <h1 className="text-3xl md:text-4xl lg:text-5xl font-bold mb-6 bg-gradient-to-r from-flashcore-purple to-flashcore-green bg-clip-text text-transparent">
            Integrations
          </h1>
          <p className="text-lg md:text-xl text-muted-foreground mb-8 max-w-2xl mx-auto">
            Connect FlashCore with your favorite tools and platforms. Automate workflows and streamline your business processes.
          </p>
        </div>
      </section>

      {/* Search and Filter */}
      <section className="container mx-auto px-4 py-8">
        <div className="max-w-4xl mx-auto">
          <div className="flex flex-col md:flex-row gap-4">
            <div className="flex-1 relative">
              <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 w-4 h-4 text-muted-foreground" />
              <Input
                placeholder="Search integrations..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className="pl-10"
              />
            </div>
            <Select value={selectedCategory} onValueChange={setSelectedCategory}>
              <SelectTrigger className="w-full md:w-48">
                <Filter className="w-4 h-4 mr-2" />
                <SelectValue placeholder="Filter by category" />
              </SelectTrigger>
              <SelectContent>
                {categories.map((category) => (
                  <SelectItem key={category.value} value={category.value}>
                    {category.label}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>
        </div>
      </section>

      {/* Integrations Grid */}
      <section className="container mx-auto px-4 py-12 md:py-20">
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6 md:gap-8">
          {filteredIntegrations.map((integration) => (
            <Card key={integration.id} className="hover:shadow-lg transition-shadow group">
              <CardHeader>
                <div className="flex items-center justify-between mb-4">
                  <div className={`w-12 h-12 bg-gradient-to-r ${integration.color} rounded-lg flex items-center justify-center`}>
                    <span className="text-white font-bold text-sm">{integration.icon}</span>
                  </div>
                  <div className="flex items-center space-x-2">
                    {integration.status === 'popular' && (
                      <Badge className="bg-flashcore-purple text-white text-xs">
                        Popular
                      </Badge>
                    )}
                    {integration.status === 'new' && (
                      <Badge className="bg-green-500 text-white text-xs">
                        New
                      </Badge>
                    )}
                  </div>
                </div>
                <CardTitle className="text-lg md:text-xl">{integration.name}</CardTitle>
                <CardDescription className="text-sm md:text-base">
                  {integration.description}
                </CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="flex items-center justify-between text-sm">
                  <span className="text-muted-foreground">Setup time:</span>
                  <span className="font-medium">{integration.setup}</span>
                </div>
                
                <div className="space-y-2">
                  <h4 className="text-sm font-semibold">Key Features:</h4>
                  <ul className="space-y-1">
                    {integration.features.map((feature, index) => (
                      <li key={index} className="flex items-center text-sm text-muted-foreground">
                        <Check className="w-3 h-3 text-green-500 mr-2 flex-shrink-0" />
                        {feature}
                      </li>
                    ))}
                  </ul>
                </div>
                
                <div className="flex space-x-2">
                  <Button variant="outline" size="sm" className="flex-1">
                    <ExternalLink className="w-4 h-4 mr-2" />
                    Learn More
                  </Button>
                  <Button size="sm" className="flex-1 bg-gradient-to-r from-flashcore-purple to-flashcore-green">
                    <Settings className="w-4 h-4 mr-2" />
                    Connect
                  </Button>
                </div>
              </CardContent>
            </Card>
          ))}
        </div>
        
        {filteredIntegrations.length === 0 && (
          <div className="text-center py-12">
            <Zap className="w-16 h-16 text-muted-foreground mx-auto mb-4" />
            <h3 className="text-xl font-semibold mb-2">No integrations found</h3>
            <p className="text-muted-foreground">
              Try adjusting your search terms or filter criteria.
            </p>
          </div>
        )}
      </section>

      {/* Integration Categories */}
      <section className="container mx-auto px-4 py-12 md:py-20 bg-muted/20">
        <div className="text-center mb-12 md:mb-16">
          <h2 className="text-2xl md:text-3xl font-bold mb-4">Integration Categories</h2>
          <p className="text-muted-foreground max-w-2xl mx-auto">
            Connect with the tools you already use and love.
          </p>
        </div>
        
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 md:gap-8 max-w-6xl mx-auto">
          <Card className="text-center hover:shadow-lg transition-shadow">
            <CardHeader>
              <div className="w-12 h-12 bg-gradient-to-r from-blue-500 to-blue-600 rounded-lg flex items-center justify-center mx-auto mb-4">
                <Users className="w-6 h-6 text-white" />
              </div>
              <CardTitle className="text-lg">CRM & Sales</CardTitle>
            </CardHeader>
            <CardContent>
              <p className="text-sm text-muted-foreground mb-4">
                Sync customer data and automate sales processes.
              </p>
              <Badge variant="secondary">12 integrations</Badge>
            </CardContent>
          </Card>
          
          <Card className="text-center hover:shadow-lg transition-shadow">
            <CardHeader>
              <div className="w-12 h-12 bg-gradient-to-r from-green-500 to-green-600 rounded-lg flex items-center justify-center mx-auto mb-4">
                <Database className="w-6 h-6 text-white" />
              </div>
              <CardTitle className="text-lg">E-commerce</CardTitle>
            </CardHeader>
            <CardContent>
              <p className="text-sm text-muted-foreground mb-4">
                Connect your online stores and automate order processing.
              </p>
              <Badge variant="secondary">8 integrations</Badge>
            </CardContent>
          </Card>
          
          <Card className="text-center hover:shadow-lg transition-shadow">
            <CardHeader>
              <div className="w-12 h-12 bg-gradient-to-r from-purple-500 to-purple-600 rounded-lg flex items-center justify-center mx-auto mb-4">
                <Cloud className="w-6 h-6 text-white" />
              </div>
              <CardTitle className="text-lg">Payment</CardTitle>
            </CardHeader>
            <CardContent>
              <p className="text-sm text-muted-foreground mb-4">
                Process payments and generate automated receipts.
              </p>
              <Badge variant="secondary">6 integrations</Badge>
            </CardContent>
          </Card>
          
          <Card className="text-center hover:shadow-lg transition-shadow">
            <CardHeader>
              <div className="w-12 h-12 bg-gradient-to-r from-orange-500 to-orange-600 rounded-lg flex items-center justify-center mx-auto mb-4">
                <Zap className="w-6 h-6 text-white" />
              </div>
              <CardTitle className="text-lg">Marketing</CardTitle>
            </CardHeader>
            <CardContent>
              <p className="text-sm text-muted-foreground mb-4">
                Sync email lists and track campaign performance.
              </p>
              <Badge variant="secondary">15 integrations</Badge>
            </CardContent>
          </Card>
        </div>
      </section>

      {/* API Section */}
      <section className="container mx-auto px-4 py-12 md:py-20">
        <div className="max-w-4xl mx-auto text-center">
          <div className="bg-gradient-to-r from-flashcore-purple to-flashcore-green rounded-2xl p-6 md:p-8 lg:p-12 text-white">
            <div className="w-16 h-16 bg-white/20 rounded-full flex items-center justify-center mx-auto mb-6">
              <Code className="w-8 h-8 text-white" />
            </div>
            <h2 className="text-2xl md:text-3xl font-bold mb-4">Build Custom Integrations</h2>
            <p className="text-lg md:text-xl mb-6 md:mb-8 opacity-90">
              Use our powerful API to build custom integrations and automate your workflows.
            </p>
            <div className="flex flex-col sm:flex-row gap-4 justify-center">
              <Link to="/api">
                <Button size="lg" variant="secondary" className="bg-white text-flashcore-purple hover:bg-gray-100">
                  <Code className="w-4 h-4 mr-2" />
                  View API Docs
                </Button>
              </Link>
              <Link to="/contact">
                <Button size="lg" variant="outline" className="border-white text-white hover:bg-white hover:text-flashcore-purple">
                  <ExternalLink className="w-4 h-4 mr-2" />
                  Request Integration
                </Button>
              </Link>
            </div>
          </div>
        </div>
      </section>

      {/* Security Section */}
      <section className="container mx-auto px-4 py-12 md:py-20 bg-muted/20">
        <div className="max-w-4xl mx-auto">
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-8 md:gap-12 items-center">
            <div>
              <h2 className="text-2xl md:text-3xl font-bold mb-4 md:mb-6">Enterprise-Grade Security</h2>
              <p className="text-base md:text-lg text-muted-foreground mb-4 md:mb-6">
                All integrations are built with security in mind. Your data is encrypted and protected at every step.
              </p>
              <div className="space-y-4">
                <div className="flex items-center">
                  <Shield className="w-5 h-5 text-green-500 mr-3" />
                  <span className="text-sm md:text-base">End-to-end encryption</span>
                </div>
                <div className="flex items-center">
                  <Lock className="w-5 h-5 text-green-500 mr-3" />
                  <span className="text-sm md:text-base">SOC 2 Type II compliant</span>
                </div>
                <div className="flex items-center">
                  <Check className="w-5 h-5 text-green-500 mr-3" />
                  <span className="text-sm md:text-base">GDPR compliant</span>
                </div>
              </div>
            </div>
            <div className="relative">
              <div className="bg-gradient-to-r from-flashcore-purple to-flashcore-green rounded-2xl p-6 md:p-8 text-white">
                <h3 className="text-xl md:text-2xl font-bold mb-4">Trusted by 10,000+ businesses</h3>
                <p className="text-sm md:text-base opacity-90 mb-6">
                  Join thousands of businesses that trust FlashCore with their data and integrations.
                </p>
                <div className="grid grid-cols-2 gap-4">
                  <div className="text-center">
                    <div className="text-2xl md:text-3xl font-bold mb-1">99.9%</div>
                    <div className="text-xs md:text-sm opacity-90">Uptime</div>
                  </div>
                  <div className="text-center">
                    <div className="text-2xl md:text-3xl font-bold mb-1">24/7</div>
                    <div className="text-xs md:text-sm opacity-90">Support</div>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* CTA Section */}
      <section className="container mx-auto px-4 py-12 md:py-20">
        <div className="bg-gradient-to-r from-flashcore-purple to-flashcore-green rounded-2xl p-6 md:p-8 lg:p-12 text-center text-white">
          <h2 className="text-2xl md:text-3xl font-bold mb-4">Ready to Connect?</h2>
          <p className="text-lg md:text-xl mb-6 md:mb-8 opacity-90">
            Start integrating FlashCore with your favorite tools today.
          </p>
          <div className="flex flex-col sm:flex-row gap-4 justify-center">
            <Link to="/register">
              <Button size="lg" variant="secondary" className="bg-white text-flashcore-purple hover:bg-gray-100">
                Start Free Trial
                <ArrowRight className="w-4 h-4 ml-2" />
              </Button>
            </Link>
            <Link to="/contact">
              <Button size="lg" variant="outline" className="border-white text-white hover:bg-white hover:text-flashcore-purple">
                Contact Sales
              </Button>
            </Link>
          </div>
        </div>
      </section>

      <Footer />
    </div>
  );
};

export default IntegrationsPage; 