import React, { useState } from "react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Badge } from "@/components/ui/badge";
import { Accordion, AccordionContent, AccordionItem, AccordionTrigger } from "@/components/ui/accordion";
import { 
  Search, 
  HelpCircle, 
  BookOpen, 
  MessageSquare, 
  Phone, 
  Mail,
  Video,
  FileText,
  Users,
  Zap,
  Shield,
  CreditCard,
  Settings,
  Globe,
  Smartphone,
  TrendingUp
} from "lucide-react";
import Layout from "@/components/Layout";

const HelpPage = () => {
  const [searchQuery, setSearchQuery] = useState("");

  const categories = [
    {
      icon: Zap,
      title: "Getting Started",
      description: "Learn the basics of FlashCore",
      color: "text-blue-500",
      bgColor: "bg-blue-50"
    },
    {
      icon: Settings,
      title: "Configuration",
      description: "Set up your dashboard",
      color: "text-green-500",
      bgColor: "bg-green-50"
    },
    {
      icon: Shield,
      title: "Security",
      description: "Account and data protection",
      color: "text-red-500",
      bgColor: "bg-red-50"
    },
    {
      icon: CreditCard,
      title: "Billing",
      description: "Payment and subscription",
      color: "text-purple-500",
      bgColor: "bg-purple-50"
    },
    {
      icon: Globe,
      title: "Integrations",
      description: "Connect with other services",
      color: "text-orange-500",
      bgColor: "bg-orange-50"
    },
    {
      icon: Smartphone,
      title: "Mobile App",
      description: "Using FlashCore on mobile",
      color: "text-indigo-500",
      bgColor: "bg-indigo-50"
    }
  ];

  const faqs = [
    {
      category: "Getting Started",
      questions: [
        {
          question: "How do I create my first project?",
          answer: "To create your first project, click on the 'New Project' button in the dashboard. Fill in the project details including name, description, and team members. You can also choose from our project templates to get started quickly."
        },
        {
          question: "What are the system requirements?",
          answer: "FlashCore works on all modern browsers including Chrome, Firefox, Safari, and Edge. For the best experience, we recommend using the latest version of your browser and a stable internet connection."
        },
        {
          question: "How do I invite team members?",
          answer: "Go to your project settings and click on 'Team Members'. You can invite users by email address. They'll receive an invitation email with a link to join your project."
        }
      ]
    },
    {
      category: "Configuration",
      questions: [
        {
          question: "How do I customize my dashboard?",
          answer: "You can customize your dashboard by clicking on the settings icon in the top right corner. From there, you can add or remove widgets, change the layout, and set your preferences."
        },
        {
          question: "Can I integrate with other tools?",
          answer: "Yes! FlashCore supports integrations with popular tools like Slack, GitHub, Jira, and many more. Go to the Integrations section in your settings to see all available options."
        },
        {
          question: "How do I set up notifications?",
          answer: "Navigate to Settings > Notifications to configure your notification preferences. You can choose to receive notifications via email, push notifications, or SMS for different types of events."
        }
      ]
    },
    {
      category: "Security",
      questions: [
        {
          question: "How secure is my data?",
          answer: "Your data is protected with enterprise-grade security measures including end-to-end encryption, regular security audits, and compliance with industry standards like SOC 2 and GDPR."
        },
        {
          question: "How do I enable two-factor authentication?",
          answer: "Go to Settings > Security and click on 'Enable Two-Factor Authentication'. You'll need to download an authenticator app and scan the QR code to complete the setup."
        },
        {
          question: "What happens if I forget my password?",
          answer: "Click on 'Forgot Password' on the login page. You'll receive an email with a secure link to reset your password. Make sure to use a strong, unique password."
        }
      ]
    },
    {
      category: "Billing",
      questions: [
        {
          question: "What payment methods do you accept?",
          answer: "We accept all major credit cards (Visa, MasterCard, American Express), PayPal, and bank transfers for annual plans. All payments are processed securely through Stripe."
        },
        {
          question: "Can I cancel my subscription anytime?",
          answer: "Yes, you can cancel your subscription at any time from your billing settings. You'll continue to have access to your account until the end of your current billing period."
        },
        {
          question: "Do you offer refunds?",
          answer: "We offer a 30-day money-back guarantee for all new subscriptions. If you're not satisfied with our service, contact our support team within 30 days for a full refund."
        }
      ]
    }
  ];

  const supportOptions = [
    {
      icon: MessageSquare,
      title: "Live Chat",
      description: "Chat with our support team",
      response: "Usually responds in 5 minutes",
      color: "text-green-500",
      bgColor: "bg-green-50"
    },
    {
      icon: Mail,
      title: "Email Support",
      description: "Send us a detailed message",
      response: "Response within 24 hours",
      color: "text-blue-500",
      bgColor: "bg-blue-50"
    },
    {
      icon: Phone,
      title: "Phone Support",
      description: "Call us directly",
      response: "Available 9AM-6PM EST",
      color: "text-purple-500",
      bgColor: "bg-purple-50"
    },
    {
      icon: Video,
      title: "Video Call",
      description: "Schedule a screen share session",
      response: "Book a 30-minute slot",
      color: "text-orange-500",
      bgColor: "bg-orange-50"
    }
  ];

  const filteredFaqs = faqs.map(category => ({
    ...category,
    questions: category.questions.filter(q => 
      q.question.toLowerCase().includes(searchQuery.toLowerCase()) ||
      q.answer.toLowerCase().includes(searchQuery.toLowerCase())
    )
  })).filter(category => category.questions.length > 0);

  return (
    <Layout>
      <div className="space-y-6">
        {/* Header with Logo */}
        <div className="flex items-center gap-3">
          <div className="w-8 h-8 bg-gradient-to-r from-flashcore-purple to-flashcore-green rounded-lg flex items-center justify-center">
            <Zap className="w-5 h-5 text-white" />
          </div>
          <div>
            <h1 className="text-3xl font-bold">Help Center</h1>
            <p className="text-muted-foreground">
              Find answers to your questions and get the support you need
            </p>
          </div>
        </div>

        {/* Search */}
        <Card>
          <CardContent className="p-6">
            <div className="relative max-w-2xl mx-auto">
              <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-muted-foreground w-5 h-5" />
              <Input
                placeholder="Search for help articles, FAQs, or topics..."
                value={searchQuery}
                onChange={(e) => setSearchQuery(e.target.value)}
                className="pl-10 pr-4 py-3 text-lg"
              />
            </div>
          </CardContent>
        </Card>

        {/* Categories */}
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
          {categories.map((category, index) => (
            <Card key={index} className="hover:shadow-lg transition-shadow cursor-pointer">
              <CardContent className="p-6">
                <div className="flex items-start gap-4">
                  <div className={`w-12 h-12 ${category.bgColor} rounded-lg flex items-center justify-center flex-shrink-0`}>
                    <category.icon className={`w-6 h-6 ${category.color}`} />
                  </div>
                  <div>
                    <h3 className="font-semibold mb-1">{category.title}</h3>
                    <p className="text-sm text-muted-foreground">{category.description}</p>
                  </div>
                </div>
              </CardContent>
            </Card>
          ))}
        </div>

        {/* Support Options */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Users className="w-5 h-5" />
              Get Support
            </CardTitle>
            <CardDescription>
              Choose the best way to get help from our support team
            </CardDescription>
          </CardHeader>
          <CardContent>
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
              {supportOptions.map((option, index) => (
                <div key={index} className={`p-4 ${option.bgColor} rounded-lg`}>
                  <div className="flex items-center gap-3 mb-3">
                    <option.icon className={`w-5 h-5 ${option.color}`} />
                    <h3 className="font-semibold">{option.title}</h3>
                  </div>
                  <p className="text-sm text-muted-foreground mb-2">{option.description}</p>
                  <p className="text-xs text-muted-foreground">{option.response}</p>
                  <Button variant="outline" size="sm" className="mt-3 w-full">
                    Get Help
                  </Button>
                </div>
              ))}
            </div>
          </CardContent>
        </Card>

        {/* FAQ Sections */}
        {filteredFaqs.map((category, categoryIndex) => (
          <Card key={categoryIndex}>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <HelpCircle className="w-5 h-5" />
                {category.category}
              </CardTitle>
            </CardHeader>
            <CardContent>
              <Accordion type="single" collapsible className="w-full">
                {category.questions.map((faq, faqIndex) => (
                  <AccordionItem key={faqIndex} value={`item-${categoryIndex}-${faqIndex}`}>
                    <AccordionTrigger className="text-left">
                      {faq.question}
                    </AccordionTrigger>
                    <AccordionContent className="text-muted-foreground">
                      {faq.answer}
                    </AccordionContent>
                  </AccordionItem>
                ))}
              </Accordion>
            </CardContent>
          </Card>
        ))}

        {/* Quick Actions */}
        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <BookOpen className="w-5 h-5" />
                Documentation
              </CardTitle>
            </CardHeader>
            <CardContent>
              <p className="text-muted-foreground mb-4">
                Explore our comprehensive documentation with step-by-step guides and tutorials.
              </p>
              <div className="space-y-2">
                <Button variant="outline" className="w-full justify-start">
                  <FileText className="w-4 h-4 mr-2" />
                  User Guide
                </Button>
                <Button variant="outline" className="w-full justify-start">
                  <Video className="w-4 h-4 mr-2" />
                  Video Tutorials
                </Button>
                <Button variant="outline" className="w-full justify-start">
                  <TrendingUp className="w-4 h-4 mr-2" />
                  Best Practices
                </Button>
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <MessageSquare className="w-5 h-5" />
                Community
              </CardTitle>
            </CardHeader>
            <CardContent>
              <p className="text-muted-foreground mb-4">
                Join our community to connect with other users and share experiences.
              </p>
              <div className="space-y-2">
                <Button variant="outline" className="w-full justify-start">
                  <Users className="w-4 h-4 mr-2" />
                  Community Forum
                </Button>
                <Button variant="outline" className="w-full justify-start">
                  <Globe className="w-4 h-4 mr-2" />
                  Discord Server
                </Button>
                <Button variant="outline" className="w-full justify-start">
                  <TrendingUp className="w-4 h-4 mr-2" />
                  Feature Requests
                </Button>
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Contact CTA */}
        <Card className="bg-gradient-to-r from-flashcore-purple to-flashcore-green text-white">
          <CardContent className="p-6 text-center">
            <h2 className="text-xl font-bold mb-2">Still Need Help?</h2>
            <p className="mb-4 opacity-90">
              Our support team is here to help you succeed with FlashCore.
            </p>
            <div className="flex flex-col sm:flex-row gap-3 justify-center">
              <Button variant="secondary" className="bg-white text-flashcore-purple hover:bg-gray-100">
                <Mail className="w-4 h-4 mr-2" />
                Contact Support
              </Button>
              <Button variant="outline" className="border-white text-white hover:bg-white hover:text-flashcore-purple">
                <Phone className="w-4 h-4 mr-2" />
                Call Us
              </Button>
            </div>
          </CardContent>
        </Card>
      </div>
    </Layout>
  );
};

export default HelpPage; 