import React, { useState } from 'react';
import { Link } from 'react-router-dom';
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import { 
  ArrowRight, 
  Search, 
  Check, 
  Star, 
  HelpCircle, 
  BookOpen, 
  MessageSquare, 
  Headphones,
  Mail,
  Video,
  FileText,
  Users,
  Zap,
  Settings,
  Globe,
  Shield,
  TrendingUp,
  Target,
  Building,
  ShoppingCart
} from "lucide-react";
import Navigation from "@/components/Navigation";
import Footer from "@/components/Footer";

const HelpCenterPage = () => {
  const [searchQuery, setSearchQuery] = useState('');

  const categories = [
    {
      icon: Zap,
      title: "Getting Started",
      description: "Quick setup guides and tutorials for new users",
      color: "from-blue-500 to-blue-600",
      articles: [
        "How to create your first email campaign",
        "Setting up your account and preferences",
        "Importing your contact list",
        "Understanding the dashboard"
      ]
    },
    {
      icon: Settings,
      title: "Account & Billing",
      description: "Manage your account, billing, and subscription",
      color: "from-green-500 to-green-600",
      articles: [
        "Updating your account information",
        "Managing your subscription",
        "Understanding your billing cycle",
        "Adding team members"
      ]
    },
    {
      icon: Target,
      title: "Campaigns & Automation",
      description: "Create and manage email campaigns and workflows",
      color: "from-purple-500 to-purple-600",
      articles: [
        "Creating email campaigns",
        "Setting up automation workflows",
        "A/B testing your emails",
        "Campaign analytics and reporting"
      ]
    },
    {
      icon: Users,
      title: "Contacts & Lists",
      description: "Manage your contact lists and segmentation",
      color: "from-orange-500 to-orange-600",
      articles: [
        "Importing and managing contacts",
        "Creating contact segments",
        "Understanding contact scoring",
        "GDPR compliance and data management"
      ]
    },
    {
      icon: Globe,
      title: "Integrations",
      description: "Connect FlashCore with your favorite tools",
      color: "from-pink-500 to-pink-600",
      articles: [
        "Shopify integration setup",
        "WooCommerce connection",
        "API documentation and usage",
        "Third-party integrations"
      ]
    },
    {
      icon: Shield,
      title: "Security & Compliance",
      description: "Security features and compliance information",
      color: "from-red-500 to-red-600",
      articles: [
        "Data security and encryption",
        "GDPR compliance guide",
        "Email authentication (SPF, DKIM)",
        "Privacy and data protection"
      ]
    }
  ];

  const popularArticles = [
    {
      title: "How to create your first email campaign",
      category: "Getting Started",
      views: "15,234",
      rating: 4.8
    },
    {
      title: "Setting up Shopify integration",
      category: "Integrations",
      views: "12,891",
      rating: 4.9
    },
    {
      title: "Understanding email deliverability",
      category: "Campaigns & Automation",
      views: "11,567",
      rating: 4.7
    },
    {
      title: "Managing your subscription and billing",
      category: "Account & Billing",
      views: "9,843",
      rating: 4.6
    },
    {
      title: "Creating effective email templates",
      category: "Campaigns & Automation",
      views: "8,921",
      rating: 4.8
    },
    {
      title: "GDPR compliance checklist",
      category: "Security & Compliance",
      views: "7,654",
      rating: 4.9
    }
  ];

  const supportOptions = [
    {
      icon: BookOpen,
      title: "Knowledge Base",
      description: "Browse our comprehensive documentation and guides",
      action: "Browse Articles",
      link: "/docs"
    },
    {
      icon: Video,
      title: "Video Tutorials",
      description: "Watch step-by-step video guides and tutorials",
      action: "Watch Videos",
      link: "/tutorials"
    },
    {
      icon: MessageSquare,
      title: "Live Chat",
      description: "Get instant help from our support team",
      action: "Start Chat",
      link: "/contact"
    },
    {
      icon: Mail,
      title: "Email Support",
      description: "Send us a detailed message and we'll respond within 24 hours",
      action: "Send Email",
      link: "/contact"
    },
    {
      icon: Headphones,
      title: "Phone Support",
      description: "Call us directly for immediate assistance (Enterprise customers)",
      action: "Call Now",
      link: "/contact"
    },
    {
      icon: Users,
      title: "Community Forum",
      description: "Connect with other FlashCore users and share tips",
      action: "Join Community",
      link: "/community"
    }
  ];

  const faqs = [
    {
      question: "How do I get started with FlashCore?",
      answer: "Getting started is easy! Simply sign up for a free trial, verify your email, and follow our step-by-step onboarding guide. You'll be sending your first campaign within minutes."
    },
    {
      question: "What email marketing features does FlashCore offer?",
      answer: "FlashCore offers comprehensive email marketing features including campaign creation, automation workflows, A/B testing, advanced analytics, contact segmentation, and integrations with popular platforms."
    },
    {
      question: "How much does FlashCore cost?",
      answer: "FlashCore offers flexible pricing starting at $19/month for startups, $29/month for small businesses, $49/month for e-commerce, $99/month for agencies, and custom pricing for enterprise customers."
    },
    {
      question: "Can I import my existing contact list?",
      answer: "Yes! You can import contacts from CSV files, or connect directly with platforms like Shopify, WooCommerce, and other popular integrations. We also provide tools to clean and validate your data."
    },
    {
      question: "What kind of support do you provide?",
      answer: "We offer comprehensive support including a knowledge base, video tutorials, live chat, email support, and phone support for enterprise customers. Our team typically responds within 24 hours."
    },
    {
      question: "Is my data secure with FlashCore?",
      answer: "Absolutely! We use bank-level encryption, SOC 2 compliance, and enterprise-grade security measures to protect your data. We're also GDPR compliant and take data privacy seriously."
    }
  ];

  return (
    <div className="min-h-screen bg-background">
      <Navigation />

      {/* Hero Section */}
      <section className="container mx-auto px-4 py-12 md:py-20">
        <div className="max-w-4xl mx-auto text-center">
          <Badge variant="secondary" className="mb-4">
            <HelpCircle className="w-3 h-3 mr-1" />
            Help Center
          </Badge>
          <h1 className="text-3xl md:text-4xl lg:text-5xl font-bold mb-6 bg-gradient-to-r from-flashcore-purple to-flashcore-green bg-clip-text text-transparent">
            How Can We Help You?
          </h1>
          <p className="text-lg md:text-xl text-muted-foreground mb-8 max-w-2xl mx-auto">
            Find answers to your questions, learn new features, and get the support you need 
            to make the most of FlashCore.
          </p>
          
          {/* Search Bar */}
          <div className="max-w-2xl mx-auto mb-8">
            <div className="relative">
              <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-muted-foreground w-5 h-5" />
              <Input
                type="text"
                placeholder="Search for help articles, tutorials, or features..."
                value={searchQuery}
                onChange={(e) => setSearchQuery(e.target.value)}
                className="pl-10 pr-4 py-3 text-lg"
              />
            </div>
          </div>
        </div>
      </section>

      {/* Support Options Section */}
      <section className="container mx-auto px-4 py-12 md:py-20 bg-muted/20">
        <div className="max-w-6xl mx-auto">
          <div className="text-center mb-12">
            <h2 className="text-2xl md:text-3xl font-bold mb-4">Get the Help You Need</h2>
            <p className="text-muted-foreground">
              Multiple ways to get support and find answers to your questions.
            </p>
          </div>
          
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6 md:gap-8">
            {supportOptions.map((option, index) => (
              <Card key={index} className="hover:shadow-lg transition-shadow">
                <CardContent className="p-6">
                  <div className="w-12 h-12 bg-gradient-to-r from-flashcore-purple to-flashcore-green rounded-lg flex items-center justify-center mb-4">
                    <option.icon className="w-6 h-6 text-white" />
                  </div>
                  <h3 className="text-lg font-semibold mb-2">{option.title}</h3>
                  <p className="text-sm text-muted-foreground mb-4">
                    {option.description}
                  </p>
                  <Link to={option.link}>
                    <Button variant="outline" size="sm" className="w-full">
                      {option.action}
                      <ArrowRight className="w-4 h-4 ml-2" />
                    </Button>
                  </Link>
                </CardContent>
              </Card>
            ))}
          </div>
        </div>
      </section>

      {/* Categories Section */}
      <section className="container mx-auto px-4 py-12 md:py-20">
        <div className="max-w-6xl mx-auto">
          <div className="text-center mb-12">
            <h2 className="text-2xl md:text-3xl font-bold mb-4">Browse by Category</h2>
            <p className="text-muted-foreground">
              Find help articles organized by topic and feature.
            </p>
          </div>
          
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6 md:gap-8">
            {categories.map((category, index) => (
              <Card key={index} className="hover:shadow-lg transition-shadow">
                <CardHeader>
                  <div className="flex items-center space-x-3">
                    <div className={`w-10 h-10 bg-gradient-to-r ${category.color} rounded-lg flex items-center justify-center`}>
                      <category.icon className="w-5 h-5 text-white" />
                    </div>
                    <div>
                      <CardTitle className="text-lg">{category.title}</CardTitle>
                      <CardDescription className="text-sm">
                        {category.description}
                      </CardDescription>
                    </div>
                  </div>
                </CardHeader>
                <CardContent>
                  <ul className="space-y-2">
                    {category.articles.map((article, articleIndex) => (
                      <li key={articleIndex} className="text-sm text-muted-foreground hover:text-foreground cursor-pointer">
                        {article}
                      </li>
                    ))}
                  </ul>
                  <Button variant="outline" size="sm" className="w-full mt-4">
                    View All Articles
                    <ArrowRight className="w-4 h-4 ml-2" />
                  </Button>
                </CardContent>
              </Card>
            ))}
          </div>
        </div>
      </section>

      {/* Popular Articles Section */}
      <section className="container mx-auto px-4 py-12 md:py-20 bg-muted/20">
        <div className="max-w-4xl mx-auto">
          <div className="text-center mb-12">
            <h2 className="text-2xl md:text-3xl font-bold mb-4">Popular Articles</h2>
            <p className="text-muted-foreground">
              Most viewed and highest-rated help articles.
            </p>
          </div>
          
          <div className="space-y-4">
            {popularArticles.map((article, index) => (
              <Card key={index} className="hover:shadow-lg transition-shadow cursor-pointer">
                <CardContent className="p-6">
                  <div className="flex items-center justify-between">
                    <div className="flex-1">
                      <h3 className="font-semibold mb-1">{article.title}</h3>
                      <div className="flex items-center space-x-4 text-sm text-muted-foreground">
                        <span>{article.category}</span>
                        <span>•</span>
                        <span>{article.views} views</span>
                        <span>•</span>
                        <div className="flex items-center">
                          <Star className="w-4 h-4 text-yellow-400 fill-current mr-1" />
                          <span>{article.rating}</span>
                        </div>
                      </div>
                    </div>
                    <ArrowRight className="w-5 h-5 text-muted-foreground" />
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>
        </div>
      </section>

      {/* FAQ Section */}
      <section className="container mx-auto px-4 py-12 md:py-20">
        <div className="max-w-4xl mx-auto">
          <div className="text-center mb-12">
            <h2 className="text-2xl md:text-3xl font-bold mb-4">Frequently Asked Questions</h2>
            <p className="text-muted-foreground">
              Quick answers to the most common questions about FlashCore.
            </p>
          </div>
          
          <div className="space-y-6">
            {faqs.map((faq, index) => (
              <Card key={index}>
                <CardContent className="p-6">
                  <h3 className="text-lg font-semibold mb-3">{faq.question}</h3>
                  <p className="text-sm text-muted-foreground">
                    {faq.answer}
                  </p>
                </CardContent>
              </Card>
            ))}
          </div>
        </div>
      </section>

      {/* Contact Support Section */}
      <section className="container mx-auto px-4 py-12 md:py-20 bg-muted/20">
        <div className="max-w-4xl mx-auto text-center">
          <h2 className="text-2xl md:text-3xl font-bold mb-4">Still Need Help?</h2>
          <p className="text-lg text-muted-foreground mb-8">
            Can't find what you're looking for? Our support team is here to help.
          </p>
          <div className="flex flex-col sm:flex-row gap-4 justify-center">
            <Link to="/contact">
              <Button size="lg" className="bg-gradient-to-r from-flashcore-purple to-flashcore-green text-white">
                Contact Support
                <ArrowRight className="w-4 h-4 ml-2" />
              </Button>
            </Link>
            <Link to="/docs">
              <Button size="lg" variant="outline">
                Browse Documentation
              </Button>
            </Link>
          </div>
        </div>
      </section>

      <Footer />
    </div>
  );
};

export default HelpCenterPage; 