import React, { useState } from "react";
import { useSubscriptionStatus } from "@/hooks/useSubscriptionStatus";
import { useAuth } from "@/hooks/useAuth";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { useToast } from "@/components/ui/use-toast";
import { CreditCard, Receipt, FileText, Send } from "lucide-react";
import FlashBillingModal from "@/components/flash/FlashBillingModal";
import GenerateReceiptModal from "@/components/flash/GenerateReceiptModal";
import GenerateTransactionModal from "@/components/flash/GenerateTransactionModal";

const FlashPage = () => {
  const { isActivated } = useSubscriptionStatus();
  const { user } = useAuth();
  const { toast } = useToast();
  
  const [showBillingModal, setShowBillingModal] = useState(false);
  const [showReceiptModal, setShowReceiptModal] = useState(false);
  const [showTransactionModal, setShowTransactionModal] = useState(false);
  
  const [customEmailData, setCustomEmailData] = useState({
    to: '',
    subject: '',
    message: '',
    template: 'custom'
  });

  if (!isActivated) {
    return (
      <div className="container mx-auto p-6">
        <div className="p-6 bg-destructive/10 border border-destructive rounded-lg text-center">
          <h2 className="text-2xl font-bold text-destructive mb-2">License Activation Required</h2>
          <p className="text-muted-foreground mb-4">
            You need an active license to access Flash features.
          </p>
          <Button asChild>
            <a href="/activation">Activate License</a>
          </Button>
        </div>
      </div>
    );
  }

  const handleSendCustomEmail = async () => {
    if (!customEmailData.to || !customEmailData.subject || !customEmailData.message) {
      toast({
        title: "Error",
        description: "Please fill in all fields",
        variant: "destructive"
      });
      return;
    }

    // TODO: Implement email sending functionality with PHP backend
    toast({
      title: "Feature Not Available",
      description: "Email sending functionality is not yet implemented in the PHP backend.",
      variant: "destructive"
    });

    // Reset form
    setCustomEmailData({ to: '', subject: '', message: '', template: 'custom' });
  };

  return (
    <div className="container mx-auto p-6 space-y-6">
      <div className="text-center mb-8">
        <h1 className="text-4xl font-bold mb-4 bg-gradient-to-r from-flashcore-purple to-flashcore-green bg-clip-text text-transparent">
          FlashCore Pro
        </h1>
        <p className="text-xl text-muted-foreground">
          Professional email automation and business tools
        </p>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
        {/* Billing Email Card */}
        <Card className="hover:shadow-lg transition-shadow">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <CreditCard className="w-5 h-5" />
              Billing Email
            </CardTitle>
            <CardDescription>
              Send professional billing emails to your clients
            </CardDescription>
          </CardHeader>
          <CardContent>
            <p className="text-sm text-muted-foreground mb-4">
              Create and send customizable billing invoices with payment links.
            </p>
            <Button 
              onClick={() => setShowBillingModal(true)}
              className="w-full bg-gradient-to-r from-blue-600 to-purple-600 hover:from-blue-700 hover:to-purple-700"
            >
              Create Billing Email
            </Button>
          </CardContent>
        </Card>

        {/* Receipt Generator Card */}
        <Card className="hover:shadow-lg transition-shadow">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Receipt className="w-5 h-5" />
              Receipt Generator
            </CardTitle>
            <CardDescription>
              Generate professional receipts for transactions
            </CardDescription>
          </CardHeader>
          <CardContent>
            <p className="text-sm text-muted-foreground mb-4">
              Create branded receipts with transaction details and company information.
            </p>
            <Button 
              onClick={() => setShowReceiptModal(true)}
              variant="outline"
              className="w-full border-purple-500 text-purple-500 hover:bg-purple-500/10"
            >
              Generate Receipt
            </Button>
          </CardContent>
        </Card>

        {/* Transaction Report Card */}
        <Card className="hover:shadow-lg transition-shadow">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <FileText className="w-5 h-5" />
              Transaction Report
            </CardTitle>
            <CardDescription>
              Generate detailed financial reports
            </CardDescription>
          </CardHeader>
          <CardContent>
            <p className="text-sm text-muted-foreground mb-4">
              Create comprehensive reports of your transactions and earnings.
            </p>
            <Button 
              onClick={() => setShowTransactionModal(true)}
              variant="outline"
              className="w-full border-green-500 text-green-500 hover:bg-green-500/10"
            >
              Generate Report
            </Button>
          </CardContent>
        </Card>
      </div>

      {/* Custom Email Section */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Send className="w-5 h-5" />
            Send Custom Email
          </CardTitle>
          <CardDescription>
            Send a personalized email using your FlashCore Pro account
          </CardDescription>
        </CardHeader>
        <CardContent className="space-y-4">
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div className="space-y-2">
              <Label htmlFor="to">To</Label>
              <Input
                id="to"
                placeholder="recipient@example.com"
                value={customEmailData.to}
                onChange={(e) => setCustomEmailData({...customEmailData, to: e.target.value})}
              />
            </div>
            <div className="space-y-2">
              <Label htmlFor="subject">Subject</Label>
              <Input
                id="subject"
                placeholder="Your email subject"
                value={customEmailData.subject}
                onChange={(e) => setCustomEmailData({...customEmailData, subject: e.target.value})}
              />
            </div>
          </div>
          
          <div className="space-y-2">
            <Label htmlFor="template">Template</Label>
            <Select 
              value={customEmailData.template} 
              onValueChange={(value) => setCustomEmailData({...customEmailData, template: value})}
            >
              <SelectTrigger>
                <SelectValue placeholder="Select template" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="custom">Custom</SelectItem>
                <SelectItem value="business">Business</SelectItem>
                <SelectItem value="newsletter">Newsletter</SelectItem>
                <SelectItem value="announcement">Announcement</SelectItem>
              </SelectContent>
            </Select>
          </div>
          
          <div className="space-y-2">
            <Label htmlFor="message">Message</Label>
            <Textarea
              id="message"
              placeholder="Write your message here..."
              rows={6}
              value={customEmailData.message}
              onChange={(e) => setCustomEmailData({...customEmailData, message: e.target.value})}
            />
          </div>
          
          <Button onClick={handleSendCustomEmail} className="w-full">
            <Send className="w-4 h-4 mr-2" />
            Send Email
          </Button>
        </CardContent>
      </Card>

      {/* Modals */}
      <FlashBillingModal 
        open={showBillingModal} 
        onOpenChange={setShowBillingModal} 
      />
      <GenerateReceiptModal 
        open={showReceiptModal} 
        onOpenChange={setShowReceiptModal} 
      />
      <GenerateTransactionModal 
        open={showTransactionModal} 
        onOpenChange={setShowTransactionModal} 
      />
    </div>
  );
};

export default FlashPage;