import React, { useState } from 'react';
import { Link } from 'react-router-dom';
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { BookOpen, Search, Play, FileText, Code, Users, ArrowRight, ExternalLink, CheckCircle, Clock, Star, Zap, Shield, Database } from "lucide-react";
import Navigation from "@/components/Navigation";
import Footer from "@/components/Footer";

const DocsPage = () => {
  const [searchTerm, setSearchTerm] = useState('');

  const categories = [
    {
      id: 'getting-started',
      title: 'Getting Started',
      description: 'Quick start guides and tutorials',
      icon: Play,
      color: 'from-blue-500 to-blue-600',
      articles: [
        { title: 'Quick Start Guide', time: '5 min read', difficulty: 'Beginner' },
        { title: 'Setting Up Your Account', time: '3 min read', difficulty: 'Beginner' },
        { title: 'Your First Email Campaign', time: '8 min read', difficulty: 'Beginner' },
        { title: 'Understanding the Dashboard', time: '6 min read', difficulty: 'Beginner' }
      ]
    },
    {
      id: 'templates',
      title: 'Email Templates',
      description: 'Create and customize email templates',
      icon: FileText,
      color: 'from-green-500 to-green-600',
      articles: [
        { title: 'Template Editor Guide', time: '10 min read', difficulty: 'Intermediate' },
        { title: 'Creating Custom Templates', time: '12 min read', difficulty: 'Intermediate' },
        { title: 'Template Variables', time: '8 min read', difficulty: 'Intermediate' },
        { title: 'Responsive Design Tips', time: '15 min read', difficulty: 'Advanced' }
      ]
    },
    {
      id: 'automation',
      title: 'Automation',
      description: 'Set up automated email workflows',
      icon: Zap,
      color: 'from-purple-500 to-purple-600',
      articles: [
        { title: 'Welcome Series Setup', time: '12 min read', difficulty: 'Intermediate' },
        { title: 'Abandoned Cart Recovery', time: '15 min read', difficulty: 'Intermediate' },
        { title: 'Advanced Workflow Rules', time: '20 min read', difficulty: 'Advanced' },
        { title: 'A/B Testing Automation', time: '18 min read', difficulty: 'Advanced' }
      ]
    },
    {
      id: 'integrations',
      title: 'Integrations',
      description: 'Connect with third-party tools',
      icon: Database,
      color: 'from-orange-500 to-orange-600',
      articles: [
        { title: 'Shopify Integration', time: '8 min read', difficulty: 'Intermediate' },
        { title: 'Salesforce Setup', time: '10 min read', difficulty: 'Intermediate' },
        { title: 'Zapier Automation', time: '12 min read', difficulty: 'Intermediate' },
        { title: 'Custom API Integration', time: '25 min read', difficulty: 'Advanced' }
      ]
    },
    {
      id: 'analytics',
      title: 'Analytics & Reporting',
      description: 'Track performance and insights',
      icon: Shield,
      color: 'from-red-500 to-red-600',
      articles: [
        { title: 'Understanding Email Metrics', time: '8 min read', difficulty: 'Beginner' },
        { title: 'Creating Custom Reports', time: '12 min read', difficulty: 'Intermediate' },
        { title: 'Advanced Analytics', time: '15 min read', difficulty: 'Advanced' },
        { title: 'Exporting Data', time: '6 min read', difficulty: 'Intermediate' }
      ]
    },
    {
      id: 'api',
      title: 'API Reference',
      description: 'Developer documentation and guides',
      icon: Code,
      color: 'from-pink-500 to-pink-600',
      articles: [
        { title: 'API Authentication', time: '10 min read', difficulty: 'Intermediate' },
        { title: 'Sending Emails via API', time: '12 min read', difficulty: 'Intermediate' },
        { title: 'Webhook Setup', time: '15 min read', difficulty: 'Advanced' },
        { title: 'SDK Documentation', time: '20 min read', difficulty: 'Advanced' }
      ]
    }
  ];

  const popularArticles = [
    {
      title: 'How to Increase Email Open Rates',
      category: 'Best Practices',
      views: '12.5k',
      time: '8 min read'
    },
    {
      title: 'Setting Up Email Authentication',
      category: 'Security',
      views: '8.2k',
      time: '10 min read'
    },
    {
      title: 'Creating Effective Subject Lines',
      category: 'Templates',
      views: '15.7k',
      time: '6 min read'
    },
    {
      title: 'GDPR Compliance Guide',
      category: 'Legal',
      views: '6.8k',
      time: '12 min read'
    }
  ];

  const tutorials = [
    {
      title: 'Build Your First Email Campaign',
      description: 'Step-by-step guide to creating and sending your first email campaign',
      duration: '15 minutes',
      level: 'Beginner',
      thumbnail: '🎯'
    },
    {
      title: 'Advanced Segmentation Strategies',
      description: 'Learn how to segment your audience for better engagement',
      duration: '25 minutes',
      level: 'Intermediate',
      thumbnail: '📊'
    },
    {
      title: 'Automation Workflow Mastery',
      description: 'Master the art of email automation and workflows',
      duration: '35 minutes',
      level: 'Advanced',
      thumbnail: '⚡'
    }
  ];

  const filteredCategories = categories.filter(category =>
    category.title.toLowerCase().includes(searchTerm.toLowerCase()) ||
    category.description.toLowerCase().includes(searchTerm.toLowerCase()) ||
    category.articles.some(article => 
      article.title.toLowerCase().includes(searchTerm.toLowerCase())
    )
  );

  return (
    <div className="min-h-screen bg-background">
      <Navigation />

      {/* Header */}
      <section className="container mx-auto px-4 py-12 md:py-20 text-center">
        <div className="max-w-4xl mx-auto">
          <Badge variant="secondary" className="mb-4">
            <BookOpen className="w-3 h-3 mr-1" />
            Documentation
          </Badge>
          <h1 className="text-3xl md:text-4xl lg:text-5xl font-bold mb-6 bg-gradient-to-r from-flashcore-purple to-flashcore-green bg-clip-text text-transparent">
            Documentation & Help
          </h1>
          <p className="text-lg md:text-xl text-muted-foreground mb-8 max-w-2xl mx-auto">
            Everything you need to know about FlashCore. From getting started to advanced features.
          </p>
          
          {/* Search */}
          <div className="max-w-md mx-auto relative">
            <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 w-4 h-4 text-muted-foreground" />
            <Input
              placeholder="Search documentation..."
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
              className="pl-10"
            />
          </div>
        </div>
      </section>

      {/* Quick Links */}
      <section className="container mx-auto px-4 py-8">
        <div className="max-w-6xl mx-auto">
          <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
            <Link to="/api">
              <Card className="hover:shadow-lg transition-shadow cursor-pointer">
                <CardContent className="p-4 text-center">
                  <Code className="w-8 h-8 mx-auto mb-2 text-flashcore-purple" />
                  <div className="text-sm font-medium">API Docs</div>
                </CardContent>
              </Card>
            </Link>
            <Link to="/templates">
              <Card className="hover:shadow-lg transition-shadow cursor-pointer">
                <CardContent className="p-4 text-center">
                  <FileText className="w-8 h-8 mx-auto mb-2 text-flashcore-purple" />
                  <div className="text-sm font-medium">Templates</div>
                </CardContent>
              </Card>
            </Link>
            <Link to="/integrations">
              <Card className="hover:shadow-lg transition-shadow cursor-pointer">
                <CardContent className="p-4 text-center">
                  <Database className="w-8 h-8 mx-auto mb-2 text-flashcore-purple" />
                  <div className="text-sm font-medium">Integrations</div>
                </CardContent>
              </Card>
            </Link>
            <Link to="/contact">
              <Card className="hover:shadow-lg transition-shadow cursor-pointer">
                <CardContent className="p-4 text-center">
                  <Users className="w-8 h-8 mx-auto mb-2 text-flashcore-purple" />
                  <div className="text-sm font-medium">Support</div>
                </CardContent>
              </Card>
            </Link>
          </div>
        </div>
      </section>

      {/* Popular Articles */}
      <section className="container mx-auto px-4 py-12 md:py-20">
        <div className="max-w-6xl mx-auto">
          <div className="text-center mb-12">
            <h2 className="text-2xl md:text-3xl font-bold mb-4">Popular Articles</h2>
            <p className="text-muted-foreground">
              Most viewed and helpful articles from our community.
            </p>
          </div>
          
          <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
            {popularArticles.map((article, index) => (
              <Card key={index} className="hover:shadow-lg transition-shadow cursor-pointer">
                <CardContent className="p-6">
                  <div className="flex items-start justify-between mb-3">
                    <Badge variant="outline" className="text-xs">
                      {article.category}
                    </Badge>
                    <div className="flex items-center text-xs text-muted-foreground">
                      <Clock className="w-3 h-3 mr-1" />
                      {article.time}
                    </div>
                  </div>
                  <h3 className="font-semibold mb-2">{article.title}</h3>
                  <div className="flex items-center text-xs text-muted-foreground">
                    <Users className="w-3 h-3 mr-1" />
                    {article.views} views
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>
        </div>
      </section>

      {/* Documentation Categories */}
      <section className="container mx-auto px-4 py-12 md:py-20 bg-muted/20">
        <div className="max-w-6xl mx-auto">
          <div className="text-center mb-12">
            <h2 className="text-2xl md:text-3xl font-bold mb-4">Documentation Categories</h2>
            <p className="text-muted-foreground">
              Browse our comprehensive documentation by category.
            </p>
          </div>
          
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
            {filteredCategories.map((category) => (
              <Card key={category.id} className="hover:shadow-lg transition-shadow">
                <CardHeader>
                  <div className="flex items-center mb-4">
                    <div className={`w-10 h-10 bg-gradient-to-r ${category.color} rounded-lg flex items-center justify-center mr-3`}>
                      <category.icon className="w-5 h-5 text-white" />
                    </div>
                    <div>
                      <CardTitle className="text-lg">{category.title}</CardTitle>
                      <CardDescription className="text-sm">
                        {category.description}
                      </CardDescription>
                    </div>
                  </div>
                </CardHeader>
                <CardContent>
                  <div className="space-y-3">
                    {category.articles.map((article, index) => (
                      <div key={index} className="flex items-center justify-between p-2 hover:bg-muted rounded cursor-pointer">
                        <div className="flex-1">
                          <div className="text-sm font-medium">{article.title}</div>
                          <div className="flex items-center text-xs text-muted-foreground">
                            <Clock className="w-3 h-3 mr-1" />
                            {article.time}
                            <Badge variant="secondary" className="ml-2 text-xs">
                              {article.difficulty}
                            </Badge>
                          </div>
                        </div>
                        <ArrowRight className="w-4 h-4 text-muted-foreground" />
                      </div>
                    ))}
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>
          
          {filteredCategories.length === 0 && (
            <div className="text-center py-12">
              <Search className="w-16 h-16 text-muted-foreground mx-auto mb-4" />
              <h3 className="text-xl font-semibold mb-2">No results found</h3>
              <p className="text-muted-foreground">
                Try adjusting your search terms or browse all categories.
              </p>
            </div>
          )}
        </div>
      </section>

      {/* Video Tutorials */}
      <section className="container mx-auto px-4 py-12 md:py-20">
        <div className="max-w-6xl mx-auto">
          <div className="text-center mb-12">
            <h2 className="text-2xl md:text-3xl font-bold mb-4">Video Tutorials</h2>
            <p className="text-muted-foreground">
              Learn FlashCore with our step-by-step video tutorials.
            </p>
          </div>
          
          <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
            {tutorials.map((tutorial, index) => (
              <Card key={index} className="hover:shadow-lg transition-shadow cursor-pointer">
                <CardHeader>
                  <div className="text-4xl mb-4">{tutorial.thumbnail}</div>
                  <CardTitle className="text-lg">{tutorial.title}</CardTitle>
                  <CardDescription className="text-sm">
                    {tutorial.description}
                  </CardDescription>
                </CardHeader>
                <CardContent>
                  <div className="flex items-center justify-between">
                    <div className="flex items-center text-sm text-muted-foreground">
                      <Clock className="w-4 h-4 mr-1" />
                      {tutorial.duration}
                    </div>
                    <Badge variant="outline" className="text-xs">
                      {tutorial.level}
                    </Badge>
                  </div>
                  <Button className="w-full mt-4 bg-gradient-to-r from-flashcore-purple to-flashcore-green">
                    <Play className="w-4 h-4 mr-2" />
                    Watch Tutorial
                  </Button>
                </CardContent>
              </Card>
            ))}
          </div>
        </div>
      </section>

      {/* Help Resources */}
      <section className="container mx-auto px-4 py-12 md:py-20 bg-muted/20">
        <div className="max-w-4xl mx-auto">
          <div className="text-center mb-12">
            <h2 className="text-2xl md:text-3xl font-bold mb-4">Need More Help?</h2>
            <p className="text-muted-foreground">
              Can't find what you're looking for? We're here to help.
            </p>
          </div>
          
          <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
            <Card className="hover:shadow-lg transition-shadow">
              <CardHeader>
                <div className="w-12 h-12 bg-gradient-to-r from-flashcore-purple to-flashcore-green rounded-lg flex items-center justify-center mb-4">
                  <Users className="w-6 h-6 text-white" />
                </div>
                <CardTitle>Community Forum</CardTitle>
                <CardDescription>
                  Connect with other users and share tips and tricks.
                </CardDescription>
              </CardHeader>
              <CardContent>
                <Button variant="outline" className="w-full">
                  <ExternalLink className="w-4 h-4 mr-2" />
                  Join Community
                </Button>
              </CardContent>
            </Card>
            
            <Card className="hover:shadow-lg transition-shadow">
              <CardHeader>
                <div className="w-12 h-12 bg-gradient-to-r from-flashcore-purple to-flashcore-green rounded-lg flex items-center justify-center mb-4">
                  <CheckCircle className="w-6 h-6 text-white" />
                </div>
                <CardTitle>Contact Support</CardTitle>
                <CardDescription>
                  Get help from our expert support team.
                </CardDescription>
              </CardHeader>
              <CardContent>
                <Link to="/contact">
                  <Button variant="outline" className="w-full">
                    <ExternalLink className="w-4 h-4 mr-2" />
                    Contact Us
                  </Button>
                </Link>
              </CardContent>
            </Card>
          </div>
        </div>
      </section>

      {/* CTA Section */}
      <section className="container mx-auto px-4 py-12 md:py-20">
        <div className="bg-gradient-to-r from-flashcore-purple to-flashcore-green rounded-2xl p-6 md:p-8 lg:p-12 text-center text-white">
          <h2 className="text-2xl md:text-3xl font-bold mb-4">Ready to Get Started?</h2>
          <p className="text-lg md:text-xl mb-6 md:mb-8 opacity-90">
            Start building amazing email campaigns with FlashCore today.
          </p>
          <div className="flex flex-col sm:flex-row gap-4 justify-center">
            <Link to="/register">
              <Button size="lg" variant="secondary" className="bg-white text-flashcore-purple hover:bg-gray-100">
                Start Free Trial
                <ArrowRight className="w-4 h-4 ml-2" />
              </Button>
            </Link>
            <Link to="/contact">
              <Button size="lg" variant="outline" className="border-white text-white hover:bg-white hover:text-flashcore-purple">
                Get Help
              </Button>
            </Link>
          </div>
        </div>
      </section>

      <Footer />
    </div>
  );
};

export default DocsPage; 