import React, { useState, useEffect } from "react";
import { useRealDashboardDataPhp } from "@/hooks/useRealDashboardDataPhp"; // Changed from useRealDashboardData
import { useDashboardLoggerPhp } from "@/hooks/useDashboardLoggerPhp"; // Changed from useDashboardLogger
import { useAuthPhp } from "@/hooks/useAuthPhp"; // Changed from useAuth
import { useSubscriptionStatusPhp } from "@/hooks/useSubscriptionStatusPhp"; // Changed from useSubscriptionStatus
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Progress } from "@/components/ui/progress";
import { 
  BarChart3, 
  Users, 
  Mail, 
  TrendingUp, 
  Target, 
  Plus,
  Eye,
  MousePointer,
  Download,
  MoreHorizontal,
  ArrowUpRight,
  ArrowDownRight,
  Play,
  Pause,
  Edit,
  Copy,
  Zap,
  Palette,
  Smartphone,
  Monitor,
  Tablet,
  RefreshCw,
  AlertCircle
} from "lucide-react";
import Layout from "@/components/Layout";
import EmailComposer from "@/components/EmailComposer";
import SuccessModal from "@/components/SuccessModal";
import MotivationPopup from "@/components/MotivationPopup";
import ReferralPopup from "@/components/ReferralPopup";
import TelegramPopup from "@/components/TelegramPopup";
import { useToast } from "@/hooks/use-toast";

const Dashboard: React.FC = () => {
  const { user, loading: authLoading } = useAuthPhp(); // Changed from useAuth
  const { data: realData, isLoading, error, refetch } = useRealDashboardDataPhp(); // Changed from useRealDashboardData
  const { isActivated } = useSubscriptionStatusPhp(); // Changed from useSubscriptionStatus
  const { logActivity } = useDashboardLoggerPhp(); // Changed from useDashboardLogger
  const { toast } = useToast();
  
  // UI State
  const [showEmailComposer, setShowEmailComposer] = useState(false);
  const [showSuccessModal, setShowSuccessModal] = useState(false);
  const [showMotivationPopup, setShowMotivationPopup] = useState(false);
  const [showReferralPopup, setShowReferralPopup] = useState(false);
  const [showTelegramPopup, setShowTelegramPopup] = useState(false);
  const [activeTab, setActiveTab] = useState("overview");
  const [isRefreshing, setIsRefreshing] = useState(false);

  // Enhanced mock data for comprehensive dashboard
  const mockDashboardData = {
    overview: {
      totalSubscribers: 12450,
      activeCampaigns: 8,
      totalEmailsSent: 45620,
      openRate: 24.8,
      clickRate: 3.2,
      conversionRate: 1.8,
      revenue: 28450,
      growth: 12.5
    },
    campaigns: [
      {
        id: 1,
        name: "Welcome Series",
        status: "active",
        type: "automation",
        sent: 5200,
        openRate: 28.5,
        clickRate: 4.2,
        revenue: 5200,
        lastSent: "2 hours ago"
      },
      {
        id: 2,
        name: "Product Launch Announcement",
        status: "draft",
        type: "campaign",
        sent: 0,
        openRate: 0,
        clickRate: 0,
        revenue: 0,
        lastSent: "Never"
      }
    ],
    subscribers: {
      total: 12450,
      segments: [
        { name: "Active Subscribers", count: 8920, percentage: 71.7 },
        { name: "New Subscribers", count: 1245, percentage: 10.0 },
        { name: "Engaged", count: 3112, percentage: 25.0 }
      ]
    }
  };

  // Use real data if available, otherwise fall back to mock data
  const dashboardData = realData ? {
    overview: {
      totalSubscribers: realData.user.referralCount * 100 + 1000,
      activeCampaigns: Math.max(1, Math.floor(realData.totalMails / 100)),
      totalEmailsSent: realData.totalMails,
      openRate: 24.8,
      clickRate: 3.2,
      conversionRate: 1.8,
      revenue: realData.user.referralEarnings * 10,
      growth: 12.5
    },
    campaigns: mockDashboardData.campaigns,
    subscribers: mockDashboardData.subscribers
  } : mockDashboardData;

  // Log dashboard access
  useEffect(() => {
    if (user) {
      logActivity('dashboard_accessed');
    }
  }, [user, logActivity]);

  // Enhanced popup management
  useEffect(() => {
    const lastReferralPopup = localStorage.getItem('lastReferralPopup');
    const lastTelegramPopup = localStorage.getItem('lastTelegramPopup');
    
    // Show success modal on first load
    if (!localStorage.getItem('dashboardFirstLoad')) {
      setShowSuccessModal(true);
      localStorage.setItem('dashboardFirstLoad', 'true');
    }
    
    // Show referral popup (every 24 hours)
    if (!lastReferralPopup || Date.now() - parseInt(lastReferralPopup) > 24 * 60 * 60 * 1000) {
      const timer = setTimeout(() => {
        setShowReferralPopup(true);
        localStorage.setItem('lastReferralPopup', Date.now().toString());
      }, 30000);
      
      return () => clearTimeout(timer);
    }
    
    // Show Telegram popup (every 48 hours)
    if (!lastTelegramPopup || Date.now() - parseInt(lastTelegramPopup) > 48 * 60 * 60 * 1000) {
      const timer = setTimeout(() => {
        setShowTelegramPopup(true);
        localStorage.setItem('lastTelegramPopup', Date.now().toString());
      }, 60000);
      
      return () => clearTimeout(timer);
    }
  }, []);

  const handleOpenEmailComposer = () => {
    if (!isActivated) {
      toast({
        title: "Account Activation Required",
        description: "Please activate your account to access email features.",
        variant: "destructive",
      });
      return;
    }
    setShowEmailComposer(true);
  };

  const handleCloseEmailComposer = () => {
    setShowEmailComposer(false);
  };

  const handleRefreshData = async () => {
    setIsRefreshing(true);
    try {
      await refetch();
      toast({
        title: "Data Refreshed",
        description: "Dashboard data has been updated successfully.",
      });
    } catch (error) {
      toast({
        title: "Refresh Failed",
        description: "Failed to refresh dashboard data. Please try again.",
        variant: "destructive",
      });
    } finally {
      setIsRefreshing(false);
    }
  };

  const handleExportData = () => {
    // Mock export functionality
    toast({
      title: "Export Started",
      description: "Your dashboard data is being prepared for download.",
    });
  };

  // Show loading while auth is loading or while data is being fetched
  if (authLoading || isLoading) {
    return (
      <Layout>
        <div className="flex items-center justify-center min-h-[calc(100vh-4rem)] bg-background">
          <div className="flex flex-col items-center">
            <div className="w-16 h-16 border-4 border-flashcore-purple border-t-transparent rounded-full animate-spin"></div>
            <p className="mt-4 text-xl font-semibold">Loading dashboard...</p>
            <p className="mt-2 text-sm text-muted-foreground">Preparing your data</p>
          </div>
        </div>
      </Layout>
    );
  }

  // Enhanced error state with retry functionality
  if (error) {
    console.error('Dashboard error:', error);
    return (
      <Layout>
        <div className="flex items-center justify-center min-h-[calc(100vh-4rem)] bg-background">
          <div className="p-6 bg-card rounded-xl border border-destructive max-w-md mx-auto text-center">
            <AlertCircle className="w-16 h-16 text-destructive mx-auto mb-4" />
            <h2 className="text-2xl font-bold text-destructive mb-2">Error Loading Dashboard</h2>
            <p className="text-muted-foreground mb-4">
              {(error as any).message || 'Please try refreshing the page'}
            </p>
            <div className="flex gap-2 justify-center">
              <Button 
                onClick={handleRefreshData}
                disabled={isRefreshing}
                className="bg-destructive text-destructive-foreground hover:bg-destructive/90"
              >
                {isRefreshing ? (
                  <>
                    <RefreshCw className="w-4 h-4 mr-2 animate-spin" />
                    Retrying...
                  </>
                ) : (
                  <>
                    <RefreshCw className="w-4 h-4 mr-2" />
                    Retry
                  </>
                )}
              </Button>
              <Button 
                variant="outline"
                onClick={() => window.location.reload()}
              >
                Refresh Page
              </Button>
            </div>
          </div>
        </div>
      </Layout>
    );
  }

  return (
    <Layout>
      <div className="space-y-6">
        {/* Dashboard Header with Actions */}
        <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
          <div>
            <h1 className="text-3xl font-bold">Dashboard</h1>
            <p className="text-muted-foreground">
              Welcome back, {realData?.user.username || 'User'}! 
              {realData && ` You've sent ${realData.totalMails} emails.`}
            </p>
          </div>
          <div className="flex items-center gap-2">
            <Button
              variant="outline"
              onClick={handleRefreshData}
              disabled={isRefreshing}
            >
              <RefreshCw className={`w-4 h-4 mr-2 ${isRefreshing ? 'animate-spin' : ''}`} />
              Refresh
            </Button>
            <Button variant="outline" onClick={handleExportData}>
              <Download className="w-4 h-4 mr-2" />
              Export
            </Button>
          </div>
        </div>
        
        {/* Quick Stats Overview */}
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
          <Card className="hover:shadow-lg transition-shadow">
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-muted-foreground">Total Subscribers</p>
                  <p className="text-2xl font-bold">{dashboardData.overview.totalSubscribers.toLocaleString()}</p>
                  <div className="flex items-center mt-2">
                    <TrendingUp className="w-4 h-4 text-green-500 mr-1" />
                    <span className="text-sm text-green-500">+{dashboardData.overview.growth}%</span>
                  </div>
                </div>
                <Users className="w-8 h-8 text-flashcore-purple" />
              </div>
            </CardContent>
          </Card>

          <Card className="hover:shadow-lg transition-shadow">
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-muted-foreground">Active Campaigns</p>
                  <p className="text-2xl font-bold">{dashboardData.overview.activeCampaigns}</p>
                  <p className="text-sm text-muted-foreground mt-2">Running now</p>
                </div>
                <Target className="w-8 h-8 text-flashcore-green" />
              </div>
            </CardContent>
          </Card>

          <Card className="hover:shadow-lg transition-shadow">
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-muted-foreground">Open Rate</p>
                  <p className="text-2xl font-bold">{dashboardData.overview.openRate}%</p>
                  <div className="flex items-center mt-2">
                    <Eye className="w-4 h-4 text-blue-500 mr-1" />
                    <span className="text-sm text-blue-500">+2.1%</span>
                  </div>
                </div>
                <Eye className="w-8 h-8 text-blue-500" />
              </div>
            </CardContent>
          </Card>

          <Card className="hover:shadow-lg transition-shadow">
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-muted-foreground">Revenue</p>
                  <p className="text-2xl font-bold">${dashboardData.overview.revenue.toLocaleString()}</p>
                  <div className="flex items-center mt-2">
                    <TrendingUp className="w-4 h-4 text-green-500 mr-1" />
                    <span className="text-sm text-green-500">+{dashboardData.overview.growth}%</span>
                  </div>
                </div>
                <BarChart3 className="w-8 h-8 text-flashcore-orange" />
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Main Dashboard Tabs */}
        <Tabs value={activeTab} onValueChange={setActiveTab} className="space-y-6">
          <TabsList className="grid w-full grid-cols-2 sm:grid-cols-3 lg:grid-cols-5">
            <TabsTrigger value="overview">Overview</TabsTrigger>
            <TabsTrigger value="campaigns">Campaigns</TabsTrigger>
            <TabsTrigger value="subscribers">Subscribers</TabsTrigger>
            <TabsTrigger value="automation">Automation</TabsTrigger>
            <TabsTrigger value="analytics">Analytics</TabsTrigger>
          </TabsList>

          {/* Overview Tab */}
          <TabsContent value="overview" className="space-y-6">
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
              {/* Recent Performance */}
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center justify-between">
                    Recent Performance
                    <Button variant="outline" size="sm" onClick={handleExportData}>
                      <Download className="w-4 h-4 mr-2" />
                      Export
                    </Button>
                  </CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="space-y-4">
                    <div className="flex items-center justify-between p-3 bg-muted/50 rounded-lg">
                      <div className="flex items-center space-x-4">
                        <div className="text-sm font-medium">Today</div>
                        <div className="text-sm text-muted-foreground">1,350 sent</div>
                      </div>
                      <div className="flex items-center space-x-4 text-sm">
                        <span className="text-blue-500">378 opened</span>
                        <span className="text-green-500">52 clicked</span>
                        <span className="text-flashcore-purple">$1,350</span>
                      </div>
                    </div>
                    <div className="flex items-center justify-between p-3 bg-muted/50 rounded-lg">
                      <div className="flex items-center space-x-4">
                        <div className="text-sm font-medium">Yesterday</div>
                        <div className="text-sm text-muted-foreground">1,200 sent</div>
                      </div>
                      <div className="flex items-center space-x-4 text-sm">
                        <span className="text-blue-500">312 opened</span>
                        <span className="text-green-500">45 clicked</span>
                        <span className="text-flashcore-purple">$1,200</span>
                      </div>
                    </div>
                  </div>
                </CardContent>
              </Card>

              {/* Quick Actions */}
              <Card>
                <CardHeader>
                  <CardTitle>Quick Actions</CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="grid grid-cols-2 gap-4">
                    <Button 
                      onClick={handleOpenEmailComposer} 
                      className="h-20 flex flex-col items-center justify-center"
                      disabled={!isActivated}
                    >
                      <Plus className="w-6 h-6 mb-2" />
                      <span>New Campaign</span>
                    </Button>
                    <Button variant="outline" className="h-20 flex flex-col items-center justify-center">
                      <Users className="w-6 h-6 mb-2" />
                      <span>Import Subscribers</span>
                    </Button>
                    <Button variant="outline" className="h-20 flex flex-col items-center justify-center">
                      <Zap className="w-6 h-6 mb-2" />
                      <span>Create Automation</span>
                    </Button>
                    <Button variant="outline" className="h-20 flex flex-col items-center justify-center">
                      <Palette className="w-6 h-6 mb-2" />
                      <span>Design Template</span>
                    </Button>
                  </div>
                </CardContent>
              </Card>
            </div>
          </TabsContent>

          {/* Campaigns Tab */}
          <TabsContent value="campaigns" className="space-y-6">
            <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
              <h2 className="text-2xl font-bold">Campaigns</h2>
              <Button onClick={handleOpenEmailComposer} disabled={!isActivated}>
                <Plus className="w-4 h-4 mr-2" />
                New Campaign
              </Button>
            </div>

            <div className="grid grid-cols-1 gap-4">
              {dashboardData.campaigns.map((campaign) => (
                <Card key={campaign.id} className="hover:shadow-md transition-shadow">
                  <CardContent className="p-6">
                    <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
                      <div className="flex-1">
                        <div className="flex items-center gap-3 mb-2">
                          <h3 className="text-lg font-semibold">{campaign.name}</h3>
                          <Badge variant={campaign.status === 'active' ? 'default' : 'secondary'}>
                            {campaign.status}
                          </Badge>
                          <Badge variant="outline">{campaign.type}</Badge>
                        </div>
                        <div className="grid grid-cols-2 sm:grid-cols-4 gap-4 text-sm">
                          <div>
                            <p className="text-muted-foreground">Sent</p>
                            <p className="font-semibold">{campaign.sent.toLocaleString()}</p>
                          </div>
                          <div>
                            <p className="text-muted-foreground">Open Rate</p>
                            <p className="font-semibold">{campaign.openRate}%</p>
                          </div>
                          <div>
                            <p className="text-muted-foreground">Click Rate</p>
                            <p className="font-semibold">{campaign.clickRate}%</p>
                          </div>
                          <div>
                            <p className="text-muted-foreground">Revenue</p>
                            <p className="font-semibold">${campaign.revenue.toLocaleString()}</p>
                          </div>
                        </div>
                      </div>
                      <div className="flex items-center gap-2">
                        <Button variant="outline" size="sm">
                          <Edit className="w-4 h-4 mr-2" />
                          Edit
                        </Button>
                        <Button variant="outline" size="sm">
                          <MoreHorizontal className="w-4 h-4" />
                        </Button>
                      </div>
                    </div>
                  </CardContent>
                </Card>
              ))}
            </div>
          </TabsContent>

          {/* Subscribers Tab */}
          <TabsContent value="subscribers" className="space-y-6">
            <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
              <h2 className="text-2xl font-bold">Subscribers</h2>
              <Button variant="outline">
                <Users className="w-4 h-4 mr-2" />
                Import Subscribers
              </Button>
            </div>

            <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
              {dashboardData.subscribers.segments.map((segment, index) => (
                <Card key={index}>
                  <CardHeader>
                    <CardTitle className="text-lg">{segment.name}</CardTitle>
                  </CardHeader>
                  <CardContent>
                    <div className="text-center">
                      <p className="text-3xl font-bold">{segment.count.toLocaleString()}</p>
                      <p className="text-sm text-muted-foreground">{segment.percentage}% of total</p>
                      <Progress value={segment.percentage} className="mt-4" />
                    </div>
                  </CardContent>
                </Card>
              ))}
            </div>
          </TabsContent>

          {/* Automation Tab */}
          <TabsContent value="automation" className="space-y-6">
            <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
              <h2 className="text-2xl font-bold">Automation</h2>
              <Button variant="outline">
                <Zap className="w-4 h-4 mr-2" />
                Create Workflow
              </Button>
            </div>

            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              <Card>
                <CardHeader>
                  <CardTitle>Welcome Series</CardTitle>
                  <CardDescription>Automated onboarding emails</CardDescription>
                </CardHeader>
                <CardContent>
                  <div className="space-y-4">
                    <div className="flex items-center justify-between">
                      <span className="text-sm">Status</span>
                      <Badge variant="default">Active</Badge>
                    </div>
                    <div className="flex items-center justify-between">
                      <span className="text-sm">Subscribers</span>
                      <span className="font-semibold">1,245</span>
                    </div>
                    <div className="flex items-center justify-between">
                      <span className="text-sm">Emails Sent</span>
                      <span className="font-semibold">3,735</span>
                    </div>
                  </div>
                </CardContent>
              </Card>

              <Card>
                <CardHeader>
                  <CardTitle>Abandoned Cart</CardTitle>
                  <CardDescription>Recovery automation</CardDescription>
                </CardHeader>
                <CardContent>
                  <div className="space-y-4">
                    <div className="flex items-center justify-between">
                      <span className="text-sm">Status</span>
                      <Badge variant="secondary">Draft</Badge>
                    </div>
                    <div className="flex items-center justify-between">
                      <span className="text-sm">Subscribers</span>
                      <span className="font-semibold">0</span>
                    </div>
                    <div className="flex items-center justify-between">
                      <span className="text-sm">Emails Sent</span>
                      <span className="font-semibold">0</span>
                    </div>
                  </div>
                </CardContent>
              </Card>
            </div>
          </TabsContent>

          {/* Analytics Tab */}
          <TabsContent value="analytics" className="space-y-6">
            <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
              <h2 className="text-2xl font-bold">Analytics</h2>
              <div className="flex items-center space-x-2">
                <Button variant="outline" onClick={handleExportData}>
                  <Download className="w-4 h-4 mr-2" />
                  Export
                </Button>
              </div>
            </div>

            <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
              {/* Performance Metrics */}
              <Card>
                <CardHeader>
                  <CardTitle>Performance Metrics</CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="space-y-6">
                    <div className="flex items-center justify-between">
                      <div>
                        <p className="text-sm text-muted-foreground">Open Rate</p>
                        <p className="text-2xl font-bold">{dashboardData.overview.openRate}%</p>
                      </div>
                      <div className="text-right">
                        <div className="flex items-center text-green-500">
                          <ArrowUpRight className="w-4 h-4 mr-1" />
                          <span className="text-sm">+2.1%</span>
                        </div>
                      </div>
                    </div>
                    
                    <div className="flex items-center justify-between">
                      <div>
                        <p className="text-sm text-muted-foreground">Click Rate</p>
                        <p className="text-2xl font-bold">{dashboardData.overview.clickRate}%</p>
                      </div>
                      <div className="text-right">
                        <div className="flex items-center text-green-500">
                          <ArrowUpRight className="w-4 h-4 mr-1" />
                          <span className="text-sm">+0.8%</span>
                        </div>
                      </div>
                    </div>
                  </div>
                </CardContent>
              </Card>

              {/* Device Analytics */}
              <Card>
                <CardHeader>
                  <CardTitle>Device Analytics</CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="space-y-4">
                    <div className="flex items-center justify-between">
                      <div className="flex items-center space-x-2">
                        <Smartphone className="w-4 h-4 text-blue-500" />
                        <span>Mobile</span>
                      </div>
                      <div className="flex items-center space-x-2">
                        <span className="font-semibold">68%</span>
                        <Progress value={68} className="w-20" />
                      </div>
                    </div>
                    
                    <div className="flex items-center justify-between">
                      <div className="flex items-center space-x-2">
                        <Monitor className="w-4 h-4 text-green-500" />
                        <span>Desktop</span>
                      </div>
                      <div className="flex items-center space-x-2">
                        <span className="font-semibold">28%</span>
                        <Progress value={28} className="w-20" />
                      </div>
                    </div>
                  </div>
                </CardContent>
              </Card>
            </div>
          </TabsContent>
        </Tabs>

        {/* Floating Email Button */}
        <button
          onClick={handleOpenEmailComposer}
          className="fixed bottom-6 right-6 bg-gradient-to-r from-flashcore-purple to-flashcore-green text-white p-4 rounded-full shadow-lg hover:shadow-xl transition-all duration-300 z-50 group"
          title="Compose Email"
        >
          <Mail className="w-6 h-6 group-hover:scale-110 transition-transform" />
        </button>
      </div>

      {/* Modals */}
      <EmailComposer 
        isOpen={showEmailComposer} 
        onClose={handleCloseEmailComposer} 
      />
      
      <SuccessModal 
        isOpen={showSuccessModal} 
        onClose={() => setShowSuccessModal(false)} 
      />
      
      <MotivationPopup 
        isOpen={showMotivationPopup} 
        onClose={() => setShowMotivationPopup(false)} 
      />
      
      <ReferralPopup 
        isOpen={showReferralPopup} 
        onClose={() => setShowReferralPopup(false)} 
      />
      
      <TelegramPopup 
        isOpen={showTelegramPopup} 
        onClose={() => setShowTelegramPopup(false)} 
      />
    </Layout>
  );
};

export default Dashboard;