import React, { useState } from 'react';
import { Link } from 'react-router-dom';
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import { 
  Search,
  MapPin,
  Clock,
  Users,
  Heart,
  Zap,
  Globe,
  Award,
  Coffee,
  Home,
  DollarSign,
  BookOpen,
  Shield,
  TrendingUp,
  Mail,
  ExternalLink,
  Filter,
  Briefcase,
  Building,
  Code,
  Palette,
  BarChart3,
  MessageSquare,
  Target,
  Star,
  ArrowRight,
  CheckCircle
} from "lucide-react";
import Navigation from "@/components/Navigation";
import Footer from "@/components/Footer";

const CareersPage = () => {
  const [searchQuery, setSearchQuery] = useState('');
  const [selectedDepartment, setSelectedDepartment] = useState('all');
  const [selectedLocation, setSelectedLocation] = useState('all');

  const departments = [
    { id: 'all', name: 'All Departments', icon: Briefcase },
    { id: 'engineering', name: 'Engineering', icon: Code },
    { id: 'design', name: 'Design', icon: Palette },
    { id: 'marketing', name: 'Marketing', icon: Target },
    { id: 'sales', name: 'Sales', icon: TrendingUp },
    { id: 'product', name: 'Product', icon: BarChart3 },
    { id: 'support', name: 'Support', icon: MessageSquare }
  ];

  const locations = [
    { id: 'all', name: 'All Locations', icon: Globe },
    { id: 'remote', name: 'Remote', icon: Home },
    { id: 'san-francisco', name: 'San Francisco', icon: MapPin },
    { id: 'new-york', name: 'New York', icon: MapPin },
    { id: 'london', name: 'London', icon: MapPin },
    { id: 'berlin', name: 'Berlin', icon: MapPin }
  ];

  const jobOpenings = [
    {
      id: 1,
      title: "Senior Frontend Engineer",
      department: "engineering",
      location: "remote",
      type: "Full-time",
      experience: "3-5 years",
      description: "Join our engineering team to build the next generation of email marketing tools. You'll work with React, TypeScript, and modern web technologies.",
      requirements: [
        "Strong experience with React, TypeScript, and modern JavaScript",
        "Experience with state management (Redux, Zustand, etc.)",
        "Knowledge of CSS frameworks (Tailwind CSS, styled-components)",
        "Experience with testing frameworks (Jest, React Testing Library)",
        "Understanding of web performance and optimization"
      ],
      benefits: ["Competitive salary", "Remote work", "Health insurance", "401k matching"],
      salary: "$120,000 - $160,000",
      postedDate: "2024-03-20"
    },
    {
      id: 2,
      title: "Product Designer",
      department: "design",
      location: "san-francisco",
      type: "Full-time",
      experience: "2-4 years",
      description: "Help shape the user experience of our email marketing platform. You'll work closely with product managers and engineers to create intuitive, beautiful interfaces.",
      requirements: [
        "Portfolio demonstrating strong UI/UX design skills",
        "Experience with design tools (Figma, Sketch, Adobe Creative Suite)",
        "Understanding of user-centered design principles",
        "Experience with design systems and component libraries",
        "Ability to work in a fast-paced, collaborative environment"
      ],
      benefits: ["Competitive salary", "Flexible work arrangements", "Professional development", "Stock options"],
      salary: "$90,000 - $130,000",
      postedDate: "2024-03-18"
    },
    {
      id: 3,
      title: "Marketing Manager",
      department: "marketing",
      location: "new-york",
      type: "Full-time",
      experience: "4-6 years",
      description: "Lead our marketing efforts to grow FlashCore's presence in the email marketing space. You'll develop and execute marketing strategies across multiple channels.",
      requirements: [
        "Experience in B2B SaaS marketing",
        "Strong analytical skills and data-driven approach",
        "Experience with marketing automation tools",
        "Excellent written and verbal communication skills",
        "Experience managing marketing campaigns and budgets"
      ],
      benefits: ["Competitive salary", "Performance bonuses", "Health benefits", "Professional development"],
      salary: "$100,000 - $140,000",
      postedDate: "2024-03-15"
    },
    {
      id: 4,
      title: "Customer Success Manager",
      department: "support",
      location: "remote",
      type: "Full-time",
      experience: "2-3 years",
      description: "Help our customers succeed with FlashCore by providing exceptional support and guidance. You'll be the bridge between our customers and our product team.",
      requirements: [
        "Experience in customer success or account management",
        "Strong problem-solving and communication skills",
        "Experience with CRM systems (Salesforce, HubSpot)",
        "Understanding of email marketing concepts",
        "Ability to manage multiple customer relationships"
      ],
      benefits: ["Competitive salary", "Remote work", "Health insurance", "Flexible PTO"],
      salary: "$70,000 - $90,000",
      postedDate: "2024-03-12"
    },
    {
      id: 5,
      title: "Sales Development Representative",
      department: "sales",
      location: "london",
      type: "Full-time",
      experience: "1-2 years",
      description: "Join our sales team to help grow FlashCore's customer base. You'll be responsible for prospecting, qualifying leads, and setting up sales meetings.",
      requirements: [
        "Strong communication and interpersonal skills",
        "Experience in B2B sales or lead generation",
        "Familiarity with CRM systems and sales tools",
        "Goal-oriented and results-driven mindset",
        "Understanding of the SaaS industry"
      ],
      benefits: ["Competitive base salary", "Uncapped commission", "Health benefits", "Career growth opportunities"],
      salary: "£35,000 - £45,000 + commission",
      postedDate: "2024-03-10"
    },
    {
      id: 6,
      title: "Product Manager",
      department: "product",
      location: "berlin",
      type: "Full-time",
      experience: "3-5 years",
      description: "Drive product strategy and execution for FlashCore's email marketing platform. You'll work with cross-functional teams to deliver exceptional user experiences.",
      requirements: [
        "Experience in product management for SaaS products",
        "Strong analytical and strategic thinking skills",
        "Experience with product analytics and user research",
        "Excellent communication and collaboration skills",
        "Understanding of email marketing and automation"
      ],
      benefits: ["Competitive salary", "Stock options", "Health insurance", "Professional development"],
      salary: "€80,000 - €110,000",
      postedDate: "2024-03-08"
    }
  ];

  const filteredJobs = jobOpenings.filter(job => {
    const matchesSearch = job.title.toLowerCase().includes(searchQuery.toLowerCase()) ||
                         job.description.toLowerCase().includes(searchQuery.toLowerCase());
    const matchesDepartment = selectedDepartment === 'all' || job.department === selectedDepartment;
    const matchesLocation = selectedLocation === 'all' || job.location === selectedLocation;
    return matchesSearch && matchesDepartment && matchesLocation;
  });

  const companyValues = [
    {
      icon: Heart,
      title: "Customer First",
      description: "Everything we do is centered around helping our customers succeed."
    },
    {
      icon: Zap,
      title: "Innovation",
      description: "We constantly push boundaries to deliver cutting-edge email marketing solutions."
    },
    {
      icon: Users,
      title: "Collaboration",
      description: "We believe the best results come from working together as a team."
    },
    {
      icon: Shield,
      title: "Integrity",
      description: "We operate with honesty, transparency, and ethical business practices."
    }
  ];

  const benefits = [
    {
      icon: Home,
      title: "Remote Work",
      description: "Work from anywhere in the world with our flexible remote policy."
    },
    {
      icon: DollarSign,
      title: "Competitive Pay",
      description: "We offer competitive salaries and comprehensive benefits packages."
    },
    {
      icon: BookOpen,
      title: "Learning & Development",
      description: "Continuous learning opportunities and professional development support."
    },
    {
      icon: Coffee,
      title: "Flexible Hours",
      description: "Work when you're most productive with flexible scheduling options."
    },
    {
      icon: Award,
      title: "Stock Options",
      description: "Own a piece of FlashCore with our employee stock option program."
    },
    {
      icon: Globe,
      title: "Global Team",
      description: "Join a diverse, international team with colleagues from around the world."
    }
  ];

  return (
    <div className="min-h-screen bg-background">
      <Navigation />

      {/* Hero Section */}
      <section className="container mx-auto px-4 py-12 md:py-20">
        <div className="max-w-4xl mx-auto text-center">
          <Badge variant="secondary" className="mb-4">
            <Briefcase className="w-3 h-3 mr-1" />
            Join Our Team
          </Badge>
          <h1 className="text-3xl md:text-4xl lg:text-5xl font-bold mb-6 bg-gradient-to-r from-flashcore-purple to-flashcore-green bg-clip-text text-transparent">
            Build the Future of Email Marketing
          </h1>
          <p className="text-lg md:text-xl text-muted-foreground mb-8 max-w-2xl mx-auto">
            Join our mission to revolutionize email marketing with cutting-edge technology, 
            innovative design, and exceptional user experiences.
          </p>
          
          {/* Search Bar */}
          <div className="max-w-2xl mx-auto mb-8">
            <div className="relative">
              <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-muted-foreground w-5 h-5" />
              <Input
                type="text"
                placeholder="Search jobs by title, skills, or keywords..."
                value={searchQuery}
                onChange={(e) => setSearchQuery(e.target.value)}
                className="pl-10 pr-4 py-3 text-lg"
              />
            </div>
          </div>
        </div>
      </section>

      {/* Filters */}
      <section className="container mx-auto px-4 py-8">
        <div className="max-w-6xl mx-auto">
          <div className="flex flex-wrap gap-4 justify-center mb-8">
            <div className="flex flex-wrap gap-2">
              <span className="text-sm font-medium text-muted-foreground flex items-center">
                <Filter className="w-4 h-4 mr-1" />
                Department:
              </span>
              {departments.map((dept) => (
                <Button
                  key={dept.id}
                  variant={selectedDepartment === dept.id ? "default" : "outline"}
                  size="sm"
                  onClick={() => setSelectedDepartment(dept.id)}
                  className="flex items-center space-x-2"
                >
                  <dept.icon className="w-4 h-4" />
                  <span>{dept.name}</span>
                </Button>
              ))}
            </div>
          </div>
          
          <div className="flex flex-wrap gap-4 justify-center">
            <div className="flex flex-wrap gap-2">
              <span className="text-sm font-medium text-muted-foreground flex items-center">
                <MapPin className="w-4 h-4 mr-1" />
                Location:
              </span>
              {locations.map((location) => (
                <Button
                  key={location.id}
                  variant={selectedLocation === location.id ? "default" : "outline"}
                  size="sm"
                  onClick={() => setSelectedLocation(location.id)}
                  className="flex items-center space-x-2"
                >
                  <location.icon className="w-4 h-4" />
                  <span>{location.name}</span>
                </Button>
              ))}
            </div>
          </div>
        </div>
      </section>

      {/* Job Listings */}
      <section className="container mx-auto px-4 py-12 md:py-20">
        <div className="max-w-6xl mx-auto">
          <div className="text-center mb-12">
            <h2 className="text-2xl md:text-3xl font-bold mb-4">Open Positions</h2>
            <p className="text-muted-foreground">
              {filteredJobs.length} position{filteredJobs.length !== 1 ? 's' : ''} available
            </p>
          </div>
          
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-6 md:gap-8">
            {filteredJobs.map((job) => (
              <Card key={job.id} className="hover:shadow-lg transition-shadow group">
                <CardHeader>
                  <div className="flex items-center justify-between mb-4">
                    <Badge variant="secondary" className="text-xs">
                      {departments.find(d => d.id === job.department)?.name}
                    </Badge>
                    <Badge variant="outline" className="text-xs">
                      {job.type}
                    </Badge>
                  </div>
                  <CardTitle className="text-xl group-hover:text-flashcore-purple transition-colors">
                    {job.title}
                  </CardTitle>
                  <CardDescription className="text-sm">
                    {job.description}
                  </CardDescription>
                </CardHeader>
                <CardContent>
                  <div className="space-y-4">
                    {/* Job Details */}
                    <div className="flex items-center justify-between text-sm">
                      <div className="flex items-center space-x-4">
                        <span className="flex items-center">
                          <MapPin className="w-4 h-4 mr-1" />
                          {locations.find(l => l.id === job.location)?.name}
                        </span>
                        <span className="flex items-center">
                          <Clock className="w-4 h-4 mr-1" />
                          {job.experience}
                        </span>
                      </div>
                      <span className="font-semibold text-flashcore-purple">
                        {job.salary}
                      </span>
                    </div>

                    {/* Requirements Preview */}
                    <div>
                      <h4 className="font-semibold text-sm mb-2">Key Requirements:</h4>
                      <ul className="text-sm text-muted-foreground space-y-1">
                        {job.requirements.slice(0, 3).map((req, index) => (
                          <li key={index} className="flex items-start">
                            <CheckCircle className="w-3 h-3 mr-2 mt-0.5 text-green-500 flex-shrink-0" />
                            {req}
                          </li>
                        ))}
                        {job.requirements.length > 3 && (
                          <li className="text-flashcore-purple text-xs">
                            +{job.requirements.length - 3} more requirements
                          </li>
                        )}
                      </ul>
                    </div>

                    {/* Benefits */}
                    <div className="flex flex-wrap gap-1">
                      {job.benefits.slice(0, 3).map((benefit, index) => (
                        <Badge key={index} variant="outline" className="text-xs">
                          {benefit}
                        </Badge>
                      ))}
                    </div>

                    {/* Apply Button */}
                    <Button className="w-full bg-flashcore-purple hover:bg-flashcore-purple/90">
                      Apply Now
                      <ArrowRight className="w-4 h-4 ml-2" />
                    </Button>
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>

          {filteredJobs.length === 0 && (
            <div className="text-center py-12">
              <Briefcase className="w-16 h-16 text-muted-foreground mx-auto mb-4" />
              <h3 className="text-xl font-semibold mb-2">No positions found</h3>
              <p className="text-muted-foreground mb-4">
                Try adjusting your search criteria or check back later for new opportunities.
              </p>
              <Button 
                variant="outline" 
                onClick={() => {
                  setSearchQuery('');
                  setSelectedDepartment('all');
                  setSelectedLocation('all');
                }}
              >
                Clear Filters
              </Button>
            </div>
          )}
        </div>
      </section>

      {/* Company Values */}
      <section className="container mx-auto px-4 py-12 md:py-20 bg-muted/20">
        <div className="max-w-6xl mx-auto">
          <div className="text-center mb-12">
            <h2 className="text-2xl md:text-3xl font-bold mb-4">Our Values</h2>
            <p className="text-muted-foreground">
              The principles that guide everything we do at FlashCore
            </p>
          </div>
          
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 md:gap-8">
            {companyValues.map((value, index) => (
              <Card key={index} className="text-center hover:shadow-lg transition-shadow">
                <CardContent className="p-6">
                  <div className="w-12 h-12 bg-gradient-to-r from-flashcore-purple to-flashcore-green rounded-lg flex items-center justify-center mx-auto mb-4">
                    <value.icon className="w-6 h-6 text-white" />
                  </div>
                  <h3 className="text-lg font-semibold mb-2">{value.title}</h3>
                  <p className="text-sm text-muted-foreground">{value.description}</p>
                </CardContent>
              </Card>
            ))}
          </div>
        </div>
      </section>

      {/* Benefits */}
      <section className="container mx-auto px-4 py-12 md:py-20">
        <div className="max-w-6xl mx-auto">
          <div className="text-center mb-12">
            <h2 className="text-2xl md:text-3xl font-bold mb-4">Why Work at FlashCore?</h2>
            <p className="text-muted-foreground">
              We offer comprehensive benefits and a supportive work environment
            </p>
          </div>
          
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6 md:gap-8">
            {benefits.map((benefit, index) => (
              <Card key={index} className="hover:shadow-lg transition-shadow">
                <CardContent className="p-6">
                  <div className="w-12 h-12 bg-gradient-to-r from-flashcore-purple to-flashcore-green rounded-lg flex items-center justify-center mb-4">
                    <benefit.icon className="w-6 h-6 text-white" />
                  </div>
                  <h3 className="text-lg font-semibold mb-2">{benefit.title}</h3>
                  <p className="text-sm text-muted-foreground">{benefit.description}</p>
                </CardContent>
              </Card>
            ))}
          </div>
        </div>
      </section>

      {/* Culture Section */}
      <section className="container mx-auto px-4 py-12 md:py-20 bg-muted/20">
        <div className="max-w-6xl mx-auto">
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-12 items-center">
            <div>
              <h2 className="text-2xl md:text-3xl font-bold mb-6">Our Culture</h2>
              <p className="text-muted-foreground mb-6">
                At FlashCore, we believe in fostering a culture of innovation, collaboration, and continuous learning. 
                Our team members are passionate about technology and committed to delivering exceptional value to our customers.
              </p>
              <div className="space-y-4">
                <div className="flex items-start space-x-3">
                  <Star className="w-5 h-5 text-yellow-500 mt-0.5" />
                  <div>
                    <h4 className="font-semibold">Excellence</h4>
                    <p className="text-sm text-muted-foreground">We strive for excellence in everything we do</p>
                  </div>
                </div>
                <div className="flex items-start space-x-3">
                  <Users className="w-5 h-5 text-blue-500 mt-0.5" />
                  <div>
                    <h4 className="font-semibold">Teamwork</h4>
                    <p className="text-sm text-muted-foreground">Collaboration is at the heart of our success</p>
                  </div>
                </div>
                <div className="flex items-start space-x-3">
                  <TrendingUp className="w-5 h-5 text-green-500 mt-0.5" />
                  <div>
                    <h4 className="font-semibold">Growth</h4>
                    <p className="text-sm text-muted-foreground">We invest in your professional development</p>
                  </div>
                </div>
              </div>
            </div>
            <div className="bg-gradient-to-r from-flashcore-purple to-flashcore-green rounded-lg p-8 text-white">
              <h3 className="text-xl font-bold mb-4">Join Our Mission</h3>
              <p className="mb-6 opacity-90">
                Help us revolutionize email marketing and empower businesses to connect with their customers in meaningful ways.
              </p>
              <Button variant="secondary" className="bg-white text-flashcore-purple hover:bg-gray-100">
                <Mail className="w-4 h-4 mr-2" />
                Contact Us
              </Button>
            </div>
          </div>
        </div>
      </section>

      {/* Application Process */}
      <section className="container mx-auto px-4 py-12 md:py-20">
        <div className="max-w-4xl mx-auto text-center">
          <h2 className="text-2xl md:text-3xl font-bold mb-6">Application Process</h2>
          <p className="text-muted-foreground mb-12">
            Here's what you can expect when applying to join our team
          </p>
          
          <div className="grid grid-cols-1 md:grid-cols-4 gap-6">
            <div className="text-center">
              <div className="w-12 h-12 bg-gradient-to-r from-flashcore-purple to-flashcore-green rounded-full flex items-center justify-center mx-auto mb-4">
                <span className="text-white font-bold">1</span>
              </div>
              <h3 className="font-semibold mb-2">Apply</h3>
              <p className="text-sm text-muted-foreground">Submit your application and resume</p>
            </div>
            <div className="text-center">
              <div className="w-12 h-12 bg-gradient-to-r from-flashcore-purple to-flashcore-green rounded-full flex items-center justify-center mx-auto mb-4">
                <span className="text-white font-bold">2</span>
              </div>
              <h3 className="font-semibold mb-2">Review</h3>
              <p className="text-sm text-muted-foreground">Our team reviews your application</p>
            </div>
            <div className="text-center">
              <div className="w-12 h-12 bg-gradient-to-r from-flashcore-purple to-flashcore-green rounded-full flex items-center justify-center mx-auto mb-4">
                <span className="text-white font-bold">3</span>
              </div>
              <h3 className="font-semibold mb-2">Interview</h3>
              <p className="text-sm text-muted-foreground">Meet with our team members</p>
            </div>
            <div className="text-center">
              <div className="w-12 h-12 bg-gradient-to-r from-flashcore-purple to-flashcore-green rounded-full flex items-center justify-center mx-auto mb-4">
                <span className="text-white font-bold">4</span>
              </div>
              <h3 className="font-semibold mb-2">Offer</h3>
              <p className="text-sm text-muted-foreground">Join the FlashCore team!</p>
            </div>
          </div>
        </div>
      </section>

      {/* Contact Section */}
      <section className="container mx-auto px-4 py-12 md:py-20 bg-muted/20">
        <div className="max-w-4xl mx-auto text-center">
          <h2 className="text-2xl md:text-3xl font-bold mb-4">Don't See the Right Fit?</h2>
          <p className="text-muted-foreground mb-8">
            We're always looking for talented individuals to join our team. 
            Send us your resume and we'll keep you in mind for future opportunities.
          </p>
          <div className="flex flex-col sm:flex-row gap-4 justify-center">
            <Button className="bg-flashcore-purple hover:bg-flashcore-purple/90">
              <Mail className="w-4 h-4 mr-2" />
              Send Resume
            </Button>
            <Button variant="outline">
              <ExternalLink className="w-4 h-4 mr-2" />
              Contact HR
            </Button>
          </div>
        </div>
      </section>

      <Footer />
    </div>
  );
};

export default CareersPage; 