import React, { useState } from 'react';
import { Link } from 'react-router-dom';
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import { 
  ArrowRight, 
  Search, 
  Calendar, 
  Clock, 
  User, 
  Tag, 
  BookOpen, 
  TrendingUp,
  Mail,
  BarChart3,
  Users,
  Zap,
  Target,
  Building,
  ShoppingCart,
  Globe,
  Shield,
  MessageSquare,
  FileText,
  Video,
  Award
} from "lucide-react";
import Navigation from "@/components/Navigation";
import Footer from "@/components/Footer";

const BlogPage = () => {
  const [searchQuery, setSearchQuery] = useState('');
  const [selectedCategory, setSelectedCategory] = useState('all');

  const categories = [
    { id: 'all', name: 'All Posts', icon: BookOpen },
    { id: 'email-marketing', name: 'Email Marketing', icon: Mail },
    { id: 'automation', name: 'Automation', icon: Zap },
    { id: 'analytics', name: 'Analytics', icon: BarChart3 },
    { id: 'ecommerce', name: 'E-commerce', icon: ShoppingCart },
    { id: 'enterprise', name: 'Enterprise', icon: Building },
    { id: 'tips', name: 'Tips & Tricks', icon: Target },
    { id: 'case-studies', name: 'Case Studies', icon: TrendingUp }
  ];

  const featuredPosts = [
    {
      id: 1,
      title: "10 Email Marketing Strategies That Actually Work in 2024",
      excerpt: "Discover the most effective email marketing strategies that successful businesses are using to increase engagement and drive conversions in 2024.",
      author: "Sarah Johnson",
      authorRole: "Email Marketing Expert",
      date: "March 15, 2024",
      readTime: "8 min read",
      category: "email-marketing",
      image: "/blog/email-strategies-2024.jpg",
      featured: true,
      tags: ["Email Marketing", "Strategy", "2024"]
    },
    {
      id: 2,
      title: "How to Build an Automated Email Funnel That Converts",
      excerpt: "Learn how to create a powerful automated email funnel that nurtures leads and converts prospects into loyal customers.",
      author: "Michael Chen",
      authorRole: "Automation Specialist",
      date: "March 12, 2024",
      readTime: "12 min read",
      category: "automation",
      image: "/blog/automated-funnel.jpg",
      featured: true,
      tags: ["Automation", "Funnel", "Conversion"]
    },
    {
      id: 3,
      title: "The Complete Guide to Email Analytics and ROI Tracking",
      excerpt: "Master email analytics to understand your campaign performance and maximize your return on investment.",
      author: "Emily Rodriguez",
      authorRole: "Data Analyst",
      date: "March 10, 2024",
      readTime: "10 min read",
      category: "analytics",
      image: "/blog/email-analytics.jpg",
      featured: true,
      tags: ["Analytics", "ROI", "Tracking"]
    }
  ];

  const recentPosts = [
    {
      id: 4,
      title: "E-commerce Email Marketing: From Abandoned Carts to Repeat Customers",
      excerpt: "Transform your e-commerce business with strategic email marketing campaigns that recover abandoned carts and build customer loyalty.",
      author: "Lisa Chen",
      authorRole: "E-commerce Specialist",
      date: "March 8, 2024",
      readTime: "6 min read",
      category: "ecommerce",
      image: "/blog/ecommerce-email.jpg",
      tags: ["E-commerce", "Abandoned Cart", "Customer Loyalty"]
    },
    {
      id: 5,
      title: "Enterprise Email Marketing: Scaling Success for Large Organizations",
      excerpt: "Discover how enterprise companies are leveraging email marketing to scale their operations and drive growth.",
      author: "David Thompson",
      authorRole: "Enterprise Consultant",
      date: "March 5, 2024",
      readTime: "9 min read",
      category: "enterprise",
      image: "/blog/enterprise-email.jpg",
      tags: ["Enterprise", "Scaling", "Growth"]
    },
    {
      id: 6,
      title: "15 Email Subject Lines That Get Opened (With Examples)",
      excerpt: "Learn the psychology behind effective email subject lines and see real examples that drive higher open rates.",
      author: "Amanda Kim",
      authorRole: "Copywriting Expert",
      date: "March 3, 2024",
      readTime: "7 min read",
      category: "tips",
      image: "/blog/subject-lines.jpg",
      tags: ["Subject Lines", "Open Rates", "Copywriting"]
    },
    {
      id: 7,
      title: "Case Study: How Company X Increased Email Revenue by 300%",
      excerpt: "An in-depth look at how one company transformed their email marketing strategy and achieved remarkable results.",
      author: "Robert Martinez",
      authorRole: "Case Study Writer",
      date: "February 28, 2024",
      readTime: "11 min read",
      category: "case-studies",
      image: "/blog/case-study-300.jpg",
      tags: ["Case Study", "Revenue", "Strategy"]
    },
    {
      id: 8,
      title: "The Future of Email Marketing: AI, Personalization, and Automation",
      excerpt: "Explore the emerging trends in email marketing and how AI and automation are shaping the future of customer communication.",
      author: "Jennifer Williams",
      authorRole: "Technology Analyst",
      date: "February 25, 2024",
      readTime: "8 min read",
      category: "automation",
      image: "/blog/future-email.jpg",
      tags: ["AI", "Personalization", "Future"]
    },
    {
      id: 9,
      title: "GDPR Compliance: A Complete Guide for Email Marketers",
      excerpt: "Everything you need to know about GDPR compliance in email marketing to protect your business and customers.",
      author: "Mark Johnson",
      authorRole: "Legal Expert",
      date: "February 22, 2024",
      readTime: "10 min read",
      category: "tips",
      image: "/blog/gdpr-compliance.jpg",
      tags: ["GDPR", "Compliance", "Legal"]
    }
  ];

  const allPosts = [...featuredPosts, ...recentPosts];

  const filteredPosts = allPosts.filter(post => {
    const matchesSearch = post.title.toLowerCase().includes(searchQuery.toLowerCase()) ||
                         post.excerpt.toLowerCase().includes(searchQuery.toLowerCase()) ||
                         post.tags.some(tag => tag.toLowerCase().includes(searchQuery.toLowerCase()));
    const matchesCategory = selectedCategory === 'all' || post.category === selectedCategory;
    return matchesSearch && matchesCategory;
  });

  return (
    <div className="min-h-screen bg-background">
      <Navigation />

      {/* Hero Section */}
      <section className="container mx-auto px-4 py-12 md:py-20">
        <div className="max-w-4xl mx-auto text-center">
          <Badge variant="secondary" className="mb-4">
            <BookOpen className="w-3 h-3 mr-1" />
            Blog & Resources
          </Badge>
          <h1 className="text-3xl md:text-4xl lg:text-5xl font-bold mb-6 bg-gradient-to-r from-flashcore-purple to-flashcore-green bg-clip-text text-transparent">
            Email Marketing Insights & Resources
          </h1>
          <p className="text-lg md:text-xl text-muted-foreground mb-8 max-w-2xl mx-auto">
            Stay ahead of the curve with the latest email marketing trends, tips, and strategies 
            from industry experts and successful businesses.
          </p>
          
          {/* Search Bar */}
          <div className="max-w-2xl mx-auto mb-8">
            <div className="relative">
              <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-muted-foreground w-5 h-5" />
              <Input
                type="text"
                placeholder="Search articles, tips, or topics..."
                value={searchQuery}
                onChange={(e) => setSearchQuery(e.target.value)}
                className="pl-10 pr-4 py-3 text-lg"
              />
            </div>
          </div>
        </div>
      </section>

      {/* Categories Filter */}
      <section className="container mx-auto px-4 py-8">
        <div className="max-w-6xl mx-auto">
          <div className="flex flex-wrap gap-2 justify-center">
            {categories.map((category) => (
              <Button
                key={category.id}
                variant={selectedCategory === category.id ? "default" : "outline"}
                size="sm"
                onClick={() => setSelectedCategory(category.id)}
                className="flex items-center space-x-2"
              >
                <category.icon className="w-4 h-4" />
                <span>{category.name}</span>
              </Button>
            ))}
          </div>
        </div>
      </section>

      {/* Featured Posts Section */}
      <section className="container mx-auto px-4 py-12 md:py-20">
        <div className="max-w-6xl mx-auto">
          <div className="text-center mb-12">
            <h2 className="text-2xl md:text-3xl font-bold mb-4">Featured Articles</h2>
            <p className="text-muted-foreground">
              Our most popular and insightful articles on email marketing.
            </p>
          </div>
          
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6 md:gap-8">
            {featuredPosts.map((post) => (
              <Card key={post.id} className="hover:shadow-lg transition-shadow group">
                <div className="aspect-video bg-gradient-to-br from-flashcore-purple to-flashcore-green rounded-t-lg mb-4"></div>
                <CardHeader>
                  <div className="flex items-center space-x-2 mb-2">
                    <Badge variant="secondary" className="text-xs">
                      {categories.find(cat => cat.id === post.category)?.name}
                    </Badge>
                    <Badge variant="outline" className="text-xs">
                      Featured
                    </Badge>
                  </div>
                  <CardTitle className="text-lg group-hover:text-flashcore-purple transition-colors">
                    {post.title}
                  </CardTitle>
                  <CardDescription className="text-sm">
                    {post.excerpt}
                  </CardDescription>
                </CardHeader>
                <CardContent>
                  <div className="flex items-center justify-between text-xs text-muted-foreground mb-4">
                    <div className="flex items-center space-x-4">
                      <span className="flex items-center">
                        <User className="w-3 h-3 mr-1" />
                        {post.author}
                      </span>
                      <span className="flex items-center">
                        <Calendar className="w-3 h-3 mr-1" />
                        {post.date}
                      </span>
                    </div>
                    <span className="flex items-center">
                      <Clock className="w-3 h-3 mr-1" />
                      {post.readTime}
                    </span>
                  </div>
                  <div className="flex flex-wrap gap-1 mb-4">
                    {post.tags.map((tag, index) => (
                      <Badge key={index} variant="outline" className="text-xs">
                        {tag}
                      </Badge>
                    ))}
                  </div>
                  <Button variant="outline" size="sm" className="w-full group-hover:bg-flashcore-purple group-hover:text-white transition-colors">
                    Read Article
                    <ArrowRight className="w-4 h-4 ml-2" />
                  </Button>
                </CardContent>
              </Card>
            ))}
          </div>
        </div>
      </section>

      {/* Recent Posts Section */}
      <section className="container mx-auto px-4 py-12 md:py-20 bg-muted/20">
        <div className="max-w-6xl mx-auto">
          <div className="text-center mb-12">
            <h2 className="text-2xl md:text-3xl font-bold mb-4">Latest Articles</h2>
            <p className="text-muted-foreground">
              Fresh insights and tips from our email marketing experts.
            </p>
          </div>
          
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6 md:gap-8">
            {filteredPosts.slice(3).map((post) => (
              <Card key={post.id} className="hover:shadow-lg transition-shadow group">
                <div className="aspect-video bg-gradient-to-br from-gray-200 to-gray-300 rounded-t-lg mb-4"></div>
                <CardHeader>
                  <div className="flex items-center space-x-2 mb-2">
                    <Badge variant="secondary" className="text-xs">
                      {categories.find(cat => cat.id === post.category)?.name}
                    </Badge>
                  </div>
                  <CardTitle className="text-lg group-hover:text-flashcore-purple transition-colors">
                    {post.title}
                  </CardTitle>
                  <CardDescription className="text-sm">
                    {post.excerpt}
                  </CardDescription>
                </CardHeader>
                <CardContent>
                  <div className="flex items-center justify-between text-xs text-muted-foreground mb-4">
                    <div className="flex items-center space-x-4">
                      <span className="flex items-center">
                        <User className="w-3 h-3 mr-1" />
                        {post.author}
                      </span>
                      <span className="flex items-center">
                        <Calendar className="w-3 h-3 mr-1" />
                        {post.date}
                      </span>
                    </div>
                    <span className="flex items-center">
                      <Clock className="w-3 h-3 mr-1" />
                      {post.readTime}
                    </span>
                  </div>
                  <div className="flex flex-wrap gap-1 mb-4">
                    {post.tags.map((tag, index) => (
                      <Badge key={index} variant="outline" className="text-xs">
                        {tag}
                      </Badge>
                    ))}
                  </div>
                  <Button variant="outline" size="sm" className="w-full group-hover:bg-flashcore-purple group-hover:text-white transition-colors">
                    Read Article
                    <ArrowRight className="w-4 h-4 ml-2" />
                  </Button>
                </CardContent>
              </Card>
            ))}
          </div>
        </div>
      </section>

      {/* Newsletter Section */}
      <section className="container mx-auto px-4 py-12 md:py-20">
        <div className="max-w-4xl mx-auto">
          <Card className="bg-gradient-to-r from-flashcore-purple to-flashcore-green text-white">
            <CardContent className="p-8 md:p-12 text-center">
              <h2 className="text-2xl md:text-3xl font-bold mb-4">Stay Updated with Email Marketing Insights</h2>
              <p className="text-lg mb-6 opacity-90">
                Get the latest email marketing tips, strategies, and industry insights delivered to your inbox.
              </p>
              <div className="flex flex-col sm:flex-row gap-4 max-w-md mx-auto">
                <Input
                  type="email"
                  placeholder="Enter your email address"
                  className="bg-white text-gray-900 placeholder-gray-500"
                />
                <Button variant="secondary" className="bg-white text-flashcore-purple hover:bg-gray-100">
                  Subscribe
                  <ArrowRight className="w-4 h-4 ml-2" />
                </Button>
              </div>
              <p className="text-sm opacity-75 mt-4">
                No spam, unsubscribe at any time. We respect your privacy.
              </p>
            </CardContent>
          </Card>
        </div>
      </section>

      {/* Resources Section */}
      <section className="container mx-auto px-4 py-12 md:py-20 bg-muted/20">
        <div className="max-w-6xl mx-auto">
          <div className="text-center mb-12">
            <h2 className="text-2xl md:text-3xl font-bold mb-4">Additional Resources</h2>
            <p className="text-muted-foreground">
              Explore more resources to help you master email marketing.
            </p>
          </div>
          
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 md:gap-8">
            <Card className="hover:shadow-lg transition-shadow">
              <CardContent className="p-6 text-center">
                <div className="w-12 h-12 bg-gradient-to-r from-flashcore-purple to-flashcore-green rounded-lg flex items-center justify-center mx-auto mb-4">
                  <FileText className="w-6 h-6 text-white" />
                </div>
                <h3 className="text-lg font-semibold mb-2">Email Templates</h3>
                <p className="text-sm text-muted-foreground mb-4">
                  Professional email templates for every occasion.
                </p>
                <Link to="/templates">
                  <Button variant="outline" size="sm" className="w-full">
                    Browse Templates
                  </Button>
                </Link>
              </CardContent>
            </Card>
            
            <Card className="hover:shadow-lg transition-shadow">
              <CardContent className="p-6 text-center">
                <div className="w-12 h-12 bg-gradient-to-r from-flashcore-purple to-flashcore-green rounded-lg flex items-center justify-center mx-auto mb-4">
                  <Video className="w-6 h-6 text-white" />
                </div>
                <h3 className="text-lg font-semibold mb-2">Video Tutorials</h3>
                <p className="text-sm text-muted-foreground mb-4">
                  Step-by-step video guides and tutorials.
                </p>
                <Link to="/tutorials">
                  <Button variant="outline" size="sm" className="w-full">
                    Watch Videos
                  </Button>
                </Link>
              </CardContent>
            </Card>
            
            <Card className="hover:shadow-lg transition-shadow">
              <CardContent className="p-6 text-center">
                <div className="w-12 h-12 bg-gradient-to-r from-flashcore-purple to-flashcore-green rounded-lg flex items-center justify-center mx-auto mb-4">
                  <MessageSquare className="w-6 h-6 text-white" />
                </div>
                <h3 className="text-lg font-semibold mb-2">Community</h3>
                <p className="text-sm text-muted-foreground mb-4">
                  Connect with other email marketers.
                </p>
                <Link to="/community">
                  <Button variant="outline" size="sm" className="w-full">
                    Join Community
                  </Button>
                </Link>
              </CardContent>
            </Card>
            
            <Card className="hover:shadow-lg transition-shadow">
              <CardContent className="p-6 text-center">
                <div className="w-12 h-12 bg-gradient-to-r from-flashcore-purple to-flashcore-green rounded-lg flex items-center justify-center mx-auto mb-4">
                  <Award className="w-6 h-6 text-white" />
                </div>
                <h3 className="text-lg font-semibold mb-2">Certification</h3>
                <p className="text-sm text-muted-foreground mb-4">
                  Get certified in email marketing.
                </p>
                <Link to="/certification">
                  <Button variant="outline" size="sm" className="w-full">
                    Get Certified
                  </Button>
                </Link>
              </CardContent>
            </Card>
          </div>
        </div>
      </section>

      <Footer />
    </div>
  );
};

export default BlogPage; 