import React, { useState } from 'react';
import { Link } from 'react-router-dom';
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Code, Copy, Check, BookOpen, Zap, Shield, Database, ArrowRight, ExternalLink, Terminal, Key, Lock } from "lucide-react";
import Navigation from "@/components/Navigation";
import Footer from "@/components/Footer";

const ApiPage = () => {
  const [copiedCode, setCopiedCode] = useState<string | null>(null);

  const copyToClipboard = (code: string, id: string) => {
    navigator.clipboard.writeText(code);
    setCopiedCode(id);
    setTimeout(() => setCopiedCode(null), 2000);
  };

  const endpoints = [
    {
      method: "GET",
      path: "/api/v1/emails",
      description: "Retrieve all emails",
      auth: "Required",
      params: ["page", "limit", "status"]
    },
    {
      method: "POST",
      path: "/api/v1/emails",
      description: "Send a new email",
      auth: "Required",
      params: ["to", "subject", "body", "template_id"]
    },
    {
      method: "GET",
      path: "/api/v1/emails/{id}",
      description: "Get email details",
      auth: "Required",
      params: ["id"]
    },
    {
      method: "PUT",
      path: "/api/v1/emails/{id}",
      description: "Update email",
      auth: "Required",
      params: ["id", "subject", "body"]
    },
    {
      method: "DELETE",
      path: "/api/v1/emails/{id}",
      description: "Delete email",
      auth: "Required",
      params: ["id"]
    },
    {
      method: "GET",
      path: "/api/v1/templates",
      description: "List email templates",
      auth: "Required",
      params: ["category", "page", "limit"]
    },
    {
      method: "POST",
      path: "/api/v1/templates",
      description: "Create new template",
      auth: "Required",
      params: ["name", "subject", "body", "category"]
    },
    {
      method: "GET",
      path: "/api/v1/analytics",
      description: "Get email analytics",
      auth: "Required",
      params: ["start_date", "end_date", "metrics"]
    }
  ];

  const codeExamples = {
    authentication: `// Get your API key from the dashboard
const API_KEY = 'your_api_key_here';

// Include in headers
const headers = {
  'Authorization': 'Bearer ' + API_KEY,
  'Content-Type': 'application/json'
};`,
    
    sendEmail: `// Send a simple email
const response = await fetch('https://api.flashcore.com/v1/emails', {
  method: 'POST',
  headers: {
    'Authorization': 'Bearer ' + API_KEY,
    'Content-Type': 'application/json'
  },
  body: JSON.stringify({
    to: 'recipient@example.com',
    subject: 'Hello from FlashCore',
    body: '<h1>Welcome!</h1><p>This is a test email.</p>',
    template_id: 'welcome_template'
  })
});

const result = await response.json();
console.log(result);`,
    
    getEmails: `// Retrieve emails with pagination
const response = await fetch('https://api.flashcore.com/v1/emails?page=1&limit=10&status=sent', {
  headers: {
    'Authorization': 'Bearer ' + API_KEY
  }
});

const emails = await response.json();
console.log(emails.data);`,
    
    webhooks: `// Set up webhook endpoint
app.post('/webhook', (req, res) => {
  const { event, data } = req.body;
  
  switch(event) {
    case 'email.sent':
      console.log('Email sent:', data);
      break;
    case 'email.delivered':
      console.log('Email delivered:', data);
      break;
    case 'email.opened':
      console.log('Email opened:', data);
      break;
  }
  
  res.status(200).json({ received: true });
});`
  };

  const sdks = [
    {
      name: "JavaScript/Node.js",
      version: "v2.1.0",
      install: "npm install @flashcore/api",
      color: "from-yellow-500 to-orange-500"
    },
    {
      name: "Python",
      version: "v1.8.0",
      install: "pip install flashcore-api",
      color: "from-blue-500 to-purple-500"
    },
    {
      name: "PHP",
      version: "v1.5.0",
      install: "composer require flashcore/api",
      color: "from-purple-500 to-pink-500"
    },
    {
      name: "Ruby",
      version: "v1.3.0",
      install: "gem install flashcore-api",
      color: "from-red-500 to-pink-500"
    }
  ];

  return (
    <div className="min-h-screen bg-background">
      <Navigation />

      {/* Header */}
      <section className="container mx-auto px-4 py-12 md:py-20 text-center">
        <div className="max-w-4xl mx-auto">
          <Badge variant="secondary" className="mb-4">
            <Code className="w-3 h-3 mr-1" />
            API Documentation
          </Badge>
          <h1 className="text-3xl md:text-4xl lg:text-5xl font-bold mb-6 bg-gradient-to-r from-flashcore-purple to-flashcore-green bg-clip-text text-transparent">
            API Reference
          </h1>
          <p className="text-lg md:text-xl text-muted-foreground mb-8 max-w-2xl mx-auto">
            Integrate FlashCore into your applications with our powerful REST API. Send emails, manage templates, and track analytics programmatically.
          </p>
          <div className="flex flex-col sm:flex-row gap-4 justify-center">
            <Button size="lg" className="bg-gradient-to-r from-flashcore-purple to-flashcore-green">
              <Key className="w-4 h-4 mr-2" />
              Get API Key
            </Button>
            <Button size="lg" variant="outline">
              <BookOpen className="w-4 h-4 mr-2" />
              View SDKs
            </Button>
          </div>
        </div>
      </section>

      {/* Quick Start */}
      <section className="container mx-auto px-4 py-12 md:py-20">
        <div className="max-w-4xl mx-auto">
          <div className="text-center mb-12">
            <h2 className="text-2xl md:text-3xl font-bold mb-4">Quick Start</h2>
            <p className="text-muted-foreground">
              Get up and running with FlashCore API in minutes.
            </p>
          </div>
          
          <Tabs defaultValue="authentication" className="w-full">
            <TabsList className="grid w-full grid-cols-4">
              <TabsTrigger value="authentication">Authentication</TabsTrigger>
              <TabsTrigger value="send-email">Send Email</TabsTrigger>
              <TabsTrigger value="get-emails">Get Emails</TabsTrigger>
              <TabsTrigger value="webhooks">Webhooks</TabsTrigger>
            </TabsList>
            
            <TabsContent value="authentication" className="mt-6">
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center">
                    <Key className="w-5 h-5 mr-2" />
                    API Authentication
                  </CardTitle>
                  <CardDescription>
                    All API requests require authentication using your API key.
                  </CardDescription>
                </CardHeader>
                <CardContent>
                  <div className="relative">
                    <pre className="bg-muted p-4 rounded-lg overflow-x-auto">
                      <code className="text-sm">{codeExamples.authentication}</code>
                    </pre>
                    <Button
                      size="sm"
                      variant="outline"
                      className="absolute top-2 right-2"
                      onClick={() => copyToClipboard(codeExamples.authentication, 'auth')}
                    >
                      {copiedCode === 'auth' ? <Check className="w-4 h-4" /> : <Copy className="w-4 h-4" />}
                    </Button>
                  </div>
                </CardContent>
              </Card>
            </TabsContent>
            
            <TabsContent value="send-email" className="mt-6">
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center">
                    <Zap className="w-5 h-5 mr-2" />
                    Send an Email
                  </CardTitle>
                  <CardDescription>
                    Send emails using the API with support for templates and custom content.
                  </CardDescription>
                </CardHeader>
                <CardContent>
                  <div className="relative">
                    <pre className="bg-muted p-4 rounded-lg overflow-x-auto">
                      <code className="text-sm">{codeExamples.sendEmail}</code>
                    </pre>
                    <Button
                      size="sm"
                      variant="outline"
                      className="absolute top-2 right-2"
                      onClick={() => copyToClipboard(codeExamples.sendEmail, 'send')}
                    >
                      {copiedCode === 'send' ? <Check className="w-4 h-4" /> : <Copy className="w-4 h-4" />}
                    </Button>
                  </div>
                </CardContent>
              </Card>
            </TabsContent>
            
            <TabsContent value="get-emails" className="mt-6">
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center">
                    <Database className="w-5 h-5 mr-2" />
                    Retrieve Emails
                  </CardTitle>
                  <CardDescription>
                    Fetch emails with pagination and filtering options.
                  </CardDescription>
                </CardHeader>
                <CardContent>
                  <div className="relative">
                    <pre className="bg-muted p-4 rounded-lg overflow-x-auto">
                      <code className="text-sm">{codeExamples.getEmails}</code>
                    </pre>
                    <Button
                      size="sm"
                      variant="outline"
                      className="absolute top-2 right-2"
                      onClick={() => copyToClipboard(codeExamples.getEmails, 'get')}
                    >
                      {copiedCode === 'get' ? <Check className="w-4 h-4" /> : <Copy className="w-4 h-4" />}
                    </Button>
                  </div>
                </CardContent>
              </Card>
            </TabsContent>
            
            <TabsContent value="webhooks" className="mt-6">
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center">
                    <Shield className="w-5 h-5 mr-2" />
                    Webhooks
                  </CardTitle>
                  <CardDescription>
                    Receive real-time notifications about email events.
                  </CardDescription>
                </CardHeader>
                <CardContent>
                  <div className="relative">
                    <pre className="bg-muted p-4 rounded-lg overflow-x-auto">
                      <code className="text-sm">{codeExamples.webhooks}</code>
                    </pre>
                    <Button
                      size="sm"
                      variant="outline"
                      className="absolute top-2 right-2"
                      onClick={() => copyToClipboard(codeExamples.webhooks, 'webhook')}
                    >
                      {copiedCode === 'webhook' ? <Check className="w-4 h-4" /> : <Copy className="w-4 h-4" />}
                    </Button>
                  </div>
                </CardContent>
              </Card>
            </TabsContent>
          </Tabs>
        </div>
      </section>

      {/* SDKs */}
      <section className="container mx-auto px-4 py-12 md:py-20 bg-muted/20">
        <div className="max-w-6xl mx-auto">
          <div className="text-center mb-12">
            <h2 className="text-2xl md:text-3xl font-bold mb-4">Official SDKs</h2>
            <p className="text-muted-foreground">
              Use our official SDKs for faster integration and better developer experience.
            </p>
          </div>
          
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
            {sdks.map((sdk) => (
              <Card key={sdk.name} className="hover:shadow-lg transition-shadow">
                <CardHeader>
                  <div className={`w-12 h-12 bg-gradient-to-r ${sdk.color} rounded-lg flex items-center justify-center mb-4`}>
                    <Terminal className="w-6 h-6 text-white" />
                  </div>
                  <CardTitle className="text-lg">{sdk.name}</CardTitle>
                  <CardDescription>
                    Version {sdk.version}
                  </CardDescription>
                </CardHeader>
                <CardContent>
                  <div className="space-y-3">
                    <div className="relative">
                      <pre className="bg-background p-3 rounded text-xs overflow-x-auto">
                        <code>{sdk.install}</code>
                      </pre>
                      <Button
                        size="sm"
                        variant="outline"
                        className="absolute top-1 right-1 h-6 w-6 p-0"
                        onClick={() => copyToClipboard(sdk.install, sdk.name)}
                      >
                        {copiedCode === sdk.name ? <Check className="w-3 h-3" /> : <Copy className="w-3 h-3" />}
                      </Button>
                    </div>
                    <Button variant="outline" size="sm" className="w-full">
                      <ExternalLink className="w-4 h-4 mr-2" />
                      Documentation
                    </Button>
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>
        </div>
      </section>

      {/* API Endpoints */}
      <section className="container mx-auto px-4 py-12 md:py-20">
        <div className="max-w-6xl mx-auto">
          <div className="text-center mb-12">
            <h2 className="text-2xl md:text-3xl font-bold mb-4">API Endpoints</h2>
            <p className="text-muted-foreground">
              Complete reference of all available API endpoints.
            </p>
          </div>
          
          <div className="space-y-4">
            {endpoints.map((endpoint, index) => (
              <Card key={index} className="hover:shadow-md transition-shadow">
                <CardContent className="p-6">
                  <div className="flex flex-col md:flex-row md:items-center justify-between gap-4">
                    <div className="flex items-center gap-4">
                      <Badge 
                        variant={endpoint.method === 'GET' ? 'default' : 
                                endpoint.method === 'POST' ? 'secondary' :
                                endpoint.method === 'PUT' ? 'outline' : 'destructive'}
                        className="w-16 text-center"
                      >
                        {endpoint.method}
                      </Badge>
                      <div>
                        <code className="text-sm md:text-base font-mono bg-muted px-2 py-1 rounded">
                          {endpoint.path}
                        </code>
                      </div>
                    </div>
                    <div className="flex items-center gap-4">
                      <div className="text-sm text-muted-foreground">
                        {endpoint.description}
                      </div>
                      <Badge variant="outline" className="text-xs">
                        {endpoint.auth}
                      </Badge>
                    </div>
                  </div>
                  {endpoint.params.length > 0 && (
                    <div className="mt-4 pt-4 border-t">
                      <div className="text-sm text-muted-foreground mb-2">Parameters:</div>
                      <div className="flex flex-wrap gap-2">
                        {endpoint.params.map((param) => (
                          <Badge key={param} variant="secondary" className="text-xs">
                            {param}
                          </Badge>
                        ))}
                      </div>
                    </div>
                  )}
                </CardContent>
              </Card>
            ))}
          </div>
        </div>
      </section>

      {/* Rate Limits & Security */}
      <section className="container mx-auto px-4 py-12 md:py-20 bg-muted/20">
        <div className="max-w-4xl mx-auto">
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-8 md:gap-12">
            <div>
              <h2 className="text-2xl md:text-3xl font-bold mb-4 md:mb-6">Rate Limits</h2>
              <p className="text-base md:text-lg text-muted-foreground mb-4 md:mb-6">
                We implement rate limiting to ensure fair usage and system stability.
              </p>
              <div className="space-y-4">
                <div className="flex items-center justify-between p-4 bg-background rounded-lg">
                  <span className="font-medium">Free Plan</span>
                  <span className="text-sm text-muted-foreground">1,000 requests/hour</span>
                </div>
                <div className="flex items-center justify-between p-4 bg-background rounded-lg">
                  <span className="font-medium">Pro Plan</span>
                  <span className="text-sm text-muted-foreground">10,000 requests/hour</span>
                </div>
                <div className="flex items-center justify-between p-4 bg-background rounded-lg">
                  <span className="font-medium">Enterprise</span>
                  <span className="text-sm text-muted-foreground">100,000 requests/hour</span>
                </div>
              </div>
            </div>
            
            <div>
              <h2 className="text-2xl md:text-3xl font-bold mb-4 md:mb-6">Security</h2>
              <p className="text-base md:text-lg text-muted-foreground mb-4 md:mb-6">
                Your data security is our top priority. All API communications are encrypted and secure.
              </p>
              <div className="space-y-4">
                <div className="flex items-center">
                  <Lock className="w-5 h-5 text-green-500 mr-3" />
                  <span className="text-sm md:text-base">HTTPS encryption for all requests</span>
                </div>
                <div className="flex items-center">
                  <Shield className="w-5 h-5 text-green-500 mr-3" />
                  <span className="text-sm md:text-base">API keys are encrypted and secure</span>
                </div>
                <div className="flex items-center">
                  <Check className="w-5 h-5 text-green-500 mr-3" />
                  <span className="text-sm md:text-base">SOC 2 Type II compliance</span>
                </div>
                <div className="flex items-center">
                  <Database className="w-5 h-5 text-green-500 mr-3" />
                  <span className="text-sm md:text-base">GDPR compliant data handling</span>
                </div>
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* CTA Section */}
      <section className="container mx-auto px-4 py-12 md:py-20">
        <div className="bg-gradient-to-r from-flashcore-purple to-flashcore-green rounded-2xl p-6 md:p-8 lg:p-12 text-center text-white">
          <h2 className="text-2xl md:text-3xl font-bold mb-4">Ready to Get Started?</h2>
          <p className="text-lg md:text-xl mb-6 md:mb-8 opacity-90">
            Start building with FlashCore API today and transform your email automation.
          </p>
          <div className="flex flex-col sm:flex-row gap-4 justify-center">
            <Link to="/register">
              <Button size="lg" variant="secondary" className="bg-white text-flashcore-purple hover:bg-gray-100">
                Get API Key
                <ArrowRight className="w-4 h-4 ml-2" />
              </Button>
            </Link>
            <Link to="/contact">
              <Button size="lg" variant="outline" className="border-white text-white hover:bg-white hover:text-flashcore-purple">
                Contact Support
              </Button>
            </Link>
          </div>
        </div>
      </section>

      <Footer />
    </div>
  );
};

export default ApiPage; 