import React, { useState } from "react";
import { Link, useNavigate } from "react-router-dom";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Card, CardContent, CardDescription, CardFooter, CardHeader, CardTitle } from "@/components/ui/card";
import { Label } from "@/components/ui/label";
import { LogIn, Shield, Lock, AlertTriangle, Eye, EyeOff } from "lucide-react";
import Logo from "@/components/Logo";
import { useToast } from "@/components/ui/use-toast";
import { useAuthPhp } from "@/hooks/useAuthPhp";

const AdminLoginPage = () => {
  const [email, setEmail] = useState("");
  const [password, setPassword] = useState("");
  const [showPassword, setShowPassword] = useState(false);
  const [loading, setLoading] = useState(false);
  const navigate = useNavigate();
  const { toast } = useToast();
  const { login } = useAuthPhp();

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setLoading(true);
    
    try {
      console.log('Attempting admin login for:', email);
      
      const response = await login(email, password);
      
      if (!response.success) {
        console.error('Login error:', response.error);
        
        // Provide more specific error messages
        let errorMessage = response.error;
        if (response.error.includes('Invalid login credentials')) {
          errorMessage = 'Invalid email or password. Please check your credentials.';
        } else if (response.error.includes('Email not confirmed')) {
          errorMessage = 'Please check your email and confirm your account before logging in.';
        } else if (response.error.includes('NetworkError')) {
          errorMessage = 'Network error. Please check your internet connection and try again.';
        }
        
        toast({
          title: "Login Error",
          description: errorMessage,
          variant: "destructive",
        });
        return;
      }

      if (response.success) {
        console.log('User authenticated, checking role...');
        
        // In the PHP version, we would check the user's role here
        // For now, we'll assume the login response includes role information
        // Fixed: Access the profile from the user object stored in localStorage
        const userProfile = localStorage.getItem('authProfile');
        const profile = userProfile ? JSON.parse(userProfile) : null;
        const isAdmin = profile?.role === 'admin';
        
        if (!isAdmin) {
          console.log('User is not admin');
          toast({
            title: "Access Denied",
            description: "This login is restricted to administrators only.",
            variant: "destructive",
          });
          return;
        }

        console.log('Admin access granted, redirecting...');
        toast({
          title: "Admin Access Granted",
          description: "Welcome, Administrator. You have successfully logged in.",
        });

        navigate('/admin');
      }
    } catch (error) {
      console.error('Unexpected error:', error);
      toast({
        title: "Login Error",
        description: "An unexpected error occurred during login. Please try again.",
        variant: "destructive",
      });
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className="min-h-screen flex items-center justify-center bg-gradient-to-br from-gray-900 via-blue-900 to-gray-900 p-4">
      <div className="absolute top-8 left-8">
        <Link to="/" className="flex items-center">
          <Logo className="w-12 h-12" />
          <h1 className="text-2xl font-bold bg-gradient-to-r from-blue-400 via-purple-500 to-blue-600 bg-clip-text text-transparent -ml-1">
            FlashCore
          </h1>
        </Link>
      </div>

      <Card className="w-full max-w-md bg-white dark:bg-gray-800 shadow-xl border-0">
        <CardHeader className="space-y-1">
          <div className="flex items-center justify-center mb-4">
            <div className="bg-gradient-to-r from-blue-600 to-purple-600 p-3 rounded-full">
              <Shield className="w-6 h-6 text-white" />
            </div>
          </div>
          <CardTitle className="text-2xl text-center text-gray-900 dark:text-white">Admin Portal</CardTitle>
          <CardDescription className="text-center text-gray-600 dark:text-gray-300">
            Secure administrative access
          </CardDescription>
        </CardHeader>
        <CardContent className="space-y-4">
          <div className="bg-blue-50 border-l-4 border-blue-500 p-4 mb-4 dark:bg-blue-900/30">
            <div className="flex">
              <Shield className="h-5 w-5 text-blue-500 mr-2 mt-0.5" />
              <div>
                <h3 className="text-sm font-medium text-blue-800 dark:text-blue-200">Administrator Access</h3>
                <div className="text-sm text-blue-700 dark:text-blue-300">
                  <p>Restricted access for authorized personnel only</p>
                </div>
              </div>
            </div>
          </div>
          
          <form onSubmit={handleSubmit}>
            <div className="space-y-4">
              <div className="space-y-2">
                <Label htmlFor="email" className="text-gray-700 dark:text-gray-300">Admin Email</Label>
                <div className="relative">
                  <Lock className="absolute left-3 top-3 h-4 w-4 text-gray-400" />
                  <Input
                    id="email"
                    type="email"
                    placeholder="admin@flashcore.com"
                    value={email}
                    onChange={(e) => setEmail(e.target.value)}
                    className="pl-10 bg-gray-50 dark:bg-gray-700 border-gray-200 dark:border-gray-600"
                    required
                  />
                </div>
              </div>
              <div className="space-y-2">
                <Label htmlFor="password" className="text-gray-700 dark:text-gray-300">Password</Label>
                <div className="relative">
                  <Lock className="absolute left-3 top-3 h-4 w-4 text-gray-400" />
                  <Input
                    id="password"
                    type={showPassword ? "text" : "password"}
                    value={password}
                    onChange={(e) => setPassword(e.target.value)}
                    className="pl-10 pr-10 bg-gray-50 dark:bg-gray-700 border-gray-200 dark:border-gray-600"
                    required
                  />
                  <Button
                    type="button"
                    variant="ghost"
                    size="icon"
                    className="absolute right-2 top-2 h-6 w-6 text-gray-400 hover:text-gray-600"
                    onClick={() => setShowPassword(!showPassword)}
                  >
                    {showPassword ? <EyeOff className="h-4 w-4" /> : <Eye className="h-4 w-4" />}
                  </Button>
                </div>
              </div>
            </div>
            <Button className="w-full mt-6 bg-gradient-to-r from-blue-600 to-purple-600 hover:from-blue-700 hover:to-purple-700" type="submit" disabled={loading}>
              {loading ? (
                <div className="flex items-center">
                  <div className="w-4 h-4 border-2 border-t-transparent border-white rounded-full animate-spin mr-2"></div>
                  Authenticating...
                </div>
              ) : (
                <>
                  <LogIn className="w-4 h-4 mr-2" />
                  Sign In
                </>
              )}
            </Button>
          </form>
        </CardContent>
        <CardFooter className="flex flex-col space-y-4">
          <div className="text-sm text-center text-gray-600 dark:text-gray-400">
            <Link to="/login" className="text-blue-600 hover:text-blue-800 dark:text-blue-400 dark:hover:text-blue-300">
              Regular User Login
            </Link>
          </div>
          <div className="text-xs text-center text-gray-500 dark:text-gray-400">
            © {new Date().getFullYear()} FlashCore. All rights reserved.
          </div>
        </CardFooter>
      </Card>
    </div>
  );
};

export default AdminLoginPage;