import React, { useState, useEffect } from "react";
import { useLocation } from "react-router-dom";
import { apiService } from "@/services/api";
import { useToast } from "@/components/ui/use-toast";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { 
  Users, 
  Key, 
  CreditCard, 
  TrendingUp, 
  Package, 
  Settings, 
  Mail, 
  Shield 
} from "lucide-react";
import AdminLayout from "@/components/admin/AdminLayout";
import LicenseManagement from "@/components/admin/LicenseManagement";
import UserManagement from "@/components/admin/UserManagement";
import SubscribersTable from "@/components/admin/SubscribersTable";
import SmtpSettings from "@/components/admin/SmtpSettings";
import EmailTemplates from "@/components/admin/EmailTemplates";

const AdminDashboard = () => {
  const location = useLocation();
  const [licenses, setLicenses] = useState([]);
  const [subscribers, setSubscribers] = useState([]);
  const [orders, setOrders] = useState([]);
  const [loading, setLoading] = useState(true);
  const { toast } = useToast();

  // Get active tab from URL query params
  const getActiveTab = () => {
    const searchParams = new URLSearchParams(location.search);
    return searchParams.get('tab') || 'dashboard';
  };

  const [activeTab, setActiveTab] = useState(getActiveTab());

  useEffect(() => {
    setActiveTab(getActiveTab());
  }, [location.search]);

  const fetchData = async () => {
    try {
      setLoading(true);
      
      // Fetch licenses using PHP API
      const response = await apiService.adminManageLicenses('list');
      
      if (response.data?.licenses) setLicenses(response.data.licenses);
      
    } catch (error) {
      console.error('Error fetching data:', error);
      toast({
        title: "Error",
        description: "Failed to fetch data",
        variant: "destructive"
      });
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchData();
  }, []);

  // Stats cards data
  const stats = [
    {
      title: "Total Users",
      value: "1,245",
      description: "+12% from last month",
      icon: Users,
      change: "+12%",
      changeType: "positive"
    },
    {
      title: "Active Licenses",
      value: "87",
      description: "+5% from last month",
      icon: Key,
      change: "+5%",
      changeType: "positive"
    },
    {
      title: "Revenue",
      value: "$24,500",
      description: "+18% from last month",
      icon: CreditCard,
      change: "+18%",
      changeType: "positive"
    },
    {
      title: "Subscribers",
      value: "3,280",
      description: "+8% from last month",
      icon: Package,
      change: "+8%",
      changeType: "positive"
    }
  ];

  if (loading) {
    return (
      <AdminLayout>
        <div className="flex items-center justify-center h-64">
          <div className="w-8 h-8 border-4 border-blue-500 border-t-transparent rounded-full animate-spin"></div>
        </div>
      </AdminLayout>
    );
  }

  // Render content based on active tab
  const renderTabContent = () => {
    switch (activeTab) {
      case 'licenses':
        return <LicenseManagement licenses={licenses} onRefresh={fetchData} />;
      case 'users':
        return <UserManagement />;
      case 'subscribers':
        return <SubscribersTable subscribers={subscribers} />;
      case 'orders':
        return (
          <Card>
            <CardHeader>
              <CardTitle>Orders Management</CardTitle>
              <CardDescription>Manage customer orders and subscriptions</CardDescription>
            </CardHeader>
            <CardContent>
              <div className="text-center py-12">
                <Package className="mx-auto h-12 w-12 text-gray-400" />
                <h3 className="mt-2 text-sm font-medium text-gray-900 dark:text-white">Orders management coming soon</h3>
                <p className="mt-1 text-sm text-gray-500 dark:text-gray-400">
                  This feature is currently under development.
                </p>
              </div>
            </CardContent>
          </Card>
        );
      case 'templates':
        return <EmailTemplates />;
      case 'settings':
        return <SmtpSettings />;
      default:
        return (
          <div className="space-y-6">
            {/* Stats Grid */}
            <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
              {stats.map((stat, index) => {
                const Icon = stat.icon;
                return (
                  <Card key={index}>
                    <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                      <CardTitle className="text-sm font-medium">
                        {stat.title}
                      </CardTitle>
                      <Icon className="h-4 w-4 text-muted-foreground" />
                    </CardHeader>
                    <CardContent>
                      <div className="text-2xl font-bold">{stat.value}</div>
                      <p className="text-xs text-muted-foreground">
                        {stat.description}
                      </p>
                    </CardContent>
                  </Card>
                );
              })}
            </div>

            {/* Recent Activity */}
            <div className="grid gap-4 md:grid-cols-2">
              <Card>
                <CardHeader>
                  <CardTitle>Recent Licenses</CardTitle>
                  <CardDescription>Latest license activations</CardDescription>
                </CardHeader>
                <CardContent>
                  <div className="space-y-4">
                    {licenses.slice(0, 5).map((license: any) => (
                      <div key={license.id} className="flex items-center">
                        <div className="ml-4 space-y-1">
                          <p className="text-sm font-medium leading-none">{license.key_value.substring(0, 14)}...</p>
                          <p className="text-sm text-muted-foreground">
                            {license.subscription_tier} • {new Date(license.created_at).toLocaleDateString()}
                          </p>
                        </div>
                        <div className="ml-auto font-medium">
                          <Badge variant={license.status === 'active' ? 'default' : 'secondary'}>
                            {license.status}
                          </Badge>
                        </div>
                      </div>
                    ))}
                  </div>
                </CardContent>
              </Card>

              <Card>
                <CardHeader>
                  <CardTitle>System Status</CardTitle>
                  <CardDescription>Current system health</CardDescription>
                </CardHeader>
                <CardContent>
                  <div className="space-y-4">
                    <div className="flex items-center justify-between">
                      <span className="text-sm font-medium">Database</span>
                      <Badge variant="default">Operational</Badge>
                    </div>
                    <div className="flex items-center justify-between">
                      <span className="text-sm font-medium">API</span>
                      <Badge variant="default">Operational</Badge>
                    </div>
                    <div className="flex items-center justify-between">
                      <span className="text-sm font-medium">Email Service</span>
                      <Badge variant="default">Operational</Badge>
                    </div>
                    <div className="flex items-center justify-between">
                      <span className="text-sm font-medium">Backup</span>
                      <Badge variant="secondary">Scheduled</Badge>
                    </div>
                  </div>
                </CardContent>
              </Card>
            </div>
          </div>
        );
    }
  };

  return (
    <AdminLayout>
      <div className="space-y-6">
        {/* Header */}
        <div className="flex flex-col md:flex-row md:items-center md:justify-between gap-4">
          <div>
            <h1 className="text-2xl md:text-3xl font-bold text-gray-900 dark:text-white">
              {activeTab === 'dashboard' ? 'Admin Dashboard' : 
               activeTab === 'licenses' ? 'License Management' :
               activeTab === 'users' ? 'User Management' :
               activeTab === 'subscribers' ? 'Subscribers' :
               activeTab === 'orders' ? 'Orders Management' :
               activeTab === 'templates' ? 'Email Templates' :
               activeTab === 'settings' ? 'SMTP Settings' : 'Admin Panel'}
            </h1>
            <p className="text-gray-500 dark:text-gray-400">
              {activeTab === 'dashboard' ? 'Overview of your system metrics and recent activity' :
               activeTab === 'licenses' ? 'Manage license keys and subscriptions' :
               activeTab === 'users' ? 'Manage user accounts and permissions' :
               activeTab === 'subscribers' ? 'View and manage newsletter subscribers' :
               activeTab === 'orders' ? 'Track customer orders and payments' :
               activeTab === 'templates' ? 'Create and manage email templates' :
               activeTab === 'settings' ? 'Configure SMTP and email settings' : 'Admin panel overview'}
            </p>
          </div>
          
          <div className="flex items-center space-x-2">
            <Button variant="outline" size="sm">
              <Settings className="h-4 w-4 mr-2" />
              Settings
            </Button>
            <Button size="sm">
              <Shield className="h-4 w-4 mr-2" />
              Security
            </Button>
          </div>
        </div>

        {/* Tab Navigation for Mobile */}
        <div className="md:hidden">
          <div className="flex space-x-2 overflow-x-auto pb-2">
            {[
              { id: 'dashboard', label: 'Dashboard' },
              { id: 'licenses', label: 'Licenses' },
              { id: 'users', label: 'Users' },
              { id: 'subscribers', label: 'Subscribers' },
              { id: 'orders', label: 'Orders' },
              { id: 'templates', label: 'Templates' },
              { id: 'settings', label: 'Settings' }
            ].map((tab) => (
              <Button
                key={tab.id}
                variant={activeTab === tab.id ? "default" : "outline"}
                size="sm"
                onClick={() => setActiveTab(tab.id)}
                className="whitespace-nowrap"
              >
                {tab.label}
              </Button>
            ))}
          </div>
        </div>

        {/* Main Content */}
        {renderTabContent()}
      </div>
    </AdminLayout>
  );
};

export default AdminDashboard;