import React, { useState, useEffect } from "react";
import Layout from "@/components/Layout";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import {
    Database,
    Download,
    Trash2,
    Plus,
    RefreshCw,
    FileJson,
    AlertCircle,
    CheckCircle2,
    Clock
} from "lucide-react";
import { apiService } from "@/services/api";
import { useToast } from "@/hooks/use-toast";
import { format } from "date-fns";

interface BackupFile {
    filename: string;
    size: number;
    created_at: string;
}

const AdminBackupPage = () => {
    const [backups, setBackups] = useState<BackupFile[]>([]);
    const [loading, setLoading] = useState(true);
    const [creating, setCreating] = useState(false);
    const { toast } = useToast();

    const fetchBackups = async () => {
        setLoading(true);
        try {
            const response = await apiService.listBackups();
            if (response.error) throw new Error(response.error);
            setBackups(response.data || []);
        } catch (error) {
            toast({
                title: "Error",
                description: error instanceof Error ? error.message : "Failed to load backups",
                variant: "destructive",
            });
        } finally {
            setLoading(false);
        }
    };

    useEffect(() => {
        fetchBackups();
    }, []);

    const handleCreateBackup = async () => {
        setCreating(true);
        try {
            const response = await apiService.createBackup();
            if (response.error) throw new Error(response.error);
            toast({
                title: "Success",
                description: "Database backup created successfully",
            });
            fetchBackups();
        } catch (error) {
            toast({
                title: "Error",
                description: error instanceof Error ? error.message : "Backup failed",
                variant: "destructive",
            });
        } finally {
            setCreating(false);
        }
    };

    const handleDeleteBackup = async (filename: string) => {
        if (!confirm(`Are you sure you want to delete backup ${filename}?`)) return;

        try {
            const response = await apiService.deleteBackup(filename);
            if (response.error) throw new Error(response.error);
            toast({
                title: "Deleted",
                description: "Backup archive removed",
            });
            fetchBackups();
        } catch (error) {
            toast({
                title: "Error",
                description: error instanceof Error ? error.message : "Deletion failed",
                variant: "destructive",
            });
        }
    };

    const formatSize = (bytes: number) => {
        if (bytes === 0) return '0 Bytes';
        const k = 1024;
        const sizes = ['Bytes', 'KB', 'MB', 'GB'];
        const i = Math.floor(Math.log(bytes) / Math.log(k));
        return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
    };

    return (
        <Layout>
            <div className="container mx-auto p-6 space-y-6">
                <div className="flex flex-col md:flex-row md:items-center justify-between gap-4">
                    <div>
                        <h1 className="text-3xl font-bold flex items-center gap-2">
                            <Database className="w-8 h-8 text-flashcore-purple" />
                            Backup & Recovery
                        </h1>
                        <p className="text-muted-foreground mt-1">
                            Manage database snapshots and system backups for disaster recovery.
                        </p>
                    </div>
                    <div className="flex items-center gap-2">
                        <Button variant="outline" onClick={fetchBackups} disabled={loading}>
                            <RefreshCw className={`w-4 h-4 mr-2 ${loading ? 'animate-spin' : ''}`} />
                            Refresh
                        </Button>
                        <Button onClick={handleCreateBackup} disabled={creating}>
                            {creating ? (
                                <>
                                    <RefreshCw className="w-4 h-4 mr-2 animate-spin" />
                                    Creating...
                                </>
                            ) : (
                                <>
                                    <Plus className="w-4 h-4 mr-2" />
                                    Create Snapshot
                                </>
                            )}
                        </Button>
                    </div>
                </div>

                <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                    <Card className="md:col-span-2">
                        <CardHeader>
                            <CardTitle>Backup History</CardTitle>
                            <CardDescription>A list of all available database snapshots.</CardDescription>
                        </CardHeader>
                        <CardContent>
                            {loading ? (
                                <div className="flex justify-center py-12">
                                    <RefreshCw className="w-8 h-8 animate-spin text-flashcore-purple" />
                                </div>
                            ) : backups.length === 0 ? (
                                <div className="text-center py-12 border-2 border-dashed rounded-xl">
                                    <Database className="w-12 h-12 text-muted-foreground mx-auto mb-4 opacity-20" />
                                    <p className="text-muted-foreground">No backups found.</p>
                                    <Button variant="link" onClick={handleCreateBackup} className="mt-2 text-flashcore-purple">
                                        Create your first backup now
                                    </Button>
                                </div>
                            ) : (
                                <div className="overflow-x-auto">
                                    <table className="w-full">
                                        <thead>
                                            <tr className="border-b text-sm font-medium text-muted-foreground">
                                                <th className="text-left py-3 px-4">Filename</th>
                                                <th className="text-left py-3 px-4">Date Created</th>
                                                <th className="text-left py-3 px-4">Size</th>
                                                <th className="text-right py-3 px-4">Actions</th>
                                            </tr>
                                        </thead>
                                        <tbody className="divide-y">
                                            {backups.map((backup) => (
                                                <tr key={backup.filename} className="hover:bg-muted/50 transition-colors">
                                                    <td className="py-4 px-4 font-mono text-xs">{backup.filename}</td>
                                                    <td className="py-4 px-4 text-sm">
                                                        <span className="flex items-center gap-1">
                                                            <Clock className="w-3 h-3 opacity-50" />
                                                            {backup.created_at}
                                                        </span>
                                                    </td>
                                                    <td className="py-4 px-4 text-sm font-medium">
                                                        {formatSize(backup.size)}
                                                    </td>
                                                    <td className="py-4 px-4 text-right">
                                                        <div className="flex items-center justify-end gap-2">
                                                            {/* Direct download link */}
                                                            <a
                                                                href={`${import.meta.env.DEV ? '' : '/fcorex'}/backend/api/backups.php?action=download&filename=${backup.filename}`}
                                                                target="_blank"
                                                                rel="noopener noreferrer"
                                                                className="inline-flex items-center justify-center rounded-md text-sm font-medium ring-offset-background transition-colors focus-visible:outline-none focus-visible:ring-2 focus-visible:ring-ring focus-visible:ring-offset-2 disabled:pointer-events-none disabled:opacity-50 border border-input bg-background hover:bg-accent hover:text-accent-foreground h-9 w-9"
                                                                title="Download"
                                                            >
                                                                <Download className="h-4 w-4" />
                                                            </a>
                                                            <Button
                                                                variant="outline"
                                                                size="icon"
                                                                className="text-destructive hover:bg-destructive/10"
                                                                onClick={() => handleDeleteBackup(backup.filename)}
                                                                title="Delete"
                                                            >
                                                                <Trash2 className="h-4 w-4" />
                                                            </Button>
                                                        </div>
                                                    </td>
                                                </tr>
                                            ))}
                                        </tbody>
                                    </table>
                                </div>
                            )}
                        </CardContent>
                    </Card>

                    <div className="space-y-6">
                        <Card>
                            <CardHeader>
                                <CardTitle className="text-lg">Storage Health</CardTitle>
                            </CardHeader>
                            <CardContent className="space-y-4">
                                <div className="flex items-center justify-between p-3 bg-green-500/10 rounded-lg border border-green-500/20">
                                    <div className="flex items-center gap-3">
                                        <CheckCircle2 className="w-5 h-5 text-green-500" />
                                        <span className="text-sm font-medium">Permissions</span>
                                    </div>
                                    <span className="text-xs font-bold text-green-500">Writable</span>
                                </div>
                                <div className="flex items-center justify-between p-3 bg-blue-500/10 rounded-lg border border-blue-500/20">
                                    <div className="flex items-center gap-3">
                                        <AlertCircle className="w-5 h-5 text-blue-500" />
                                        <span className="text-sm font-medium">Protection</span>
                                    </div>
                                    <span className="text-xs font-bold text-blue-500">Locked</span>
                                </div>
                                <div className="pt-2">
                                    <p className="text-xs text-muted-foreground leading-relaxed">
                                        Backups are stored in a secured directory protected by .htaccess.
                                        Recommended frequency: **Daily**.
                                    </p>
                                </div>
                            </CardContent>
                        </Card>

                        <Card className="bg-gradient-to-br from-flashcore-purple/10 via-background to-flashcore-green/10 border-flashcore-purple/20">
                            <CardHeader>
                                <CardTitle className="text-lg">Quick Tip</CardTitle>
                            </CardHeader>
                            <CardContent>
                                <p className="text-sm text-muted-foreground italic leading-relaxed">
                                    "Always verify your backups after creation. A backup that cannot be restored
                                    is no backup at all."
                                </p>
                                <div className="mt-4 flex justify-end">
                                    <FileJson className="w-8 h-8 opacity-20" />
                                </div>
                            </CardContent>
                        </Card>
                    </div>
                </div>
            </div>
        </Layout>
    );
};

export default AdminBackupPage;
