import { useEffect, useRef, useCallback } from 'react';

interface AutoSaveOptions<T> {
  data: T;
  onSave: (data: T) => Promise<void> | void;
  delay?: number;
  enabled?: boolean;
  onError?: (error: Error) => void;
  onSuccess?: () => void;
  debounce?: boolean;
}

export function useAutoSave<T>({
  data,
  onSave,
  delay = 2000,
  enabled = true,
  onError,
  onSuccess,
  debounce = true
}: AutoSaveOptions<T>) {
  const timeoutRef = useRef<NodeJS.Timeout>();
  const lastSavedDataRef = useRef<T>();
  const isSavingRef = useRef(false);

  const saveData = useCallback(async () => {
    if (!enabled || isSavingRef.current) return;

    // Check if data has actually changed
    if (JSON.stringify(data) === JSON.stringify(lastSavedDataRef.current)) {
      return;
    }

    isSavingRef.current = true;

    try {
      await onSave(data);
      lastSavedDataRef.current = data;
      onSuccess?.();
    } catch (error) {
      console.error('Auto-save failed:', error);
      onError?.(error instanceof Error ? error : new Error('Auto-save failed'));
    } finally {
      isSavingRef.current = false;
    }
  }, [data, onSave, enabled, onSuccess, onError]);

  useEffect(() => {
    if (!enabled) return;

    if (debounce) {
      // Clear existing timeout
      if (timeoutRef.current) {
        clearTimeout(timeoutRef.current);
      }

      // Set new timeout
      timeoutRef.current = setTimeout(() => {
        saveData();
      }, delay);
    } else {
      // Immediate save
      saveData();
    }

    return () => {
      if (timeoutRef.current) {
        clearTimeout(timeoutRef.current);
      }
    };
  }, [data, delay, enabled, debounce, saveData]);

  // Save on unmount if there are unsaved changes
  useEffect(() => {
    return () => {
      if (enabled && JSON.stringify(data) !== JSON.stringify(lastSavedDataRef.current)) {
        saveData();
      }
    };
  }, [enabled, data, saveData]);

  return {
    isSaving: isSavingRef.current,
    lastSaved: lastSavedDataRef.current,
    hasUnsavedChanges: JSON.stringify(data) !== JSON.stringify(lastSavedDataRef.current),
  };
}

// Form-specific auto-save hook
export function useFormAutoSave<T extends Record<string, any>>(
  formData: T,
  saveFunction: (data: T) => Promise<void>,
  options?: Partial<AutoSaveOptions<T>>
) {
  const { isSaving, hasUnsavedChanges } = useAutoSave({
    data: formData,
    onSave: saveFunction,
    delay: 1000,
    enabled: true,
    debounce: true,
    ...options,
  });

  return {
    isSaving,
    hasUnsavedChanges,
    saveIndicator: hasUnsavedChanges ? 'Unsaved changes' : 'All changes saved',
  };
}

// Local storage auto-save hook
export function useLocalStorageAutoSave<T>(
  key: string,
  data: T,
  options?: Partial<AutoSaveOptions<T>>
) {
  const saveToStorage = useCallback(async (dataToSave: T) => {
    try {
      localStorage.setItem(key, JSON.stringify(dataToSave));
    } catch (error) {
      console.error('Failed to save to localStorage:', error);
      throw error;
    }
  }, [key]);

  return useAutoSave({
    data,
    onSave: saveToStorage,
    delay: 500,
    enabled: true,
    debounce: true,
    ...options,
  });
}

// Session storage auto-save hook
export function useSessionStorageAutoSave<T>(
  key: string,
  data: T,
  options?: Partial<AutoSaveOptions<T>>
) {
  const saveToStorage = useCallback(async (dataToSave: T) => {
    try {
      sessionStorage.setItem(key, JSON.stringify(dataToSave));
    } catch (error) {
      console.error('Failed to save to sessionStorage:', error);
      throw error;
    }
  }, [key]);

  return useAutoSave({
    data,
    onSave: saveToStorage,
    delay: 500,
    enabled: true,
    debounce: true,
    ...options,
  });
} 