import React, { createContext, useState, useEffect, useContext } from 'react';
import { apiService } from '@/services/api';

interface UserProfile {
    id: string;
    role: 'admin' | 'user';
    first_name: string;
    last_name: string;
}

interface AuthContextType {
    user: any | null;
    profile: UserProfile | null;
    loading: boolean;
    token: string | null;
    isAdmin: boolean;
    isUser: boolean;
    login: (email: string, password: string) => Promise<{ success: boolean; user?: any; error?: string }>;
    register: (email: string, password: string, firstName?: string, lastName?: string) => Promise<{ success: boolean; user?: any; error?: string }>;
    logout: () => Promise<void>;
}

const AuthContext = createContext<AuthContextType | undefined>(undefined);

export const AuthProvider: React.FC<{ children: React.ReactNode }> = ({ children }) => {
    const [user, setUser] = useState<any | null>(null);
    const [profile, setProfile] = useState<UserProfile | null>(null);
    const [loading, setLoading] = useState(true);
    const [token, setToken] = useState<string | null>(null);

    const logout = async () => {
        try {
            await apiService.logout();
        } catch (error) {
            console.error('Logout error:', error);
        } finally {
            setUser(null);
            setProfile(null);
            setToken(null);
            apiService.setToken(null);
            localStorage.removeItem('authToken');
            localStorage.removeItem('authUser');
            localStorage.removeItem('authProfile');
        }
    };

    const initAuth = async () => {
        const savedToken = localStorage.getItem('authToken');
        const savedUser = localStorage.getItem('authUser');
        const savedProfile = localStorage.getItem('authProfile');

        if (savedToken && savedUser) {
            try {
                setToken(savedToken);
                apiService.setToken(savedToken);
                const response = await apiService.getSubscriptionStatus();

                if (response.error) {
                    console.warn('Session validation failed:', response.error);
                    await logout();
                } else {
                    setUser(JSON.parse(savedUser));
                    if (savedProfile) {
                        setProfile(JSON.parse(savedProfile));
                    }
                }
            } catch (error) {
                console.error('Auth initialization error:', error);
                await logout();
            }
        }
        setLoading(false);
    };

    useEffect(() => {
        initAuth();
    }, []);

    const login = async (email: string, password: string) => {
        setLoading(true);
        try {
            const response = await apiService.login(email, password);
            if (response.error) throw new Error(response.error);

            if (response.data?.user && response.data?.session_token) {
                const userData = response.data.user;
                const profileData = response.data.profile;
                const userToken = response.data.session_token;

                const profileObj: UserProfile = {
                    id: userData.id,
                    role: profileData?.role === 'admin' ? 'admin' : 'user',
                    first_name: userData.first_name || '',
                    last_name: userData.last_name || ''
                };

                setUser(userData);
                setProfile(profileObj);
                setToken(userToken);
                apiService.setToken(userToken);

                localStorage.setItem('authToken', userToken);
                localStorage.setItem('authUser', JSON.stringify(userData));
                localStorage.setItem('authProfile', JSON.stringify(profileObj));

                return { success: true, user: userData };
            }
            throw new Error('Login failed');
        } catch (error) {
            return { success: false, error: error instanceof Error ? error.message : 'Login failed' };
        } finally {
            setLoading(false);
        }
    };

    const register = async (email: string, password: string, firstName?: string, lastName?: string) => {
        setLoading(true);
        try {
            const response = await apiService.register(email, password, firstName, lastName);
            if (response.error) throw new Error(response.error);

            if (response.data?.user && response.data?.session_token) {
                const userData = response.data.user;
                const profileData = response.data.profile;
                const userToken = response.data.session_token;

                const profileObj: UserProfile = {
                    id: userData.id,
                    role: profileData?.role === 'admin' ? 'admin' : 'user',
                    first_name: userData.first_name || '',
                    last_name: userData.last_name || ''
                };

                setUser(userData);
                setProfile(profileObj);
                setToken(userToken);
                apiService.setToken(userToken);

                localStorage.setItem('authToken', userToken);
                localStorage.setItem('authUser', JSON.stringify(userData));
                localStorage.setItem('authProfile', JSON.stringify(profileObj));

                return { success: true, user: userData };
            }
            throw new Error('Registration failed');
        } catch (error) {
            return { success: false, error: error instanceof Error ? error.message : 'Registration failed' };
        } finally {
            setLoading(false);
        }
    };

    const isAdmin = profile?.role === 'admin';
    const isUser = profile?.role === 'user' || !profile;

    return (
        <AuthContext.Provider value={{ user, profile, loading, token, isAdmin, isUser, login, register, logout }}>
            {children}
        </AuthContext.Provider>
    );
};

export const useAuthContext = () => {
    const context = useContext(AuthContext);
    if (context === undefined) {
        throw new Error('useAuthContext must be used within an AuthProvider');
    }
    return context;
};
