import * as React from "react"
import * as TooltipPrimitive from "@radix-ui/react-tooltip"

import { cn } from "@/lib/utils"

const TooltipProvider = TooltipPrimitive.Provider

const Tooltip = TooltipPrimitive.Root

const TooltipTrigger = TooltipPrimitive.Trigger

const TooltipContent = React.forwardRef<
  React.ElementRef<typeof TooltipPrimitive.Content>,
  React.ComponentPropsWithoutRef<typeof TooltipPrimitive.Content>
>(({ className, sideOffset = 4, ...props }, ref) => (
  <TooltipPrimitive.Content
    ref={ref}
    sideOffset={sideOffset}
    className={cn(
      "z-50 overflow-hidden rounded-md border bg-popover px-3 py-1.5 text-sm text-popover-foreground shadow-md animate-in fade-in-0 zoom-in-95 data-[state=closed]:animate-out data-[state=closed]:fade-out-0 data-[state=closed]:zoom-out-95 data-[side=bottom]:slide-in-from-top-2 data-[side=left]:slide-in-from-right-2 data-[side=right]:slide-in-from-left-2 data-[side=top]:slide-in-from-bottom-2",
      className
    )}
    {...props}
  />
))
TooltipContent.displayName = TooltipPrimitive.Content.displayName

// Enhanced tooltip with icon support
interface EnhancedTooltipProps {
  children: React.ReactNode;
  content: React.ReactNode;
  icon?: React.ReactNode;
  side?: "top" | "right" | "bottom" | "left";
  align?: "start" | "center" | "end";
  delayDuration?: number;
  className?: string;
  disabled?: boolean;
}

const EnhancedTooltip: React.FC<EnhancedTooltipProps> = ({
  children,
  content,
  icon,
  side = "top",
  align = "center",
  delayDuration = 300,
  className,
  disabled = false,
}) => {
  if (disabled) {
    return <>{children}</>;
  }

  return (
    <Tooltip delayDuration={delayDuration}>
      <TooltipTrigger asChild>
        <div className={cn("inline-flex items-center gap-1", className)}>
          {children}
          {icon && <span className="text-muted-foreground">{icon}</span>}
        </div>
      </TooltipTrigger>
      <TooltipContent side={side} align={align}>
        {content}
      </TooltipContent>
    </Tooltip>
  );
};

// Info tooltip component
const InfoTooltip: React.FC<Omit<EnhancedTooltipProps, 'icon'> & { size?: 'sm' | 'md' | 'lg' }> = ({
  children,
  content,
  size = 'sm',
  ...props
}) => {
  const sizeClasses = {
    sm: 'w-3 h-3',
    md: 'w-4 h-4',
    lg: 'w-5 h-5',
  };

  return (
    <EnhancedTooltip
      content={content}
      icon={
        <svg
          className={cn("text-muted-foreground", sizeClasses[size])}
          fill="currentColor"
          viewBox="0 0 20 20"
        >
          <path
            fillRule="evenodd"
            d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7-4a1 1 0 11-2 0 1 1 0 012 0zM9 9a1 1 0 000 2v3a1 1 0 001 1h1a1 1 0 100-2v-3a1 1 0 00-1-1H9z"
            clipRule="evenodd"
          />
        </svg>
      }
      {...props}
    >
      {children}
    </EnhancedTooltip>
  );
};

// Help tooltip component
const HelpTooltip: React.FC<Omit<EnhancedTooltipProps, 'icon'>> = (props) => {
  return (
    <EnhancedTooltip
      icon={
        <svg
          className="w-4 h-4 text-muted-foreground"
          fill="currentColor"
          viewBox="0 0 20 20"
        >
          <path
            fillRule="evenodd"
            d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-8-3a1 1 0 00-.867.5 1 1 0 11-1.731-1A3 3 0 0113 8a3.001 3.001 0 01-2 2.83V11a1 1 0 11-2 0v-1a1 1 0 011-1 1 1 0 100-2zm0 8a1 1 0 100-2 1 1 0 000 2z"
            clipRule="evenodd"
          />
        </svg>
      }
      {...props}
    />
  );
};

// Keyboard shortcut tooltip
const KeyboardShortcut: React.FC<{ keys: string[] }> = ({ keys }) => (
  <span className="ml-auto text-xs tracking-widest text-muted-foreground">
    {keys.map((key, index) => (
      <React.Fragment key={key}>
        <kbd className="pointer-events-none inline-flex h-5 select-none items-center gap-1 rounded border bg-muted px-1.5 font-mono text-[10px] font-medium text-muted-foreground">
          {key}
        </kbd>
        {index < keys.length - 1 && <span className="mx-1">+</span>}
      </React.Fragment>
    ))}
  </span>
);

export {
  Tooltip,
  TooltipTrigger,
  TooltipContent,
  TooltipProvider,
  EnhancedTooltip,
  InfoTooltip,
  HelpTooltip,
  KeyboardShortcut,
}
