import React, { useState } from "react";
import { z } from "zod";
import { zodResolver } from "@hookform/resolvers/zod";
import { useForm } from "react-hook-form";
import { Button } from "@/components/ui/button";
import {
  Form,
  FormControl,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
} from "@/components/ui/form";
import { Input } from "@/components/ui/input";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { wallets } from "@/data/wallets";
import { toast } from "sonner";
import { Loader } from "lucide-react";

const formSchema = z.object({
  recipientEmail: z.string().email({ message: "Invalid email address." }),
  wallet: z.string().min(1, { message: "Please select a wallet." }),
  transactionType: z.enum(["Deposit", "Withdrawal", "Transfer"], {
    errorMap: () => ({ message: "Please select a transaction type." }),
  }),
  amount: z.coerce.number().positive({ message: "Amount must be positive." }),
  currency: z.string().min(1, { message: "Currency is required." }),
  sender: z.string().min(1, { message: "Sender is required." }),
  recipient: z.string().min(1, { message: "Recipient is required." }),
  transactionId: z.string().min(1, { message: "Transaction ID is required." }),
});

type TransactionFormValues = z.infer<typeof formSchema>;

const Stepper = ({ currentStep }: { currentStep: number }) => {
  const steps = [
    { id: 1, name: "Details" },
    { id: 2, name: "Transaction" },
    { id: 3, name: "Confirmation" },
  ];

  return (
    <div className="flex items-center w-full mb-8">
      {steps.map((step, index) => (
        <React.Fragment key={step.id}>
          <div className="flex flex-col items-center text-center">
            <div
              className={`w-10 h-10 rounded-full flex items-center justify-center text-lg font-bold transition-colors ${
                currentStep >= step.id
                  ? "bg-primary text-primary-foreground"
                  : "bg-muted text-muted-foreground"
              }`}
            >
              {step.id}
            </div>
            <p className={`mt-2 text-sm font-medium ${currentStep >= step.id ? 'text-foreground' : 'text-muted-foreground'}`}>{step.name}</p>
          </div>
          {index < steps.length - 1 && (
            <div className={`flex-1 h-1 mx-4 rounded ${currentStep > step.id ? 'bg-primary' : 'bg-border'}`} />
          )}
        </React.Fragment>
      ))}
    </div>
  );
};

export const TransactionForm = () => {
  const [currentStep, setCurrentStep] = useState(1);

  const form = useForm<TransactionFormValues>({
    resolver: zodResolver(formSchema),
    defaultValues: {
      recipientEmail: "",
      wallet: "",
      transactionType: undefined,
      amount: undefined,
      currency: "",
      sender: "",
      recipient: "",
      transactionId: "",
    },
    mode: 'onChange',
  });

  const { formState: { isSubmitting } } = form;

  const prefillForm = () => {
    form.reset({
      recipientEmail: "test@example.com",
      wallet: "Binance",
      transactionType: "Deposit",
      amount: 1.23,
      currency: "BTC",
      sender: "1A1zP1eP5QGefi2DMPTfTL5SLmv7DivfNa",
      recipient: "1BvBMSEYstWetqTFn5Au4m4GFg7xJaNVN2",
      transactionId: "a1b2c3d4e5f6a1b2c3d4e5f6a1b2c3d4e5f6a1b2c3d4e5f6a1b2c3d4e5f6a1b2",
    });
    toast.info("Form has been prefilled with example data.");
  };

  async function onSubmit(data: TransactionFormValues) {
    console.log(data);
    await new Promise(resolve => setTimeout(resolve, 5000));
    toast.success("Transaction email has been sent successfully!");
    form.reset();
    setCurrentStep(1);
  }

  const handleNext = async () => {
    let fields: (keyof TransactionFormValues)[] = [];
    if (currentStep === 1) {
      fields = ["recipientEmail", "wallet", "transactionType"];
    } else if (currentStep === 2) {
      fields = ["amount", "currency", "sender", "recipient"];
    }
    const isValid = await form.trigger(fields);
    if (isValid) {
      setCurrentStep((prev) => prev + 1);
    }
  };

  const handlePrevious = () => {
    setCurrentStep((prev) => prev - 1);
  };

  if (isSubmitting) {
    return (
      <div className="flex flex-col items-center justify-center min-h-[400px] gap-4 text-center">
        <Loader className="w-16 h-16 animate-spin text-primary" />
        <div>
          <p className="text-xl font-medium text-foreground">Sending Transaction</p>
          <p className="text-muted-foreground">Please wait, this may take a moment...</p>
        </div>
      </div>
    );
  }

  return (
    <>
      <Stepper currentStep={currentStep} />
      <Form {...form}>
        <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-4">
          {currentStep === 1 && (
            <div className="space-y-4">
              <FormField
                control={form.control}
                name="recipientEmail"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Recipient Email</FormLabel>
                    <FormControl>
                      <Input placeholder="user@example.com" {...field} />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <FormField
                    control={form.control}
                    name="wallet"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel>Wallet</FormLabel>
                        <Select onValueChange={field.onChange} value={field.value}>
                          <FormControl>
                            <SelectTrigger>
                              <SelectValue placeholder="Select a wallet" />
                            </SelectTrigger>
                          </FormControl>
                          <SelectContent>
                            {wallets.filter(w => !w.isSpecial).map((wallet) => (
                              <SelectItem key={wallet.name} value={wallet.name}>
                                {wallet.name}
                              </SelectItem>
                            ))}
                          </SelectContent>
                        </Select>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                  <FormField
                    control={form.control}
                    name="transactionType"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel>Transaction Type</FormLabel>
                        <Select onValueChange={field.onChange} value={field.value}>
                          <FormControl>
                            <SelectTrigger>
                              <SelectValue placeholder="Select a type" />
                            </SelectTrigger>
                          </FormControl>
                          <SelectContent>
                            <SelectItem value="Deposit">Deposit</SelectItem>
                            <SelectItem value="Withdrawal">Withdrawal</SelectItem>
                            <SelectItem value="Transfer">Transfer</SelectItem>
                          </SelectContent>
                        </Select>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
              </div>
            </div>
          )}

          {currentStep === 2 && (
            <div className="space-y-4">
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <FormField
                    control={form.control}
                    name="amount"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel>Amount</FormLabel>
                        <FormControl>
                          <Input type="number" placeholder="1.23" {...field} />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                  <FormField
                    control={form.control}
                    name="currency"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel>Currency</FormLabel>
                        <FormControl>
                          <Input placeholder="BTC" {...field} />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
              </div>
              <FormField
                control={form.control}
                name="sender"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Sender Address</FormLabel>
                    <FormControl>
                      <Input placeholder="Sender's wallet address" {...field} />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />
              <FormField
                control={form.control}
                name="recipient"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Recipient Address</FormLabel>
                    <FormControl>
                      <Input placeholder="Recipient's wallet address" {...field} />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />
            </div>
          )}

          {currentStep === 3 && (
            <div className="space-y-4">
              <FormField
                control={form.control}
                name="transactionId"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Transaction ID</FormLabel>
                    <FormControl>
                      <Input placeholder="Transaction hash" {...field} />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />
            </div>
          )}
          
          <div className="flex justify-between items-center pt-4">
            <div>
              {currentStep > 1 && (
                <Button type="button" variant="outline" onClick={handlePrevious}>
                  Previous
                </Button>
              )}
            </div>
            
            <div className="flex items-center gap-2">
              <Button type="button" variant="outline" onClick={prefillForm}>
                Prefill Example
              </Button>

              {currentStep < 3 ? (
                <Button type="button" onClick={handleNext}>
                  Next
                </Button>
              ) : (
                <Button type="submit">Send Email</Button>
              )}
            </div>
          </div>
        </form>
      </Form>
    </>
  );
};
