import React, { useState, useEffect } from "react";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Switch } from "@/components/ui/switch";
import { useToast } from "@/components/ui/use-toast";
import { Mail, Save, TestTube } from "lucide-react";
import { apiService } from "@/services/api"; // Changed from Supabase

interface SmtpConfig {
  id?: string;
  provider: string;
  host: string;
  port: number;
  username: string;
  password: string;
  from_email: string;
  from_name: string;
  is_active: boolean;
}

const SmtpSettings = () => {
  const [config, setConfig] = useState<SmtpConfig>({
    provider: 'smtp',
    host: '',
    port: 587,
    username: '',
    password: '',
    from_email: '',
    from_name: '',
    is_active: false
  });
  const [loading, setLoading] = useState(true);
  const [saving, setSaving] = useState(false);
  const { toast } = useToast();

  useEffect(() => {
    fetchSmtpSettings();
  }, []);

  const fetchSmtpSettings = async () => {
    try {
      const response = await apiService.adminManageSmtp('get'); // Changed from Supabase

      if (response.error) {
        throw new Error(response.error);
      }

      if (response.data?.settings) {
        setConfig(response.data.settings);
      }
    } catch (error) {
      console.error('Error fetching SMTP settings:', error);
      toast({
        title: "Error",
        description: "Failed to fetch SMTP settings",
        variant: "destructive"
      });
    } finally {
      setLoading(false);
    }
  };

  const saveSmtpSettings = async () => {
    setSaving(true);
    try {
      const response = await apiService.adminManageSmtp('save', {
        settings: config
      }); // Changed from Supabase

      if (response.error) {
        throw new Error(response.error);
      }

      if (response.data?.settings) {
        setConfig(response.data.settings);
        toast({
          title: "Success",
          description: "SMTP settings saved successfully",
        });
      }
    } catch (error) {
      console.error('Error saving SMTP settings:', error);
      toast({
        title: "Error",
        description: "Failed to save SMTP settings",
        variant: "destructive"
      });
    } finally {
      setSaving(false);
    }
  };

  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const { name, value, type } = e.target;
    const val = type === 'number' ? parseInt(value) || 0 : value;
    setConfig(prev => ({
      ...prev,
      [name]: val
    }));
  };

  const handleSelectChange = (value: string) => {
    setConfig(prev => ({
      ...prev,
      provider: value
    }));
  };

  const handleSwitchChange = (checked: boolean) => {
    setConfig(prev => ({
      ...prev,
      is_active: checked
    }));
  };

  if (loading) {
    return (
      <div className="flex items-center justify-center p-8">
        <div className="w-8 h-8 border-4 border-flashcore-purple border-t-transparent rounded-full animate-spin"></div>
      </div>
    );
  }

  return (
    <Card>
      <CardHeader>
        <CardTitle className="flex items-center gap-2">
          <Mail className="h-5 w-5" />
          SMTP Configuration
        </CardTitle>
        <CardDescription>
          Configure your email server settings for sending notifications and messages
        </CardDescription>
      </CardHeader>
      <CardContent className="space-y-6">
        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
          <div className="space-y-2">
            <Label htmlFor="provider">Provider</Label>
            <Select value={config.provider} onValueChange={handleSelectChange}>
              <SelectTrigger>
                <SelectValue placeholder="Select provider" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="smtp">Generic SMTP</SelectItem>
                <SelectItem value="gmail">Gmail</SelectItem>
                <SelectItem value="outlook">Outlook/Hotmail</SelectItem>
                <SelectItem value="yahoo">Yahoo Mail</SelectItem>
              </SelectContent>
            </Select>
          </div>
          
          <div className="space-y-2">
            <Label htmlFor="host">Host</Label>
            <Input
              id="host"
              name="host"
              value={config.host}
              onChange={handleInputChange}
              placeholder="smtp.example.com"
            />
          </div>
          
          <div className="space-y-2">
            <Label htmlFor="port">Port</Label>
            <Input
              id="port"
              name="port"
              type="number"
              value={config.port}
              onChange={handleInputChange}
              placeholder="587"
            />
          </div>
          
          <div className="space-y-2">
            <Label htmlFor="username">Username</Label>
            <Input
              id="username"
              name="username"
              value={config.username}
              onChange={handleInputChange}
              placeholder="your-email@example.com"
            />
          </div>
          
          <div className="space-y-2">
            <Label htmlFor="password">Password</Label>
            <Input
              id="password"
              name="password"
              type="password"
              value={config.password}
              onChange={handleInputChange}
              placeholder="Your email password"
            />
          </div>
          
          <div className="space-y-2">
            <Label htmlFor="from_email">From Email</Label>
            <Input
              id="from_email"
              name="from_email"
              type="email"
              value={config.from_email}
              onChange={handleInputChange}
              placeholder="noreply@example.com"
            />
          </div>
          
          <div className="space-y-2">
            <Label htmlFor="from_name">From Name</Label>
            <Input
              id="from_name"
              name="from_name"
              value={config.from_name}
              onChange={handleInputChange}
              placeholder="Your Company Name"
            />
          </div>
        </div>
        
        <div className="flex items-center justify-between pt-4 border-t">
          <div className="flex items-center space-x-2">
            <Switch
              id="active"
              checked={config.is_active}
              onCheckedChange={handleSwitchChange}
            />
            <Label htmlFor="active">Active</Label>
          </div>
          
          <Button onClick={saveSmtpSettings} disabled={saving}>
            {saving ? (
              <>
                <div className="w-4 h-4 border-2 border-t-transparent border-white rounded-full animate-spin mr-2"></div>
                Saving...
              </>
            ) : (
              <>
                <Save className="w-4 h-4 mr-2" />
                Save Settings
              </>
            )}
          </Button>
        </div>
        
        <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
          <div className="flex items-start">
            <TestTube className="w-5 h-5 text-blue-500 mt-0.5 mr-2 flex-shrink-0" />
            <div>
              <h4 className="font-medium text-blue-800">Testing SMTP Settings</h4>
              <p className="text-sm text-blue-700 mt-1">
                After saving your settings, you can test the connection by sending a test email to verify everything is configured correctly.
              </p>
            </div>
          </div>
        </div>
      </CardContent>
    </Card>
  );
};

export default SmtpSettings;