import React, { useState } from "react";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Badge } from "@/components/ui/badge";
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { Label } from "@/components/ui/label";
import { useToast } from "@/components/ui/use-toast";
import { apiService } from "@/services/api";
import { Key, Plus, Search, Filter } from "lucide-react";

interface LicenseManagementProps {
  licenses: any[];
  onRefresh: () => void;
}

const LicenseManagement: React.FC<LicenseManagementProps> = ({ licenses, onRefresh }) => {
  const [createDialogOpen, setCreateDialogOpen] = useState(false);
  const [newLicenseData, setNewLicenseData] = useState({
    subscription_tier: "basic",
    quantity: 1
  });
  const { toast } = useToast();

  const createLicenses = async () => {
    try {
      const response = await apiService.adminManageLicenses('create', {
        subscription_tier: newLicenseData.subscription_tier,
        quantity: newLicenseData.quantity
      });

      if (response.error) throw new Error(response.error);

      toast({
        title: "Success",
        description: `Created ${newLicenseData.quantity} license key(s)`,
      });

      setCreateDialogOpen(false);
      setNewLicenseData({
        subscription_tier: "basic",
        quantity: 1
      });
      onRefresh();
    } catch (error) {
      toast({
        title: "Error",
        description: "Failed to create license keys",
        variant: "destructive"
      });
    }
  };

  const updateLicenseStatus = async (id: string, status: string) => {
    try {
      const response = await apiService.adminManageLicenses('update', {
        id,
        status
      });

      if (response.error) throw new Error(response.error);

      toast({
        title: "Success",
        description: "License status updated",
      });

      onRefresh();
    } catch (error) {
      toast({
        title: "Error",
        description: "Failed to update license status",
        variant: "destructive"
      });
    }
  };

  const getStatusVariant = (status: string) => {
    switch (status.toLowerCase()) {
      case 'active': return 'default';
      case 'pending': return 'secondary';
      case 'expired': return 'destructive';
      case 'suspended': return 'outline';
      default: return 'secondary';
    }
  };

  return (
    <Card className="border-0 shadow-lg">
      <CardHeader className="bg-white dark:bg-gray-800 border-b border-gray-100 dark:border-gray-700">
        <div className="flex flex-col md:flex-row md:items-center md:justify-between gap-4">
          <div>
            <CardTitle className="flex items-center gap-2">
              <Key className="h-5 w-5 text-blue-500" />
              License Management
            </CardTitle>
            <CardDescription>
              Create and manage software license keys
            </CardDescription>
          </div>
          <div className="flex flex-col sm:flex-row gap-2">
            <div className="relative">
              <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 h-4 w-4 text-gray-400" />
              <Input
                placeholder="Search licenses..."
                className="pl-10 w-full md:w-64"
              />
            </div>
            <Button variant="outline" size="sm">
              <Filter className="h-4 w-4 mr-2" />
              Filter
            </Button>
            <Dialog open={createDialogOpen} onOpenChange={setCreateDialogOpen}>
              <DialogTrigger asChild>
                <Button size="sm">
                  <Plus className="h-4 w-4 mr-2" />
                  Create Licenses
                </Button>
              </DialogTrigger>
              <DialogContent>
                <DialogHeader>
                  <DialogTitle>Create License Keys</DialogTitle>
                  <DialogDescription>
                    Generate new license keys for users
                  </DialogDescription>
                </DialogHeader>
                <div className="grid gap-4 py-4">
                  <div className="grid grid-cols-4 items-center gap-4">
                    <Label htmlFor="tier" className="text-right">Tier</Label>
                    <Select 
                      value={newLicenseData.subscription_tier} 
                      onValueChange={(value) => setNewLicenseData({...newLicenseData, subscription_tier: value})}
                    >
                      <SelectTrigger className="col-span-3">
                        <SelectValue />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="basic">Basic</SelectItem>
                        <SelectItem value="pro">Pro</SelectItem>
                        <SelectItem value="enterprise">Enterprise</SelectItem>
                      </SelectContent>
                    </Select>
                  </div>
                  <div className="grid grid-cols-4 items-center gap-4">
                    <Label htmlFor="quantity" className="text-right">Quantity</Label>
                    <Input
                      id="quantity"
                      type="number"
                      min="1"
                      max="100"
                      value={newLicenseData.quantity}
                      onChange={(e) => setNewLicenseData({...newLicenseData, quantity: parseInt(e.target.value) || 1})}
                      className="col-span-3"
                    />
                  </div>
                </div>
                <DialogFooter>
                  <Button variant="outline" onClick={() => setCreateDialogOpen(false)}>
                    Cancel
                  </Button>
                  <Button onClick={createLicenses}>
                    Create Licenses
                  </Button>
                </DialogFooter>
              </DialogContent>
            </Dialog>
          </div>
        </div>
      </CardHeader>
      <CardContent className="p-0">
        <Table>
          <TableHeader>
            <TableRow className="bg-gray-50 dark:bg-gray-700">
              <TableHead className="font-semibold">License Key</TableHead>
              <TableHead className="font-semibold">Tier</TableHead>
              <TableHead className="font-semibold">Status</TableHead>
              <TableHead className="font-semibold">Created</TableHead>
              <TableHead className="font-semibold text-right">Actions</TableHead>
            </TableRow>
          </TableHeader>
          <TableBody>
            {licenses.length === 0 ? (
              <TableRow>
                <TableCell colSpan={5} className="text-center py-8 text-gray-500">
                  No licenses found. Create your first license key to get started.
                </TableCell>
              </TableRow>
            ) : (
              licenses.map((license: any) => (
                <TableRow key={license.id} className="border-b border-gray-100 dark:border-gray-700 hover:bg-gray-50 dark:hover:bg-gray-800">
                  <TableCell className="font-mono text-sm py-3">
                    <div className="flex items-center">
                      <Key className="h-4 w-4 mr-2 text-gray-400" />
                      {license.key_value}
                    </div>
                  </TableCell>
                  <TableCell className="py-3">
                    <Badge variant="secondary" className="capitalize">
                      {license.subscription_tier}
                    </Badge>
                  </TableCell>
                  <TableCell className="py-3">
                    <Badge variant={getStatusVariant(license.status)}>
                      {license.status}
                    </Badge>
                  </TableCell>
                  <TableCell className="py-3 text-sm text-gray-500 dark:text-gray-400">
                    {new Date(license.created_at).toLocaleDateString()}
                  </TableCell>
                  <TableCell className="py-3 text-right">
                    <Select onValueChange={(value) => updateLicenseStatus(license.id, value)}>
                      <SelectTrigger className="w-32 ml-auto">
                        <SelectValue placeholder="Update Status" />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="active">Activate</SelectItem>
                        <SelectItem value="suspended">Suspend</SelectItem>
                        <SelectItem value="expired">Expire</SelectItem>
                      </SelectContent>
                    </Select>
                  </TableCell>
                </TableRow>
              ))
            )}
          </TableBody>
        </Table>
      </CardContent>
    </Card>
  );
};

export default LicenseManagement;