import React, { useState, useEffect } from "react";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Label } from "@/components/ui/label";
import { Switch } from "@/components/ui/switch";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Badge } from "@/components/ui/badge";
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { useToast } from "@/components/ui/use-toast";
import { apiService } from "@/services/api";
import { Plus, Edit, Trash2 } from "lucide-react";

interface EmailTemplate {
  id: string;
  name: string;
  subject: string;
  content: string;
  template_type: string;
  is_active: boolean;
  created_by_email: string;
  created_at: string;
  updated_at: string;
}

const EmailTemplates = () => {
  const [templates, setTemplates] = useState<EmailTemplate[]>([]);
  const [loading, setLoading] = useState(true);
  const [dialogOpen, setDialogOpen] = useState(false);
  const [editingTemplate, setEditingTemplate] = useState<EmailTemplate | null>(null);
  const [templateData, setTemplateData] = useState({
    name: "",
    subject: "",
    content: "",
    template_type: "welcome",
    is_active: true
  });
  const { toast } = useToast();

  useEffect(() => {
    fetchTemplates();
  }, []);

  const fetchTemplates = async () => {
    try {
      const response = await apiService.adminManageEmailTemplates('list');
      
      if (response.error) {
        throw new Error(response.error);
      }
      
      setTemplates(response.data.templates || []);
    } catch (error) {
      console.error('Error fetching email templates:', error);
      toast({
        title: "Error",
        description: "Failed to fetch email templates",
        variant: "destructive"
      });
    } finally {
      setLoading(false);
    }
  };

  const createTemplate = async () => {
    try {
      const response = await apiService.adminManageEmailTemplates('create', {
        templateData: templateData
      });
      
      if (response.error) {
        throw new Error(response.error);
      }
      
      toast({
        title: "Success",
        description: "Email template created successfully",
      });
      
      setDialogOpen(false);
      resetForm();
      fetchTemplates();
    } catch (error) {
      toast({
        title: "Error",
        description: "Failed to create email template",
        variant: "destructive"
      });
    }
  };

  const updateTemplate = async () => {
    if (!editingTemplate) return;
    
    try {
      const response = await apiService.adminManageEmailTemplates('update', {
        id: editingTemplate.id,
        templateData: templateData
      });
      
      if (response.error) {
        throw new Error(response.error);
      }
      
      toast({
        title: "Success",
        description: "Email template updated successfully",
      });
      
      setDialogOpen(false);
      setEditingTemplate(null);
      resetForm();
      fetchTemplates();
    } catch (error) {
      toast({
        title: "Error",
        description: "Failed to update email template",
        variant: "destructive"
      });
    }
  };

  const deleteTemplate = async (id: string) => {
    try {
      const response = await apiService.adminManageEmailTemplates('delete', {
        id
      });
      
      if (response.error) {
        throw new Error(response.error);
      }
      
      toast({
        title: "Success",
        description: "Email template deleted successfully",
      });
      
      fetchTemplates();
    } catch (error) {
      toast({
        title: "Error",
        description: "Failed to delete email template",
        variant: "destructive"
      });
    }
  };

  const resetForm = () => {
    setTemplateData({
      name: "",
      subject: "",
      content: "",
      template_type: "welcome",
      is_active: true
    });
  };

  const handleEdit = (template: EmailTemplate) => {
    setEditingTemplate(template);
    setTemplateData({
      name: template.name,
      subject: template.subject,
      content: template.content,
      template_type: template.template_type,
      is_active: template.is_active
    });
    setDialogOpen(true);
  };

  const handleSubmit = () => {
    if (editingTemplate) {
      updateTemplate();
    } else {
      createTemplate();
    }
  };

  const getTypeBadge = (type: string) => {
    const variants: { [key: string]: "default" | "destructive" | "outline" | "secondary" } = {
      welcome: "default",
      notification: "secondary",
      license_activation: "outline",
      receipt: "destructive"
    };
    const variant = variants[type] || "outline";
    return <Badge variant={variant}>{type}</Badge>;
  };

  const getStatusBadge = (isActive: boolean) => {
    return <Badge variant={isActive ? "default" : "secondary"}>{isActive ? "Active" : "Inactive"}</Badge>;
  };

  if (loading) {
    return <div className="p-6">Loading email templates...</div>;
  }

  return (
    <Card>
      <CardHeader>
        <div className="flex items-center justify-between">
          <div>
            <CardTitle>Email Templates</CardTitle>
            <CardDescription>Manage email templates for different types of communications</CardDescription>
          </div>
          <Dialog open={dialogOpen} onOpenChange={(open) => {
            setDialogOpen(open);
            if (!open) {
              setEditingTemplate(null);
              resetForm();
            }
          }}>
            <DialogTrigger asChild>
              <Button>
                <Plus className="w-4 h-4 mr-2" />
                Create Template
              </Button>
            </DialogTrigger>
            <DialogContent className="max-w-2xl">
              <DialogHeader>
                <DialogTitle>{editingTemplate ? "Edit Email Template" : "Create Email Template"}</DialogTitle>
                <DialogDescription>
                  {editingTemplate ? "Edit the email template details" : "Create a new email template"}
                </DialogDescription>
              </DialogHeader>
              <div className="grid gap-4 py-4">
                <div className="grid grid-cols-4 items-center gap-4">
                  <Label htmlFor="name" className="text-right">Name</Label>
                  <Input
                    id="name"
                    value={templateData.name}
                    onChange={(e) => setTemplateData({...templateData, name: e.target.value})}
                    className="col-span-3"
                  />
                </div>
                <div className="grid grid-cols-4 items-center gap-4">
                  <Label htmlFor="subject" className="text-right">Subject</Label>
                  <Input
                    id="subject"
                    value={templateData.subject}
                    onChange={(e) => setTemplateData({...templateData, subject: e.target.value})}
                    className="col-span-3"
                  />
                </div>
                <div className="grid grid-cols-4 items-center gap-4">
                  <Label htmlFor="template_type" className="text-right">Type</Label>
                  <Select value={templateData.template_type} onValueChange={(value) => setTemplateData({...templateData, template_type: value})}>
                    <SelectTrigger className="col-span-3">
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="welcome">Welcome</SelectItem>
                      <SelectItem value="notification">Notification</SelectItem>
                      <SelectItem value="license_activation">License Activation</SelectItem>
                      <SelectItem value="receipt">Receipt</SelectItem>
                      <SelectItem value="other">Other</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
                <div className="grid grid-cols-4 items-center gap-4">
                  <Label htmlFor="is_active" className="text-right">Active</Label>
                  <div className="col-span-3">
                    <Switch
                      id="is_active"
                      checked={templateData.is_active}
                      onCheckedChange={(checked) => setTemplateData({...templateData, is_active: checked})}
                    />
                  </div>
                </div>
                <div className="grid grid-cols-4 items-start gap-4">
                  <Label htmlFor="content" className="text-right pt-2">Content</Label>
                  <Textarea
                    id="content"
                    value={templateData.content}
                    onChange={(e) => setTemplateData({...templateData, content: e.target.value})}
                    className="col-span-3"
                    rows={10}
                  />
                </div>
              </div>
              <DialogFooter>
                <Button onClick={handleSubmit}>
                  {editingTemplate ? "Update Template" : "Create Template"}
                </Button>
              </DialogFooter>
            </DialogContent>
          </Dialog>
        </div>
      </CardHeader>
      <CardContent>
        <Table>
          <TableHeader>
            <TableRow>
              <TableHead>Name</TableHead>
              <TableHead>Type</TableHead>
              <TableHead>Subject</TableHead>
              <TableHead>Status</TableHead>
              <TableHead>Created By</TableHead>
              <TableHead>Actions</TableHead>
            </TableRow>
          </TableHeader>
          <TableBody>
            {templates.map((template) => (
              <TableRow key={template.id}>
                <TableCell className="font-medium">{template.name}</TableCell>
                <TableCell>{getTypeBadge(template.template_type)}</TableCell>
                <TableCell>{template.subject}</TableCell>
                <TableCell>{getStatusBadge(template.is_active)}</TableCell>
                <TableCell>{template.created_by_email}</TableCell>
                <TableCell>
                  <div className="flex space-x-2">
                    <Button variant="outline" size="sm" onClick={() => handleEdit(template)}>
                      <Edit className="w-4 h-4" />
                    </Button>
                    <Button variant="outline" size="sm" onClick={() => deleteTemplate(template.id)}>
                      <Trash2 className="w-4 h-4" />
                    </Button>
                  </div>
                </TableCell>
              </TableRow>
            ))}
          </TableBody>
        </Table>
      </CardContent>
    </Card>
  );
};

export default EmailTemplates;