import React, { useState } from "react";
import { Link, useLocation } from "react-router-dom";
import { SidebarLink } from "@/types";
import { ChevronDown, ChevronRight } from "lucide-react";
import { cn } from "@/lib/utils";

const sidebarLinks: SidebarLink[] = [
  { label: "Dashboard", icon: "📊", path: "/dashboard" },
  { label: "Refer & Earn", icon: "💰", path: "/refer" },
  { label: "World Chat", icon: "💬", path: "/world-chat", isNew: true },
  { label: "My Activation", icon: "🔑", path: "/activation" },
  { label: "Contact us", icon: "📞", path: "/contact" },
  { label: "Child Panel", icon: "👥", path: "/child-panel", isNew: true },
  { label: "Request Features", icon: "💡", path: "/request-features" },
  { label: "Flash", icon: "⚡", path: "/flash", isNew: true },
  { label: "Pricing", icon: "💳", path: "/pricing" },
  { 
    label: "History", 
    icon: "📜", 
    path: "/history",
    subLinks: [
      { label: "Mail History", icon: "📧", path: "/history/mail" },
      { label: "Payment History", icon: "💰", path: "/history/payment" }
    ]
  },
  { label: "Tutorials", icon: "📚", path: "/tutorials" },
  { label: "Logout", icon: "🚪", path: "/logout" }
];

interface SidebarProps {
  isOpen: boolean;
  onClose: () => void;
}

const Sidebar: React.FC<SidebarProps> = ({ isOpen, onClose }) => {
  const [expandedItems, setExpandedItems] = useState<string[]>([]);
  const location = useLocation();
  
  const toggleExpand = (label: string) => {
    setExpandedItems(prev => 
      prev.includes(label) 
        ? prev.filter(item => item !== label) 
        : [...prev, label]
    );
  };

  // Check if a path is active (exact match or subpath)
  const isActivePath = (path: string) => {
    if (path === "/dashboard" && location.pathname === "/") {
      return true; // Consider dashboard active on home page too
    }
    return location.pathname === path || 
           (path !== "/" && location.pathname.startsWith(path));
  };
  
  return (
    <>
      {/* Overlay for mobile */}
      {isOpen && (
        <div 
          className="fixed inset-0 bg-black/50 z-40 lg:hidden"
          onClick={onClose}
        />
      )}
      
      {/* Sidebar */}
      <aside 
        className={cn(
          "fixed top-0 left-0 h-full w-64 bg-sidebar border-r border-sidebar-border transition-transform duration-300 ease-in-out z-50 lg:relative lg:z-0 lg:translate-x-0",
          isOpen ? "translate-x-0" : "-translate-x-full"
        )}
      >
        <div className="flex flex-col h-full">
          {/* Logo section */}
          <div className="flex items-center justify-center h-16 border-b border-sidebar-border px-4 bg-sidebar">
            <Link to="/" className="flex items-center gap-2">
              <div className="w-8 h-8 bg-gradient-to-r from-flashcore-purple to-flashcore-green rounded-lg flex items-center justify-center">
                <span className="text-white font-bold text-sm">F</span>
              </div>
              <h1 className="text-xl font-bold bg-gradient-to-r from-flashcore-purple via-flashcore-green to-flashcore-orange bg-clip-text text-transparent">
                FlashCore
              </h1>
            </Link>
          </div>
          
          {/* Links */}
          <nav className="flex-1 overflow-y-auto py-4 px-3 bg-sidebar">
            <ul className="space-y-1">
              {sidebarLinks.map((link) => (
                <li key={link.label}>
                  {link.subLinks ? (
                    <div className="space-y-1">
                      <button
                        className={cn(
                          "flex items-center w-full px-3 py-2.5 text-sidebar-foreground hover:bg-sidebar-accent hover:text-sidebar-accent-foreground rounded-lg transition-colors text-left",
                          isActivePath(link.path) && "bg-sidebar-accent text-sidebar-accent-foreground font-medium"
                        )}
                        onClick={() => toggleExpand(link.label)}
                      >
                        <span className="mr-3 text-base flex-shrink-0">{link.icon}</span>
                        <span className="flex-1 text-sm font-medium">{link.label}</span>
                        <span className="ml-2 flex-shrink-0">
                          {expandedItems.includes(link.label) ? (
                            <ChevronDown size={16} />
                          ) : (
                            <ChevronRight size={16} />
                          )}
                        </span>
                        {link.isNew && (
                          <span className="ml-2 text-xs font-medium px-1.5 py-0.5 rounded-full bg-flashcore-purple text-white flex-shrink-0">
                            New
                          </span>
                        )}
                      </button>
                      
                      {expandedItems.includes(link.label) && (
                        <ul className="ml-6 space-y-1 border-l border-sidebar-border pl-3">
                          {link.subLinks.map((subLink) => (
                            <li key={subLink.label}>
                              <Link
                                to={subLink.path}
                                className={cn(
                                  "flex items-center px-3 py-2 text-sidebar-foreground hover:bg-sidebar-accent hover:text-sidebar-accent-foreground rounded-md transition-colors",
                                  isActivePath(subLink.path) && "bg-sidebar-accent text-sidebar-accent-foreground font-medium"
                                )}
                              >
                                <span className="mr-3 text-sm flex-shrink-0">{subLink.icon}</span>
                                <span className="text-sm">{subLink.label}</span>
                              </Link>
                            </li>
                          ))}
                        </ul>
                      )}
                    </div>
                  ) : (
                    <Link
                      to={link.path}
                      className={cn(
                        "flex items-center px-3 py-2.5 text-sidebar-foreground hover:bg-sidebar-accent hover:text-sidebar-accent-foreground rounded-lg transition-colors",
                        isActivePath(link.path) && "bg-sidebar-accent text-sidebar-accent-foreground font-medium"
                      )}
                    >
                      <span className="mr-3 text-base flex-shrink-0">{link.icon}</span>
                      <span className="text-sm font-medium">{link.label}</span>
                      {link.isNew && (
                        <span className="ml-2 text-xs font-medium px-1.5 py-0.5 rounded-full bg-flashcore-purple text-white flex-shrink-0">
                          New
                        </span>
                      )}
                    </Link>
                  )}
                </li>
              ))}
            </ul>
          </nav>
          
          {/* Footer */}
          <div className="p-4 border-t border-sidebar-border bg-sidebar">
            <div className="flex items-center space-x-3">
              <div className="w-8 h-8 rounded-full bg-gradient-to-r from-flashcore-purple to-flashcore-green flex items-center justify-center">
                <span className="text-sm font-medium text-white">AM</span>
              </div>
              <div className="flex-1 min-w-0">
                <p className="text-sm font-medium truncate">alex_mitchell</p>
                <p className="text-xs text-sidebar-foreground">Pro Account</p>
              </div>
            </div>
          </div>
        </div>
      </aside>
    </>
  );
};

export default Sidebar;