import React, { useState } from 'react';
import { Link } from 'react-router-dom';
import { Button } from "@/components/ui/button";
import { Sheet, SheetContent, SheetTrigger } from "@/components/ui/sheet";
import { Menu, X, ChevronDown } from "lucide-react";
import Logo from "@/components/Logo";

const Navigation = () => {
  const [isOpen, setIsOpen] = useState(false);

  const menuItems = [
    {
      label: "Product",
      href: "#",
      submenu: [
        { label: "Features", href: "/flash" },
        { label: "Templates", href: "/templates" },
        { label: "Integrations", href: "/integrations" },
        { label: "Pricing", href: "/pricing" },
      ]
    },
    {
      label: "Solutions",
      href: "#",
      submenu: [
        { label: "Small Business", href: "/solutions/small-business" },
        { label: "Enterprise", href: "/solutions/enterprise" },
        { label: "Agencies", href: "/solutions/agencies" },
        { label: "E-commerce", href: "/solutions/ecommerce" },
      ]
    },
    {
      label: "Resources",
      href: "#",
      submenu: [
        { label: "Documentation", href: "/docs" },
        { label: "Help Center", href: "/help" },
        { label: "Blog", href: "/blog" },
        { label: "Webinars", href: "/webinars" },
      ]
    },
    {
      label: "Company",
      href: "#",
      submenu: [
        { label: "About", href: "/about" },
        { label: "Contact", href: "/contact" },
        { label: "Careers", href: "/careers" },
        { label: "Press", href: "/press" },
      ]
    }
  ];

  return (
    <nav className="border-b border-border/40 bg-background/95 backdrop-blur supports-[backdrop-filter]:bg-background/60 sticky top-0 z-50">
      <div className="container mx-auto px-4 py-4">
        <div className="flex items-center justify-between">
          {/* Logo */}
          <div className="flex items-center space-x-2">
            <Logo className="w-8 h-8" />
            <span className="text-xl font-bold bg-gradient-to-r from-flashcore-purple to-flashcore-green bg-clip-text text-transparent">
              FlashCore
            </span>
          </div>

          {/* Desktop Menu */}
          <div className="hidden lg:flex items-center space-x-8">
            {menuItems.map((item, index) => (
              <div key={index} className="relative group">
                <button className="flex items-center space-x-1 text-sm font-medium text-foreground hover:text-flashcore-purple transition-colors">
                  <span>{item.label}</span>
                  <ChevronDown className="w-3 h-3" />
                </button>
                
                {/* Dropdown Menu */}
                <div className="absolute top-full left-0 mt-2 w-48 bg-background border border-border rounded-lg shadow-lg opacity-0 invisible group-hover:opacity-100 group-hover:visible transition-all duration-200 z-50">
                  <div className="py-2">
                    {item.submenu.map((subItem, subIndex) => (
                      <Link
                        key={subIndex}
                        to={subItem.href}
                        className="block px-4 py-2 text-sm text-muted-foreground hover:text-foreground hover:bg-muted transition-colors"
                        onClick={() => setIsOpen(false)}
                      >
                        {subItem.label}
                      </Link>
                    ))}
                  </div>
                </div>
              </div>
            ))}
          </div>

          {/* Desktop CTA Buttons */}
          <div className="hidden lg:flex items-center space-x-4">
            <Link to="/login" className="text-white hover:text-flashcore-green transition-colors">
              Sign In
            </Link>
            <Link to="/admin-login" className="text-white hover:text-flashcore-green transition-colors text-sm">
              Admin
            </Link>
            <Link to="/register">
              <Button className="bg-gradient-to-r from-flashcore-purple to-flashcore-green hover:from-flashcore-purple/90 hover:to-flashcore-green/90 text-white">
                Get Started
              </Button>
            </Link>
          </div>

          {/* Mobile Menu Button */}
          <div className="lg:hidden">
            <Sheet open={isOpen} onOpenChange={setIsOpen}>
              <SheetTrigger asChild>
                <Button variant="ghost" size="sm">
                  <Menu className="w-5 h-5" />
                </Button>
              </SheetTrigger>
              <SheetContent side="right" className="w-[300px] sm:w-[400px]">
                <div className="flex flex-col h-full">
                  {/* Mobile Menu Header */}
                  <div className="flex items-center justify-between mb-8">
                    <div className="flex items-center space-x-2">
                      <Logo className="w-6 h-6" />
                      <span className="text-lg font-bold bg-gradient-to-r from-flashcore-purple to-flashcore-green bg-clip-text text-transparent">
                        FlashCore
                      </span>
                    </div>
                    <Button
                      variant="ghost"
                      size="sm"
                      onClick={() => setIsOpen(false)}
                    >
                      <X className="w-5 h-5" />
                    </Button>
                  </div>

                  {/* Mobile Menu Items */}
                  <div className="flex-1 space-y-6">
                    {menuItems.map((item, index) => (
                      <div key={index} className="space-y-2">
                        <h3 className="text-sm font-semibold text-foreground">{item.label}</h3>
                        <div className="space-y-1 pl-4">
                          {item.submenu.map((subItem, subIndex) => (
                            <Link
                              key={subIndex}
                              to={subItem.href}
                              className="block py-2 text-sm text-muted-foreground hover:text-foreground transition-colors"
                              onClick={() => setIsOpen(false)}
                            >
                              {subItem.label}
                            </Link>
                          ))}
                        </div>
                      </div>
                    ))}
                  </div>

                  {/* Mobile CTA Buttons */}
                  <div className="space-y-3 pt-6 border-t border-border">
                    <div className="space-y-2">
                      <Link to="/login" className="block px-3 py-2 text-sm text-white hover:text-flashcore-green transition-colors">
                        Sign In
                      </Link>
                      <Link to="/admin-login" className="block px-3 py-2 text-sm text-white hover:text-flashcore-green transition-colors">
                        Admin Login
                      </Link>
                      <Link to="/register" className="block px-3 py-2 text-sm bg-gradient-to-r from-flashcore-purple to-flashcore-green text-white rounded-md">
                        Get Started
                      </Link>
                    </div>
                  </div>
                </div>
              </SheetContent>
            </Sheet>
          </div>
        </div>
      </div>
    </nav>
  );
};

export default Navigation; 