import React, { useState } from "react";
import { Dialog, DialogContent, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Label } from "@/components/ui/label";
import { Send, Mail, Users, Loader2, CheckCircle, AlertCircle } from "lucide-react";
import { useToast } from "@/hooks/use-toast";
import { useAuth } from "@/hooks/useAuth";
import { apiService } from "@/services/api"; // Changed from Supabase

interface EmailComposerProps {
  isOpen: boolean;
  onClose: () => void;
}

const EmailComposer: React.FC<EmailComposerProps> = ({ isOpen, onClose }) => {
  const [formData, setFormData] = useState({
    to: '',
    subject: '',
    message: ''
  });
  const [isLoading, setIsLoading] = useState(false);
  const [status, setStatus] = useState<'idle' | 'sending' | 'success' | 'error'>('idle');
  const { toast } = useToast();
  const { user } = useAuth();

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    if (!user) return;

    setIsLoading(true);
    setStatus('sending');

    try {
      // Send email via PHP API
      const response = await apiService.sendEmail({
        to: formData.to,
        subject: formData.subject,
        message: formData.message,
        from: user.email
      }); // Changed from Supabase

      if (response.error) throw new Error(response.error);

      setStatus('success');
      toast({
        title: "Email Sent Successfully!",
        description: "Your email has been delivered.",
      });

      setTimeout(() => {
        onClose();
        setFormData({ to: '', subject: '', message: '' });
        setStatus('idle');
      }, 2000);

    } catch (error) {
      console.error('Email sending failed:', error);
      setStatus('error');
      toast({
        title: "Failed to Send Email",
        description: "Please try again later.",
        variant: "destructive"
      });
    } finally {
      setIsLoading(false);
    }
  };

  const getStatusIcon = () => {
    switch (status) {
      case 'sending':
        return <Loader2 className="w-6 h-6 animate-spin text-blue-500" />;
      case 'success':
        return <CheckCircle className="w-6 h-6 text-green-500" />;
      case 'error':
        return <AlertCircle className="w-6 h-6 text-red-500" />;
      default:
        return <Mail className="w-6 h-6 text-flashcore-purple" />;
    }
  };

  return (
    <Dialog open={isOpen} onOpenChange={onClose}>
      <DialogContent className="max-w-2xl">
        <DialogHeader>
          <DialogTitle className="flex items-center gap-2">
            <Mail className="w-5 h-5" />
            Compose Email
          </DialogTitle>
        </DialogHeader>
        
        <div className="flex flex-col items-center justify-center py-8" style={{ display: status === 'idle' ? 'none' : 'flex' }}>
          {getStatusIcon()}
          <p className="mt-4 text-center text-muted-foreground">
            {status === 'sending' && 'Sending your email...'}
            {status === 'success' && 'Email sent successfully!'}
            {status === 'error' && 'Failed to send email. Please try again.'}
          </p>
        </div>

        <form onSubmit={handleSubmit} style={{ display: status === 'idle' ? 'block' : 'none' }}>
          <div className="space-y-4">
            <div>
              <Label htmlFor="to">To</Label>
              <Input
                id="to"
                type="email"
                placeholder="recipient@example.com"
                value={formData.to}
                onChange={(e) => setFormData({...formData, to: e.target.value})}
                required
              />
            </div>
            
            <div>
              <Label htmlFor="subject">Subject</Label>
              <Input
                id="subject"
                placeholder="Email subject"
                value={formData.subject}
                onChange={(e) => setFormData({...formData, subject: e.target.value})}
                required
              />
            </div>
            
            <div>
              <Label htmlFor="message">Message</Label>
              <Textarea
                id="message"
                placeholder="Write your message here..."
                rows={8}
                value={formData.message}
                onChange={(e) => setFormData({...formData, message: e.target.value})}
                required
              />
            </div>
            
            <div className="flex justify-end space-x-2">
              <Button type="button" variant="outline" onClick={onClose}>
                Cancel
              </Button>
              <Button type="submit" disabled={isLoading}>
                {isLoading ? (
                  <>
                    <Loader2 className="w-4 h-4 mr-2 animate-spin" />
                    Sending...
                  </>
                ) : (
                  <>
                    <Send className="w-4 h-4 mr-2" />
                    Send Email
                  </>
                )}
              </Button>
            </div>
          </div>
        </form>
      </DialogContent>
    </Dialog>
  );
};

export default EmailComposer;