<?php
/**
 * FlashCore Universal Setup Script
 * Supports: Windows (XAMPP/WAMP), Linux (Ubuntu/Debian/CentOS), and cPanel
 * 
 * Features:
 * - Environment detection
 * - Dependency verification (PHP, PDO, Node.js)
 * - Automated Database Creation & Seeding
 * - Configuration generation
 * - Professional UI with Glassmorphism
 */

error_reporting(E_ALL);
ini_set('display_errors', 1);

define('INSTALL_LOCK_FILE', __DIR__ . '/install.lock');

// If install is locked, redirect or die
if (file_exists(INSTALL_LOCK_FILE) && !isset($_GET['force'])) {
    die("Installation is locked. Please remove 'install.lock' to re-run the setup.");
}

$step = isset($_GET['step']) ? (int)$_GET['step'] : 1;
$status = '';
$error = '';

// Handle Actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';

    if ($action === 'database_setup') {
        $host = $_POST['db_host'] ?? 'localhost';
        $name = $_POST['db_name'] ?? 'flashcore_dashboard';
        $user = $_POST['db_user'] ?? 'root';
        $pass = $_POST['db_pass'] ?? '';

        try {
            // First connect without DB to create it
            $tempPdo = new PDO("mysql:host=$host", $user, $pass);
            $tempPdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
            
            // Create Database
            $tempPdo->exec("CREATE DATABASE IF NOT EXISTS `$name` CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");
            
            // Connect to actual DB
            $pdo = new PDO("mysql:host=$host;dbname=$name;charset=utf8mb4", $user, $pass);
            $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

            // Import consolidated SQL if available, otherwise fallback
            $productionFile = __DIR__ . '/database/production_ready.sql';
            $schemaFile = __DIR__ . '/database/schema.sql';
            $seedFile = __DIR__ . '/database/seed-data.sql';

            // Helper function to execute multi-statement SQL
            // Helper function to execute multi-statement SQL
            $executeMultiSQL = function($sqlContent) use ($pdo) {
                // 1. Remove SQL comments (lines starting with --)
                $sqlContent = preg_replace('/^\s*--.*$/m', '', $sqlContent);

                // 2. Remove CREATE DATABASE and USE statements
                $sqlContent = preg_replace('/CREATE DATABASE IF NOT EXISTS.*?;/i', '', $sqlContent);
                $sqlContent = preg_replace('/USE\s+[^;]+;/i', '', $sqlContent);

                // 3. Remove SET variable statements
                $sqlContent = preg_replace('/^\s*SET\s+@[^;]+;/mi', '', $sqlContent);

                // 4. Remove DELIMITER statements and replace custom delimiters
                $sqlContent = preg_replace('/DELIMITER\s+\S+/i', '', $sqlContent);
                $sqlContent = str_replace('//', ';', $sqlContent);

                // 5. Truncate at Start of Procedures
                $procPos = stripos($sqlContent, 'CREATE PROCEDURE');
                if ($procPos !== false) {
                    $sqlContent = substr($sqlContent, 0, $procPos);
                }

                // 6. Replace placeholder variables
                $adminId = '88888888-4444-4444-4444-121212121212';
                $userId  = '99999999-4444-4444-4444-121212121212';
                $sqlContent = str_replace('@admin_id', $adminId, $sqlContent);
                $sqlContent = str_replace('@user_id', $userId, $sqlContent);
                // Also handle variables from seed-data.sql if used
                $sqlContent = str_replace('@user1_id', '11111111-1111-1111-1111-111111111111', $sqlContent);
                $sqlContent = str_replace('@user2_id', '22222222-2222-2222-2222-222222222222', $sqlContent);
                $sqlContent = str_replace('@license1_id', 'aaaa-aaaa', $sqlContent); // Dummy replacements if needed

                // 7. Split and Execute
                $statements = explode(';', $sqlContent);
                foreach ($statements as $stmt) {
                    $stmt = trim($stmt);
                    if ($stmt === '') continue;
                    
                    try {
                        $pdo->exec($stmt);
                    } catch (PDOException $e) {
                        // Ignore harmless errors
                        $msg = $e->getMessage();
                        if (strpos($msg, 'already exists') === false && 
                            strpos($msg, 'Duplicate entry') === false) {
                            // Add specific statement to error for debugging
                            throw new Exception($msg . " [SQL: " . substr($stmt, 0, 50) . "...]");
                        }
                    }
                }
            };

            if (file_exists($productionFile)) {
                $executeMultiSQL(file_get_contents($productionFile));
            } else {
                // Legacy fallback
                if (file_exists($schemaFile)) {
                    $executeMultiSQL(file_get_contents($schemaFile));
                }
                if (file_exists($seedFile)) {
                    $executeMultiSQL(file_get_contents($seedFile));
                }
            }

            // Save Config
            $configContent = "<?php
// Database configuration
define('DB_HOST', '$host');
define('DB_NAME', '$name');
define('DB_USER', '$user');
define('DB_PASS', '$pass');

try {
    \$pdo = new PDO(
        \"mysql:host=\" . DB_HOST . \";dbname=\" . DB_NAME . \";charset=utf8mb4\",
        DB_USER,
        DB_PASS,
        [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_EMULATE_PREPARES => false
        ]
    );
} catch (PDOException \$e) {
    die(\"Database connection failed: \" . \$e->getMessage());
}
?>";
            
            file_put_contents(__DIR__ . '/backend/config/database.php', $configContent);
            
            file_put_contents(INSTALL_LOCK_FILE, date('Y-m-d H:i:s'));
            
            header("Location: setup.php?step=3");
            exit;

        } catch (PDOException $e) {
            $error = "Database Error: " . $e->getMessage();
        }
    }
}

// System Checks
$checks = [
    'PHP Version (>= 7.4)' => version_compare(PHP_VERSION, '7.4.0', '>='),
    'PDO Extension' => extension_loaded('pdo_mysql'),
    'Config Directory Writable' => is_writable(__DIR__ . '/backend/config'),
    'Database Directory' => is_dir(__DIR__ . '/database'),
];

$allPassed = !in_array(false, $checks);

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>FlashCore | Universal Setup</title>
    <link href="https://fonts.googleapis.com/css2?family=Outfit:wght@300;400;600;700&display=swap" rel="stylesheet">
    <style>
        :root {
            --primary: #8B5CF6;
            --secondary: #10B981;
            --bg: #0F172A;
            --card: rgba(30, 41, 59, 0.7);
            --border: rgba(255, 255, 255, 0.1);
        }

        body {
            font-family: 'Outfit', sans-serif;
            background: var(--bg);
            background-image: 
                radial-gradient(circle at 20% 20%, rgba(139, 92, 246, 0.15) 0%, transparent 40%),
                radial-gradient(circle at 80% 80%, rgba(16, 185, 129, 0.15) 0%, transparent 40%);
            color: white;
            margin: 0;
            display: flex;
            align-items: center;
            justify-content: center;
            min-height: 100vh;
            overflow-x: hidden;
        }

        .container {
            width: 100%;
            max-width: 500px;
            padding: 20px;
            z-index: 10;
        }

        .card {
            background: var(--card);
            backdrop-filter: blur(12px);
            border: 1px solid var(--border);
            border-radius: 24px;
            padding: 40px;
            box-shadow: 0 25px 50px -12px rgba(0, 0, 0, 0.5);
            animation: fadeIn 0.6s ease-out;
        }

        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(20px); }
            to { opacity: 1; transform: translateY(0); }
        }

        .logo {
            font-size: 32px;
            font-weight: 700;
            text-align: center;
            margin-bottom: 8px;
            background: linear-gradient(to right, #8B5CF6, #10B981);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
        }

        .subtitle {
            text-align: center;
            color: #94A3B8;
            font-size: 14px;
            margin-bottom: 40px;
        }

        h2 {
            font-size: 20px;
            margin-bottom: 24px;
            text-align: center;
        }

        .check-item {
            display: flex;
            justify-content: space-between;
            padding: 12px 16px;
            background: rgba(255, 255, 255, 0.05);
            border-radius: 12px;
            margin-bottom: 8px;
            font-size: 14px;
        }

        .status-pass { color: #10B981; font-weight: 600; }
        .status-fail { color: #EF4444; font-weight: 600; }

        .form-group {
            margin-bottom: 20px;
        }

        label {
            display: block;
            font-size: 13px;
            color: #94A3B8;
            margin-bottom: 8px;
            padding-left: 4px;
        }

        input {
            width: 100%;
            padding: 12px 16px;
            background: rgba(255, 255, 255, 0.05);
            border: 1px solid var(--border);
            border-radius: 12px;
            color: white;
            font-family: inherit;
            box-sizing: border-box;
            transition: all 0.3s;
        }

        input:focus {
            outline: none;
            border-color: var(--primary);
            background: rgba(255, 255, 255, 0.1);
        }

        .btn {
            display: block;
            width: 100%;
            padding: 14px;
            background: linear-gradient(to right, var(--primary), #7C3AED);
            border: none;
            border-radius: 12px;
            color: white;
            font-weight: 600;
            font-size: 16px;
            cursor: pointer;
            transition: all 0.3s;
            text-align: center;
            text-decoration: none;
            margin-top: 24px;
        }

        .btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 15px -3px rgba(139, 92, 246, 0.4);
        }

        .btn:disabled {
            opacity: 0.5;
            cursor: not-allowed;
            transform: none;
        }

        .error-msg {
            background: rgba(239, 68, 68, 0.1);
            border: 1px solid rgba(239, 68, 68, 0.2);
            color: #FCA5A5;
            padding: 12px;
            border-radius: 12px;
            margin-bottom: 24px;
            font-size: 14px;
            text-align: center;
        }

        .success-icon {
            font-size: 64px;
            color: #10B981;
            text-align: center;
            margin-bottom: 20px;
        }

        .footer {
            text-align: center;
            margin-top: 40px;
            color: #64748B;
            font-size: 12px;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="card">
            <div class="logo">FLASHCORE</div>
            <div class="subtitle">Universal Automation Setup</div>

            <?php if ($step === 1): ?>
                <h2>System Verification</h2>
                <div class="check-list">
                    <?php foreach ($checks as $label => $passed): ?>
                        <div class="check-item">
                            <span><?php echo $label; ?></span>
                            <span class="<?php echo $passed ? 'status-pass' : 'status-fail'; ?>">
                                <?php echo $passed ? 'PASSED' : 'FAILED'; ?>
                            </span>
                        </div>
                    <?php endforeach; ?>
                </div>

                <?php if ($allPassed): ?>
                    <a href="?step=2" class="btn">Next: Database Setup</a>
                <?php else: ?>
                    <button class="btn" disabled>Fix Requirements to Continue</button>
                    <p style="text-align:center; font-size:12px; color:#94A3B8; margin-top:16px;">
                        Ensure PHP 7.4+ and PDO are enabled.
                    </p>
                <?php endif; ?>

            <?php elseif ($step === 2): ?>
                <h2>Database Configuration</h2>
                
                <?php if ($error): ?>
                    <div class="error-msg"><?php echo $error; ?></div>
                <?php endif; ?>

                <form method="POST">
                    <input type="hidden" name="action" value="database_setup">
                    <div class="form-group">
                        <label>Database Host</label>
                        <input type="text" name="db_host" value="localhost" required>
                    </div>
                    <div class="form-group">
                        <label>Database Name</label>
                        <input type="text" name="db_name" value="flashcore_dashboard" required>
                    </div>
                    <div class="form-group">
                        <label>Username</label>
                        <input type="text" name="db_user" value="root" required>
                    </div>
                    <div class="form-group">
                        <label>Password</label>
                        <input type="password" name="db_pass" value="">
                    </div>
                    
                    <button type="submit" class="btn">Initialize & Install</button>
                </form>

            <?php elseif ($step === 3): ?>
                <div class="success-icon">✓</div>
                <h2 style="margin-bottom: 8px;">Installation Complete!</h2>
                <p style="text-align: center; color: #94A3B8; font-size: 14px; margin-bottom: 32px;">
                    FlashCore has been successfully configured and the database is initialized.
                </p>

                <div class="check-item">
                    <span>Database Status</span>
                    <span class="status-pass">Online</span>
                </div>
                <div class="check-item">
                    <span>Config File</span>
                    <span class="status-pass">Generated</span>
                </div>
                <div class="check-item">
                    <span>Security Lock</span>
                    <span class="status-pass">Active</span>
                </div>

                <a href="/dashboard" class="btn">Go to Dashboard</a>
                
                <p style="text-align:center; font-size:11px; color:#FCA5A5; margin-top:20px; font-weight:600;">
                    IMPORTANT: Delete setup.php for security before production.
                </p>
            <?php endif; ?>
        </div>
        
        <div class="footer">
            &copy; <?php echo date('Y'); ?> FlashCore Automation Engine. All rights reserved.
        </div>
    </div>
</body>
</html>
