#!/bin/bash

# Replit Deployment Script for FlashCore Dashboard
# This script automates the setup process for Replit environment

echo "🚀 Starting FlashCore Dashboard deployment for Replit..."

# Check if we're in the correct directory
if [ ! -f "package.json" ]; then
    echo "❌ Error: Please run this script from the project root directory"
    exit 1
fi

# Create necessary directories
echo "📁 Creating directories..."
mkdir -p backend/config
mkdir -p database/backups

# Install Node.js dependencies
echo "📦 Installing Node.js dependencies..."
npm install

# Create replit.nix file for Replit
echo "🔧 Creating Replit configuration..."
cat > replit.nix << 'EOF'
{ pkgs }: {
    deps = [
        pkgs.nodejs-18_x
        pkgs.nodePackages.typescript-language-server
        pkgs.yarn
        pkgs.replitPackages.jest
        pkgs.php81
        pkgs.php81Extensions.pdo
        pkgs.php81Extensions.pdo_mysql
        pkgs.mysql80
    ];
}
EOF

# Create .replit file for Replit
echo "⚙️ Creating Replit run configuration..."
cat > .replit << 'EOF'
# Replit configuration for FlashCore Dashboard
run = "npm run dev"

[general]
autoStart = true

[database]
type = "mysql"
primary = true
</EOF]

# Create a config file for database connection
echo "⚙️ Creating database configuration..."
cat > backend/config/database.php << 'EOF'
<?php
// Database configuration for Replit
$host = getenv('DB_HOST') ?: 'localhost';
$dbname = getenv('DB_NAME') ?: 'flashcore_dashboard';
$username = getenv('DB_USER') ?: 'repl_user';
$password = getenv('DB_PASS') ?: 'repl_password';

try {
    $pdo = new PDO("mysql:host=$host;dbname=$dbname;charset=utf8mb4", $username, $password, [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        PDO::ATTR_EMULATE_PREPARES => false,
    ]);
} catch (PDOException $e) {
    error_log("Database connection failed: " . $e->getMessage());
    die("Connection failed. Please check your database configuration.");
}
?>
EOF

# Update package.json to work with Replit
echo "📝 Updating package.json for Replit..."
npm pkg set scripts.replit="vite --host 0.0.0.0 --port 8080"

echo "✅ Replit deployment configuration completed!"
echo ""
echo "Next steps:"
echo "1. Import this project to Replit"
echo "2. Replit will automatically use the configuration files created"
echo "3. The development server will start automatically on port 8080"
echo "4. Set up your database using Replit's database feature"
echo "5. Update backend/config/database.php with your database credentials"
echo ""
echo "Note: Replit has specific limitations. For production use,"
echo "consider using a more robust hosting solution."