#!/bin/bash

# Render Deployment Script for FlashCore Dashboard
# This script automates the setup process for Render cloud platform

echo "🚀 Starting FlashCore Dashboard deployment for Render..."

# Check if we're in the correct directory
if [ ! -f "package.json" ]; then
    echo "❌ Error: Please run this script from the project root directory"
    exit 1
fi

# Create necessary directories
echo "📁 Creating directories..."
mkdir -p backend/config
mkdir -p database/backups

# Create render.yaml file for Render
echo "🔧 Creating Render configuration..."
cat > render.yaml << 'EOF'
services:
  - type: web
    name: flashcore-dashboard
    env: static
    buildCommand: npm install && npm run build
    staticPublishPath: ./dist
    envVars:
      - key: NODE_VERSION
        value: 18
    routes:
      - type: rewrite
        source: /backend/api/*
        destination: https://YOUR_RENDER_REGION.onrender.com/backend/api/

databases:
  - name: flashcore-db
    databaseName: flashcore_dashboard
    user: flashcore_user
    plan: free
EOF

# Install Node.js dependencies
echo "📦 Installing Node.js dependencies..."
npm install

# Build the frontend
echo "🔨 Building frontend..."
npm run build

# Create a config file for database connection
echo "⚙️ Creating database configuration..."
cat > backend/config/database.php << 'EOF'
<?php
// Database configuration for Render
$host = getenv('DB_HOST') ?: 'localhost';
$dbname = getenv('DB_NAME') ?: 'flashcore_dashboard';
$username = getenv('DB_USER') ?: 'flashcore_user';
$password = getenv('DB_PASS') ?: 'flashcore_password';

try {
    $pdo = new PDO("mysql:host=$host;dbname=$dbname;charset=utf8mb4", $username, $password, [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        PDO::ATTR_EMULATE_PREPARES => false,
    ]);
} catch (PDOException $e) {
    error_log("Database connection failed: " . $e->getMessage());
    die("Connection failed. Please check your database configuration.");
}
?>
EOF

echo "✅ Render deployment configuration completed!"
echo ""
echo "Next steps:"
echo "1. Create a new Web Service on Render"
echo "2. Connect your repository to Render"
echo "3. Render will automatically use the render.yaml file"
echo "4. Create a PostgreSQL database on Render"
echo "5. Update the environment variables in Render dashboard:"
echo "   - DB_HOST: Your database host"
echo "   - DB_NAME: Your database name"
echo "   - DB_USER: Your database user"
echo "   - DB_PASS: Your database password"
echo "6. Deploy the application"
echo ""
echo "Note: For the backend API to work properly on Render,"
echo "you may need to create a separate service for the PHP backend"
echo "or use Render's managed services for a more complex setup."