#!/bin/bash

# HPanel Deployment Script for FlashCore Dashboard
# This script automates the setup process for shared hosting environments

echo "🚀 Starting FlashCore Dashboard deployment for HPanel..."

# Check if we're in the correct directory
if [ ! -f "package.json" ]; then
    echo "❌ Error: Please run this script from the project root directory"
    exit 1
fi

# Create necessary directories
echo "📁 Creating directories..."
mkdir -p backend/config
mkdir -p database/backups

# Install Node.js dependencies
echo "📦 Installing Node.js dependencies..."
npm install --production

# Build the frontend
echo "🔨 Building frontend..."
npm run build

# Set permissions
echo "🔐 Setting permissions..."
chmod -R 755 backend/
chmod -R 755 database/

# Create .htaccess for routing
echo "🔗 Creating .htaccess files..."

cat > .htaccess << 'EOF'
# Enable rewrite engine
RewriteEngine On

# Protect sensitive files
<Files ~ "\.env|\.json|\.lock">
    Order allow,deny
    Deny from all
</Files>

# Route API requests to backend
RewriteRule ^backend/api/(.*)$ backend/api/$1 [QSA,L]

# Route all other requests to frontend
RewriteCond %{REQUEST_FILENAME} !-f
RewriteCond %{REQUEST_FILENAME} !-d
RewriteRule ^(.*)$ index.html [QSA,L]
EOF

# Create backend .htaccess for security
cat > backend/.htaccess << 'EOF'
# Prevent direct access to PHP files
<Files "*.php">
    Order Allow,Deny
    Deny from all
</Files>

# Allow access only through the API
<FilesMatch "\.php$">
    Order Allow,Deny
    Allow from env=REDIRECT_STATUS
</FilesMatch>
EOF

# Create a config file for database connection
echo "⚙️ Creating database configuration..."
cat > backend/config/database.php << 'EOF'
<?php
// Database configuration for shared hosting
$host = getenv('DB_HOST') ?: 'localhost';
$dbname = getenv('DB_NAME') ?: 'flashcore_db';
$username = getenv('DB_USER') ?: 'db_user';
$password = getenv('DB_PASS') ?: 'db_password';

try {
    $pdo = new PDO("mysql:host=$host;dbname=$dbname;charset=utf8mb4", $username, $password, [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        PDO::ATTR_EMULATE_PREPARES => false,
    ]);
} catch (PDOException $e) {
    error_log("Database connection failed: " . $e->getMessage());
    die("Connection failed. Please check your database configuration.");
}
?>
EOF

echo "✅ HPanel deployment completed!"
echo ""
echo "Next steps:"
echo "1. Upload all files to your hosting account"
echo "2. Create a MySQL database in your HPanel"
echo "3. Update backend/config/database.php with your database credentials"
echo "4. Import the database schema from database/schema.sql"
echo "5. Configure your domain to point to the public directory"
echo ""
echo "For additional help, refer to your hosting provider's documentation."