#!/bin/bash

# cPanel Deployment Script for FlashCore Dashboard
# This script automates the setup process for cPanel hosting environments

echo "🚀 Starting FlashCore Dashboard deployment for cPanel..."

# Check if we're in the correct directory
if [ ! -f "package.json" ]; then
    echo "❌ Error: Please run this script from the project root directory"
    exit 1
fi

# Create necessary directories
echo "📁 Creating directories..."
mkdir -p backend/config
mkdir -p database/backups
mkdir -p public_html

# Install Node.js dependencies
echo "📦 Installing Node.js dependencies..."
npm install --production

# Build the frontend
echo "🔨 Building frontend..."
npm run build

# Move built files to public_html
echo "🚚 Moving built files to public_html..."
cp -r dist/* public_html/

# Set permissions
echo "🔐 Setting permissions..."
chmod -R 755 backend/
chmod -R 755 database/
chmod -R 755 public_html/

# Create .htaccess for routing
echo "🔗 Creating .htaccess files..."

cat > public_html/.htaccess << 'EOF'
# Enable rewrite engine
RewriteEngine On

# Protect sensitive files
<Files ~ "\.env|\.json|\.lock">
    Order allow,deny
    Deny from all
</Files>

# Route API requests to backend
RewriteRule ^backend/api/(.*)$ ../backend/api/$1 [QSA,L]

# Route all other requests to frontend
RewriteCond %{REQUEST_FILENAME} !-f
RewriteCond %{REQUEST_FILENAME} !-d
RewriteRule ^(.*)$ index.html [QSA,L]
EOF

# Create backend .htaccess for security
cat > backend/.htaccess << 'EOF'
# Prevent direct access to PHP files
<Files "*.php">
    Order Allow,Deny
    Deny from all
</Files>

# Allow access only through the API
<FilesMatch "\.php$">
    Order Allow,Deny
    Allow from env=REDIRECT_STATUS
</FilesMatch>
EOF

# Create a config file for database connection
echo "⚙️ Creating database configuration..."
cat > backend/config/database.php << 'EOF'
<?php
// Database configuration for cPanel
$host = getenv('DB_HOST') ?: 'localhost';
$dbname = getenv('DB_NAME') ?: 'flashcore_db';
$username = getenv('DB_USER') ?: 'db_user';
$password = getenv('DB_PASS') ?: 'db_password';

try {
    $pdo = new PDO("mysql:host=$host;dbname=$dbname;charset=utf8mb4", $username, $password, [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        PDO::ATTR_EMULATE_PREPARES => false,
    ]);
} catch (PDOException $e) {
    error_log("Database connection failed: " . $e->getMessage());
    die("Connection failed. Please check your database configuration.");
}
?>
EOF

# Create a cron job for backups (optional)
echo "⏰ Setting up backup cron job..."
(crontab -l 2>/dev/null; echo "0 2 * * * cd /home/$(whoami)/public_html && node scripts/backup-system/backup.js --daily") | crontab -

echo "✅ cPanel deployment completed!"
echo ""
echo "Next steps:"
echo "1. Create a MySQL database using cPanel's MySQL Database Wizard"
echo "2. Update backend/config/database.php with your database credentials"
echo "3. Import the database schema from database/schema.sql"
echo "4. Set up your domain to point to the public_html directory"
echo "5. Test the application by visiting your domain"
echo ""
echo "A daily backup cron job has been set up to run at 2 AM server time."
echo "You can modify this in your cPanel Cron Jobs section."