#!/usr/bin/env node

import fs from 'fs-extra';
import path from 'path';
import archiver from 'archiver';
import chalk from 'chalk';
import ora from 'ora';
import { program } from 'commander';
import { glob } from 'glob';
import { z } from 'zod';
import dotenv from 'dotenv';
import crypto from 'crypto';

// Load environment variables
dotenv.config();

// Verification configuration schema
const VerifyConfigSchema = z.object({
  backupFile: z.string().optional(),
  backupDir: z.string().default('backups'),
  extract: z.boolean().default(false),
  checksum: z.boolean().default(true),
  detailed: z.boolean().default(false),
  fix: z.boolean().default(false)
});

class BackupVerifier {
  constructor(options = {}) {
    this.config = VerifyConfigSchema.parse(options);
  }

  log(message, type = 'info') {
    const timestamp = new Date().toISOString();
    const colors = {
      info: chalk.blue,
      success: chalk.green,
      error: chalk.red,
      warning: chalk.yellow,
      debug: chalk.gray
    };
    
    const prefix = {
      info: 'ℹ️',
      success: '✅',
      error: '❌',
      warning: '⚠️',
      debug: '🔍'
    }[type];
    
    console.log(`${colors[type](`${prefix} [${timestamp}] ${message}`)}`);
  }

  async getBackupFiles() {
    try {
      if (this.config.backupFile) {
        return [this.config.backupFile];
      }
      
      const backupFiles = await glob(path.join(this.config.backupDir, '*.tar.gz'));
      return backupFiles;
    } catch (error) {
      throw new Error(`Failed to get backup files: ${error.message}`);
    }
  }

  async calculateChecksum(filePath) {
    return new Promise((resolve, reject) => {
      const hash = crypto.createHash('sha256');
      const stream = fs.createReadStream(filePath);
      
      stream.on('data', (data) => {
        hash.update(data);
      });
      
      stream.on('end', () => {
        resolve(hash.digest('hex'));
      });
      
      stream.on('error', (error) => {
        reject(error);
      });
    });
  }

  async verifyArchiveIntegrity(backupFile) {
    const spinner = ora(`🔍 Verifying archive integrity: ${path.basename(backupFile)}`).start();
    
    return new Promise((resolve, reject) => {
      const input = fs.createReadStream(backupFile);
      const extract = archiver('tar', { gzip: true });
      
      let hasError = false;
      let fileCount = 0;
      
      extract.on('entry', (entry) => {
        fileCount++;
        if (entry.type === 'file') {
          // Verify file can be read
          entry.on('data', () => {});
          entry.on('end', () => {});
        }
      });
      
      extract.on('error', (err) => {
        hasError = true;
        spinner.fail(`Archive integrity check failed: ${err.message}`);
        reject(err);
      });
      
      extract.on('end', () => {
        if (!hasError) {
          spinner.succeed(`Archive integrity verified: ${fileCount} files`);
          resolve({ valid: true, fileCount });
        }
      });
      
      input.pipe(extract);
    });
  }

  async extractManifest(backupFile) {
    try {
      const tempDir = path.join(process.cwd(), 'temp-verify');
      await fs.ensureDir(tempDir);
      
      const { execSync } = await import('child_process');
      execSync(`tar -xzf "${backupFile}" -C "${tempDir}" backup-manifest.json`, { 
        stdio: 'pipe' 
      });
      
      const manifestPath = path.join(tempDir, 'backup-manifest.json');
      if (await fs.pathExists(manifestPath)) {
        const manifest = JSON.parse(await fs.readFile(manifestPath, 'utf8'));
        await fs.remove(tempDir);
        return manifest;
      }
      
      await fs.remove(tempDir);
      return null;
    } catch (error) {
      // Clean up temp directory if it exists
      try {
        await fs.remove(path.join(process.cwd(), 'temp-verify'));
      } catch {}
      return null;
    }
  }

  async verifyManifest(manifest, backupFile) {
    const spinner = ora('📋 Verifying backup manifest...').start();
    
    try {
      if (!manifest) {
        spinner.fail('No manifest found in backup');
        return { valid: false, errors: ['No manifest found'] };
      }
      
      const errors = [];
      const warnings = [];
      
      // Check required fields
      const requiredFields = ['project', 'backupDate', 'version', 'files'];
      for (const field of requiredFields) {
        if (!manifest[field]) {
          errors.push(`Missing required field: ${field}`);
        }
      }
      
      // Check file count
      if (manifest.files && Array.isArray(manifest.files)) {
        if (manifest.files.length === 0) {
          warnings.push('No files listed in manifest');
        }
        
        // Check for critical files
        const criticalFiles = ['package.json', 'src/App.tsx', 'src/main.tsx'];
        const manifestFilePaths = manifest.files.map(f => f.path);
        
        for (const criticalFile of criticalFiles) {
          if (!manifestFilePaths.some(path => path.includes(criticalFile))) {
            warnings.push(`Critical file not found: ${criticalFile}`);
          }
        }
      }
      
      // Check backup date
      if (manifest.backupDate) {
        const backupDate = new Date(manifest.backupDate);
        const now = new Date();
        const ageInDays = (now.getTime() - backupDate.getTime()) / (1000 * 60 * 60 * 24);
        
        if (ageInDays > 365) {
          warnings.push(`Backup is very old: ${Math.floor(ageInDays)} days`);
        }
      }
      
      if (errors.length > 0) {
        spinner.fail(`Manifest verification failed: ${errors.length} errors`);
        return { valid: false, errors, warnings };
      } else if (warnings.length > 0) {
        spinner.warn(`Manifest verified with ${warnings.length} warnings`);
        return { valid: true, warnings };
      } else {
        spinner.succeed('Manifest verification passed');
        return { valid: true };
      }
    } catch (error) {
      spinner.fail(`Manifest verification failed: ${error.message}`);
      return { valid: false, errors: [error.message] };
    }
  }

  async extractBackupForVerification(backupFile) {
    if (!this.config.extract) {
      return null;
    }
    
    const spinner = ora('📦 Extracting backup for detailed verification...').start();
    
    try {
      const extractDir = path.join(process.cwd(), 'extracted-verify');
      await fs.ensureDir(extractDir);
      
      const { execSync } = await import('child_process');
      execSync(`tar -xzf "${backupFile}" -C "${extractDir}"`, { stdio: 'pipe' });
      
      spinner.succeed('Backup extracted for verification');
      return extractDir;
    } catch (error) {
      spinner.fail(`Failed to extract backup: ${error.message}`);
      throw error;
    }
  }

  async verifyExtractedFiles(extractDir, manifest) {
    const spinner = ora('🔍 Verifying extracted files...').start();
    
    try {
      const errors = [];
      const warnings = [];
      
      if (!manifest || !manifest.files) {
        spinner.fail('No file list in manifest');
        return { valid: false, errors: ['No file list in manifest'] };
      }
      
      let verifiedCount = 0;
      let missingCount = 0;
      
      for (const fileInfo of manifest.files) {
        const filePath = path.join(extractDir, fileInfo.path);
        
        if (await fs.pathExists(filePath)) {
          const stat = await fs.stat(filePath);
          
          // Check file size
          if (stat.size !== fileInfo.size) {
            warnings.push(`Size mismatch for ${fileInfo.path}: expected ${fileInfo.size}, got ${stat.size}`);
          }
          
          verifiedCount++;
        } else {
          errors.push(`Missing file: ${fileInfo.path}`);
          missingCount++;
        }
      }
      
      if (errors.length > 0) {
        spinner.fail(`File verification failed: ${missingCount} missing files`);
        return { valid: false, errors, warnings };
      } else if (warnings.length > 0) {
        spinner.warn(`File verification passed with ${warnings.length} warnings`);
        return { valid: true, warnings, verifiedCount };
      } else {
        spinner.succeed(`File verification passed: ${verifiedCount} files verified`);
        return { valid: true, verifiedCount };
      }
    } catch (error) {
      spinner.fail(`File verification failed: ${error.message}`);
      return { valid: false, errors: [error.message] };
    }
  }

  async cleanupExtractedFiles(extractDir) {
    if (extractDir && await fs.pathExists(extractDir)) {
      try {
        await fs.remove(extractDir);
        this.log('🧹 Cleaned up extracted files');
      } catch (error) {
        this.log(`Warning: Could not clean up extracted files: ${error.message}`, 'warning');
      }
    }
  }

  formatBytes(bytes) {
    if (bytes === 0) return '0 Bytes';
    const k = 1024;
    const sizes = ['Bytes', 'KB', 'MB', 'GB', 'TB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
  }

  formatDate(date) {
    return new Date(date).toLocaleString();
  }

  async verifyBackup(backupFile) {
    const results = {
      file: path.basename(backupFile),
      path: backupFile,
      archiveIntegrity: null,
      manifest: null,
      manifestVerification: null,
      fileVerification: null,
      checksum: null,
      overall: false
    };
    
    try {
      // Get file info
      const stat = await fs.stat(backupFile);
      results.size = stat.size;
      results.modified = stat.mtime;
      
      // Verify archive integrity
      results.archiveIntegrity = await this.verifyArchiveIntegrity(backupFile);
      
      // Calculate checksum
      if (this.config.checksum) {
        const spinner = ora('🔐 Calculating checksum...').start();
        results.checksum = await this.calculateChecksum(backupFile);
        spinner.succeed(`Checksum: ${results.checksum}`);
      }
      
      // Extract and verify manifest
      results.manifest = await this.extractManifest(backupFile);
      results.manifestVerification = await this.verifyManifest(results.manifest, backupFile);
      
      // Extract and verify files if requested
      let extractDir = null;
      if (this.config.extract && results.manifest) {
        extractDir = await this.extractBackupForVerification(backupFile);
        results.fileVerification = await this.verifyExtractedFiles(extractDir, results.manifest);
      }
      
      // Determine overall status
      results.overall = results.archiveIntegrity.valid && 
                       results.manifestVerification.valid &&
                       (!results.fileVerification || results.fileVerification.valid);
      
      // Cleanup
      await this.cleanupExtractedFiles(extractDir);
      
      return results;
    } catch (error) {
      results.error = error.message;
      results.overall = false;
      return results;
    }
  }

  async showVerificationResults(results) {
    console.log('\n🔍 FlashCore Backup Verification Results');
    console.log('========================================');
    
    for (const result of results) {
      console.log(`\n📦 Backup: ${result.file}`);
      console.log(`   Size: ${this.formatBytes(result.size)}`);
      console.log(`   Modified: ${this.formatDate(result.modified)}`);
      
      if (result.checksum) {
        console.log(`   Checksum: ${result.checksum}`);
      }
      
      // Archive integrity
      if (result.archiveIntegrity) {
        const status = result.archiveIntegrity.valid ? '✅' : '❌';
        console.log(`   Archive Integrity: ${status} ${result.archiveIntegrity.fileCount} files`);
      }
      
      // Manifest verification
      if (result.manifestVerification) {
        const status = result.manifestVerification.valid ? '✅' : '❌';
        console.log(`   Manifest: ${status}`);
        
        if (result.manifestVerification.errors) {
          for (const error of result.manifestVerification.errors) {
            console.log(`     ❌ ${error}`);
          }
        }
        
        if (result.manifestVerification.warnings) {
          for (const warning of result.manifestVerification.warnings) {
            console.log(`     ⚠️  ${warning}`);
          }
        }
      }
      
      // File verification
      if (result.fileVerification) {
        const status = result.fileVerification.valid ? '✅' : '❌';
        console.log(`   Files: ${status} ${result.fileVerification.verifiedCount} verified`);
        
        if (result.fileVerification.errors) {
          for (const error of result.fileVerification.errors.slice(0, 5)) {
            console.log(`     ❌ ${error}`);
          }
          if (result.fileVerification.errors.length > 5) {
            console.log(`     ... and ${result.fileVerification.errors.length - 5} more errors`);
          }
        }
      }
      
      // Overall status
      const overallStatus = result.overall ? '✅ VALID' : '❌ INVALID';
      console.log(`   Overall: ${overallStatus}`);
      
      if (result.error) {
        console.log(`   Error: ${result.error}`);
      }
    }
    
    // Summary
    const validCount = results.filter(r => r.overall).length;
    const totalCount = results.length;
    
    console.log(`\n📊 Summary:`);
    console.log(`   Total backups: ${totalCount}`);
    console.log(`   Valid backups: ${validCount}`);
    console.log(`   Invalid backups: ${totalCount - validCount}`);
    
    if (validCount === totalCount) {
      console.log(`\n🎉 All backups are valid!`);
    } else {
      console.log(`\n⚠️  Some backups have issues that need attention.`);
    }
  }

  async run() {
    try {
      this.log('🔍 Starting backup verification...');
      
      const backupFiles = await this.getBackupFiles();
      
      if (backupFiles.length === 0) {
        this.log('No backup files found to verify', 'warning');
        return;
      }
      
      this.log(`Found ${backupFiles.length} backup file(s) to verify`);
      
      const results = [];
      for (const backupFile of backupFiles) {
        const result = await this.verifyBackup(backupFile);
        results.push(result);
      }
      
      await this.showVerificationResults(results);
      
    } catch (error) {
      this.log(`Verification failed: ${error.message}`, 'error');
      process.exit(1);
    }
  }
}

// CLI setup
program
  .name('flashcore-verify')
  .description('Verify FlashCore backup integrity and completeness')
  .version('1.0.0');

program
  .argument('[backup-file]', 'Specific backup file to verify')
  .option('-d, --backup-dir <path>', 'Backup directory', 'backups')
  .option('-e, --extract', 'Extract and verify all files')
  .option('--no-checksum', 'Skip checksum calculation')
  .option('--detailed', 'Show detailed verification information')
  .option('--fix', 'Attempt to fix issues (not implemented yet)');

program.parse();

const options = program.opts();
const backupFile = program.args[0];

// Run verification
const verifier = new BackupVerifier({
  backupFile,
  backupDir: options.backupDir,
  extract: options.extract,
  checksum: options.checksum,
  detailed: options.detailed,
  fix: options.fix
});

verifier.run(); 