#!/usr/bin/env node

import fs from 'fs-extra';
import path from 'path';
import chalk from 'chalk';
import ora from 'ora';
import { program } from 'commander';
import { z } from 'zod';
import dotenv from 'dotenv';
import FlashCoreBackup from './backup.js';
import FlashCoreRestore from './restore.js';

// Load environment variables
dotenv.config();

// Test configuration schema
const TestConfigSchema = z.object({
  testDir: z.string().default('test-backup'),
  cleanup: z.boolean().default(true),
  verbose: z.boolean().default(false),
  testRestore: z.boolean().default(true),
  testIntegrity: z.boolean().default(true)
});

class BackupTester {
  constructor(options = {}) {
    this.config = TestConfigSchema.parse(options);
    this.testResults = [];
  }

  log(message, type = 'info') {
    const timestamp = new Date().toISOString();
    const colors = {
      info: chalk.blue,
      success: chalk.green,
      error: chalk.red,
      warning: chalk.yellow,
      debug: chalk.gray
    };
    
    const prefix = {
      info: 'ℹ️',
      success: '✅',
      error: '❌',
      warning: '⚠️',
      debug: '🔍'
    }[type];
    
    if (this.config.verbose || type !== 'debug') {
      console.log(`${colors[type](`${prefix} [${timestamp}] ${message}`)}`);
    }
  }

  async createTestProject() {
    const spinner = ora('🧪 Creating test project...').start();
    
    try {
      await fs.ensureDir(this.config.testDir);
      
      // Create package.json
      const packageJson = {
        name: "test-project",
        version: "1.0.0",
        description: "Test project for backup system",
        main: "index.js",
        scripts: {
          test: "echo \"Error: no test specified\" && exit 1"
        },
        dependencies: {
          "react": "^18.3.1",
          "typescript": "^5.5.3"
        },
        devDependencies: {
          "vite": "^5.4.1"
        }
      };
      
      await fs.writeFile(
        path.join(this.config.testDir, 'package.json'),
        JSON.stringify(packageJson, null, 2)
      );
      
      // Create source files
      await fs.ensureDir(path.join(this.config.testDir, 'src'));
      await fs.writeFile(
        path.join(this.config.testDir, 'src', 'App.tsx'),
        'import React from "react";\n\nexport default function App() {\n  return <div>Test App</div>;\n}'
      );
      
      await fs.writeFile(
        path.join(this.config.testDir, 'src', 'main.tsx'),
        'import React from "react";\nimport App from "./App";\n\nReactDOM.render(<App />, document.getElementById("root"));'
      );
      
      // Create configuration files
      await fs.writeFile(
        path.join(this.config.testDir, 'tsconfig.json'),
        JSON.stringify({
          compilerOptions: {
            target: "ES2020",
            module: "ESNext",
            moduleResolution: "node"
          }
        }, null, 2)
      );
      
      await fs.writeFile(
        path.join(this.config.testDir, 'vite.config.ts'),
        'import { defineConfig } from "vite";\n\nexport default defineConfig({\n  plugins: []\n});'
      );
      
      // Create environment file
      await fs.writeFile(
        path.join(this.config.testDir, '.env'),
        'VITE_TEST_VAR=test_value\nNODE_ENV=development'
      );
      
      // Create documentation
      await fs.writeFile(
        path.join(this.config.testDir, 'README.md'),
        '# Test Project\n\nThis is a test project for the backup system.'
      );
      
      // Create Supabase config
      await fs.ensureDir(path.join(this.config.testDir, 'supabase'));
      await fs.writeFile(
        path.join(this.config.testDir, 'supabase', 'config.toml'),
        '[api]\nport = 54321\n\n[db]\nport = 54322'
      );
      
      spinner.succeed('Test project created successfully');
      return true;
    } catch (error) {
      spinner.fail(`Failed to create test project: ${error.message}`);
      return false;
    }
  }

  async testBackup() {
    const spinner = ora('📦 Testing backup functionality...').start();
    
    try {
      // Change to test directory
      const originalCwd = process.cwd();
      process.chdir(this.config.testDir);
      
      // Create backup
      const backup = new FlashCoreBackup({
        backupDir: 'test-backups',
        compressionLevel: 6,
        maxBackups: 3
      });
      
      await backup.run();
      
      // Check if backup was created
      const backupFiles = await fs.readdir('test-backups');
      const backupFile = backupFiles.find(file => file.endsWith('.tar.gz'));
      
      if (!backupFile) {
        throw new Error('No backup file created');
      }
      
      // Restore to original directory
      process.chdir(originalCwd);
      
      spinner.succeed('Backup test completed successfully');
      return { success: true, backupFile: path.join(this.config.testDir, 'test-backups', backupFile) };
    } catch (error) {
      spinner.fail(`Backup test failed: ${error.message}`);
      return { success: false, error: error.message };
    }
  }

  async testRestore(backupFile) {
    if (!this.config.testRestore) {
      this.log('Skipping restore test', 'warning');
      return { success: true, skipped: true };
    }
    
    const spinner = ora('🔄 Testing restore functionality...').start();
    
    try {
      const restoreDir = path.join(this.config.testDir, 'restored');
      await fs.ensureDir(restoreDir);
      
      // Change to restore directory
      const originalCwd = process.cwd();
      process.chdir(restoreDir);
      
      // Restore backup
      const restore = new FlashCoreRestore({
        backupFile: path.join(originalCwd, backupFile),
        restoreDir: 'temp-restore',
        overwrite: true,
        skipDependencies: true
      });
      
      await restore.run();
      
      // Verify restored files
      const criticalFiles = [
        'package.json',
        'src/App.tsx',
        'src/main.tsx',
        'tsconfig.json',
        'vite.config.ts',
        '.env',
        'README.md',
        'supabase/config.toml'
      ];
      
      for (const file of criticalFiles) {
        if (!await fs.pathExists(file)) {
          throw new Error(`Critical file missing after restore: ${file}`);
        }
      }
      
      // Restore to original directory
      process.chdir(originalCwd);
      
      spinner.succeed('Restore test completed successfully');
      return { success: true };
    } catch (error) {
      spinner.fail(`Restore test failed: ${error.message}`);
      return { success: false, error: error.message };
    }
  }

  async testIntegrity(backupFile) {
    if (!this.config.testIntegrity) {
      this.log('Skipping integrity test', 'warning');
      return { success: true, skipped: true };
    }
    
    const spinner = ora('🔍 Testing backup integrity...').start();
    
    try {
      // Test archive integrity
      const { execSync } = await import('child_process');
      execSync(`tar -tzf "${backupFile}" > /dev/null`, { stdio: 'pipe' });
      
      // Test manifest extraction
      const tempDir = path.join(this.config.testDir, 'temp-integrity');
      await fs.ensureDir(tempDir);
      
      execSync(`tar -xzf "${backupFile}" -C "${tempDir}" backup-manifest.json`, { stdio: 'pipe' });
      
      const manifestPath = path.join(tempDir, 'backup-manifest.json');
      if (!await fs.pathExists(manifestPath)) {
        throw new Error('Manifest not found in backup');
      }
      
      const manifest = JSON.parse(await fs.readFile(manifestPath, 'utf8'));
      
      // Validate manifest structure
      const requiredFields = ['project', 'backupDate', 'version', 'files'];
      for (const field of requiredFields) {
        if (!manifest[field]) {
          throw new Error(`Missing required field in manifest: ${field}`);
        }
      }
      
      // Cleanup
      await fs.remove(tempDir);
      
      spinner.succeed('Integrity test completed successfully');
      return { success: true, manifest };
    } catch (error) {
      spinner.fail(`Integrity test failed: ${error.message}`);
      return { success: false, error: error.message };
    }
  }

  async testPerformance(backupFile) {
    const spinner = ora('⚡ Testing backup performance...').start();
    
    try {
      const stats = await fs.stat(backupFile);
      const sizeMB = (stats.size / 1024 / 1024).toFixed(2);
      
      // Calculate compression ratio
      const originalSize = await this.calculateOriginalSize();
      const compressionRatio = ((1 - stats.size / originalSize) * 100).toFixed(1);
      
      spinner.succeed(`Performance test completed: ${sizeMB}MB (${compressionRatio}% compression)`);
      return { 
        success: true, 
        sizeMB: parseFloat(sizeMB), 
        compressionRatio: parseFloat(compressionRatio) 
      };
    } catch (error) {
      spinner.fail(`Performance test failed: ${error.message}`);
      return { success: false, error: error.message };
    }
  }

  async calculateOriginalSize() {
    let totalSize = 0;
    
    const calculateDirSize = async (dir) => {
      const entries = await fs.readdir(dir);
      
      for (const entry of entries) {
        const fullPath = path.join(dir, entry);
        const stat = await fs.stat(fullPath);
        
        if (stat.isDirectory()) {
          await calculateDirSize(fullPath);
        } else {
          totalSize += stat.size;
        }
      }
    };
    
    await calculateDirSize(this.config.testDir);
    return totalSize;
  }

  async cleanup() {
    if (!this.config.cleanup) {
      this.log('Skipping cleanup (use --no-cleanup to preserve test files)', 'warning');
      return;
    }
    
    const spinner = ora('🧹 Cleaning up test files...').start();
    
    try {
      await fs.remove(this.config.testDir);
      spinner.succeed('Cleanup completed successfully');
    } catch (error) {
      spinner.fail(`Cleanup failed: ${error.message}`);
    }
  }

  async runAllTests() {
    this.log('🧪 Starting FlashCore Backup System Tests');
    this.log(`📁 Test directory: ${this.config.testDir}`);
    this.log(`🧹 Cleanup: ${this.config.cleanup ? 'enabled' : 'disabled'}`);
    this.log(`📊 Verbose: ${this.config.verbose ? 'enabled' : 'disabled'}`);
    console.log('');
    
    const startTime = Date.now();
    
    try {
      // Test 1: Create test project
      const testProjectResult = await this.createTestProject();
      this.testResults.push({ test: 'Create Test Project', result: testProjectResult });
      
      if (!testProjectResult) {
        throw new Error('Failed to create test project');
      }
      
      // Test 2: Backup functionality
      const backupResult = await this.testBackup();
      this.testResults.push({ test: 'Backup Functionality', result: backupResult });
      
      if (!backupResult.success) {
        throw new Error(`Backup test failed: ${backupResult.error}`);
      }
      
      // Test 3: Integrity verification
      const integrityResult = await this.testIntegrity(backupResult.backupFile);
      this.testResults.push({ test: 'Integrity Verification', result: integrityResult });
      
      // Test 4: Restore functionality
      const restoreResult = await this.testRestore(backupResult.backupFile);
      this.testResults.push({ test: 'Restore Functionality', result: restoreResult });
      
      // Test 5: Performance metrics
      const performanceResult = await this.testPerformance(backupResult.backupFile);
      this.testResults.push({ test: 'Performance Metrics', result: performanceResult });
      
      // Cleanup
      await this.cleanup();
      
      // Show results
      this.showTestResults(startTime);
      
    } catch (error) {
      this.log(`Test suite failed: ${error.message}`, 'error');
      this.showTestResults(startTime);
      process.exit(1);
    }
  }

  showTestResults(startTime) {
    const duration = ((Date.now() - startTime) / 1000).toFixed(2);
    
    console.log('\n📊 Test Results Summary');
    console.log('=======================');
    
    let passedTests = 0;
    let failedTests = 0;
    let skippedTests = 0;
    
    for (const testResult of this.testResults) {
      const status = testResult.result.success ? '✅ PASS' : 
                    testResult.result.skipped ? '⏭️ SKIP' : '❌ FAIL';
      
      console.log(`${status} ${testResult.test}`);
      
      if (testResult.result.success) {
        passedTests++;
      } else if (testResult.result.skipped) {
        skippedTests++;
      } else {
        failedTests++;
        if (testResult.result.error) {
          console.log(`   Error: ${testResult.result.error}`);
        }
      }
      
      // Show additional info for performance test
      if (testResult.test === 'Performance Metrics' && testResult.result.success) {
        console.log(`   Size: ${testResult.result.sizeMB}MB`);
        console.log(`   Compression: ${testResult.result.compressionRatio}%`);
      }
    }
    
    console.log('\n📈 Summary:');
    console.log(`   Total Tests: ${this.testResults.length}`);
    console.log(`   Passed: ${passedTests}`);
    console.log(`   Failed: ${failedTests}`);
    console.log(`   Skipped: ${skippedTests}`);
    console.log(`   Duration: ${duration} seconds`);
    
    if (failedTests === 0) {
      console.log('\n🎉 All tests passed! Backup system is working correctly.');
    } else {
      console.log('\n⚠️ Some tests failed. Please check the errors above.');
    }
  }
}

// CLI setup
program
  .name('flashcore-test-backup')
  .description('Test FlashCore backup system functionality')
  .version('1.0.0');

program
  .option('-d, --test-dir <path>', 'Test directory', 'test-backup')
  .option('--no-cleanup', 'Preserve test files after testing')
  .option('-v, --verbose', 'Verbose output')
  .option('--no-test-restore', 'Skip restore testing')
  .option('--no-test-integrity', 'Skip integrity testing');

program.parse();

const options = program.opts();

// Run tests
const tester = new BackupTester({
  testDir: options.testDir,
  cleanup: options.cleanup,
  verbose: options.verbose,
  testRestore: options.testRestore,
  testIntegrity: options.testIntegrity
});

tester.runAllTests(); 