#!/usr/bin/env node

import fs from 'fs-extra';
import path from 'path';
import chalk from 'chalk';
import { program } from 'commander';
import { glob } from 'glob';
import { z } from 'zod';
import dotenv from 'dotenv';

// Load environment variables
dotenv.config();

// Status configuration schema
const StatusConfigSchema = z.object({
  backupDir: z.string().default('backups'),
  detailed: z.boolean().default(false),
  format: z.enum(['text', 'json']).default('text')
});

class BackupStatus {
  constructor(options = {}) {
    this.config = StatusConfigSchema.parse(options);
  }

  log(message, type = 'info') {
    const colors = {
      info: chalk.blue,
      success: chalk.green,
      error: chalk.red,
      warning: chalk.yellow,
      debug: chalk.gray
    };
    
    const prefix = {
      info: 'ℹ️',
      success: '✅',
      error: '❌',
      warning: '⚠️',
      debug: '🔍'
    }[type];
    
    console.log(`${colors[type](`${prefix} ${message}`)}`);
  }

  async getBackupFiles() {
    try {
      const backupFiles = await glob(path.join(this.config.backupDir, '*.tar.gz'));
      
      const backups = [];
      for (const file of backupFiles) {
        const stat = await fs.stat(file);
        const size = (stat.size / 1024 / 1024).toFixed(2);
        
        backups.push({
          name: path.basename(file),
          path: file,
          size: stat.size,
          sizeMB: parseFloat(size),
          modified: stat.mtime,
          created: stat.birthtime
        });
      }
      
      // Sort by modification time (newest first)
      backups.sort((a, b) => b.modified.getTime() - a.modified.getTime());
      
      return backups;
    } catch (error) {
      throw new Error(`Failed to get backup files: ${error.message}`);
    }
  }

  async getBackupInfo(backupFile) {
    try {
      // Extract manifest from backup
      const tempDir = path.join(process.cwd(), 'temp-extract');
      await fs.ensureDir(tempDir);
      
      const { execSync } = await import('child_process');
      execSync(`tar -xzf "${backupFile}" -C "${tempDir}" backup-manifest.json`, { 
        stdio: 'pipe' 
      });
      
      const manifestPath = path.join(tempDir, 'backup-manifest.json');
      if (await fs.pathExists(manifestPath)) {
        const manifest = JSON.parse(await fs.readFile(manifestPath, 'utf8'));
        await fs.remove(tempDir);
        return manifest;
      }
      
      await fs.remove(tempDir);
      return null;
    } catch (error) {
      // Clean up temp directory if it exists
      try {
        await fs.remove(path.join(process.cwd(), 'temp-extract'));
      } catch {}
      return null;
    }
  }

  async getDiskUsage() {
    try {
      const { execSync } = await import('child_process');
      
      if (process.platform === 'win32') {
        const output = execSync('wmic logicaldisk get size,freespace,caption', { encoding: 'utf8' });
        const lines = output.trim().split('\n').slice(1);
        const disks = [];
        
        for (const line of lines) {
          const parts = line.trim().split(/\s+/);
          if (parts.length >= 3) {
            const caption = parts[0];
            const freeSpace = parseInt(parts[1]);
            const size = parseInt(parts[2]);
            const usedSpace = size - freeSpace;
            const usagePercent = ((usedSpace / size) * 100).toFixed(1);
            
            disks.push({
              drive: caption,
              total: size,
              used: usedSpace,
              free: freeSpace,
              usagePercent: parseFloat(usagePercent)
            });
          }
        }
        
        return disks;
      } else {
        const output = execSync('df -h', { encoding: 'utf8' });
        const lines = output.trim().split('\n').slice(1);
        const disks = [];
        
        for (const line of lines) {
          const parts = line.trim().split(/\s+/);
          if (parts.length >= 5) {
            const filesystem = parts[0];
            const size = parts[1];
            const used = parts[2];
            const available = parts[3];
            const usagePercent = parts[4].replace('%', '');
            
            disks.push({
              filesystem,
              size,
              used,
              available,
              usagePercent: parseFloat(usagePercent)
            });
          }
        }
        
        return disks;
      }
    } catch (error) {
      this.log(`Warning: Could not get disk usage: ${error.message}`, 'warning');
      return [];
    }
  }

  async getSystemInfo() {
    const systemInfo = {
      platform: process.platform,
      arch: process.arch,
      nodeVersion: process.version,
      cwd: process.cwd(),
      uptime: process.uptime(),
      memory: process.memoryUsage(),
      pid: process.pid
    };
    
    return systemInfo;
  }

  formatBytes(bytes) {
    if (bytes === 0) return '0 Bytes';
    const k = 1024;
    const sizes = ['Bytes', 'KB', 'MB', 'GB', 'TB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
  }

  formatDate(date) {
    return new Date(date).toLocaleString();
  }

  async showTextStatus() {
    console.log('\n📊 FlashCore Backup System Status');
    console.log('==================================');
    
    // System Information
    const systemInfo = await this.getSystemInfo();
    console.log('\n🖥️  System Information:');
    console.log(`   Platform: ${systemInfo.platform} ${systemInfo.arch}`);
    console.log(`   Node.js: ${systemInfo.nodeVersion}`);
    console.log(`   Working Directory: ${systemInfo.cwd}`);
    console.log(`   Process ID: ${systemInfo.pid}`);
    console.log(`   Uptime: ${(systemInfo.uptime / 3600).toFixed(2)} hours`);
    
    // Disk Usage
    const diskUsage = await this.getDiskUsage();
    if (diskUsage.length > 0) {
      console.log('\n💾 Disk Usage:');
      for (const disk of diskUsage) {
        if (disk.filesystem || disk.drive) {
          const name = disk.filesystem || disk.drive;
          const usage = disk.usagePercent || 0;
          const status = usage > 90 ? '🔴' : usage > 70 ? '🟡' : '🟢';
          console.log(`   ${status} ${name}: ${usage}% used`);
        }
      }
    }
    
    // Backup Files
    try {
      const backups = await this.getBackupFiles();
      
      console.log('\n📦 Backup Files:');
      if (backups.length === 0) {
        console.log('   No backup files found');
      } else {
        console.log(`   Total backups: ${backups.length}`);
        
        const totalSize = backups.reduce((sum, backup) => sum + backup.size, 0);
        console.log(`   Total size: ${this.formatBytes(totalSize)}`);
        
        if (backups.length > 0) {
          const latest = backups[0];
          console.log(`   Latest backup: ${latest.name}`);
          console.log(`   Latest size: ${this.formatBytes(latest.size)}`);
          console.log(`   Latest date: ${this.formatDate(latest.modified)}`);
        }
        
        if (this.config.detailed && backups.length > 0) {
          console.log('\n📋 Detailed Backup List:');
          for (let i = 0; i < Math.min(backups.length, 10); i++) {
            const backup = backups[i];
            console.log(`   ${i + 1}. ${backup.name}`);
            console.log(`      Size: ${this.formatBytes(backup.size)}`);
            console.log(`      Modified: ${this.formatDate(backup.modified)}`);
            
            // Try to get manifest info
            const manifest = await this.getBackupInfo(backup.path);
            if (manifest) {
              console.log(`      Version: ${manifest.version}`);
              console.log(`      Files: ${manifest.files?.length || 'Unknown'}`);
            }
            console.log('');
          }
          
          if (backups.length > 10) {
            console.log(`   ... and ${backups.length - 10} more backups`);
          }
        }
      }
    } catch (error) {
      console.log(`   Error reading backups: ${error.message}`);
    }
    
    // Configuration
    console.log('\n⚙️  Configuration:');
    console.log(`   Backup Directory: ${this.config.backupDir}`);
    console.log(`   Detailed Mode: ${this.config.detailed ? 'Yes' : 'No'}`);
    console.log(`   Output Format: ${this.config.format}`);
    
    console.log('\n');
  }

  async showJsonStatus() {
    const status = {
      timestamp: new Date().toISOString(),
      system: await this.getSystemInfo(),
      diskUsage: await this.getDiskUsage(),
      backups: await this.getBackupFiles(),
      config: this.config
    };
    
    // Add manifest info for each backup if detailed
    if (this.config.detailed) {
      for (const backup of status.backups) {
        backup.manifest = await this.getBackupInfo(backup.path);
      }
    }
    
    console.log(JSON.stringify(status, null, 2));
  }

  async run() {
    try {
      if (this.config.format === 'json') {
        await this.showJsonStatus();
      } else {
        await this.showTextStatus();
      }
    } catch (error) {
      this.log(`Status check failed: ${error.message}`, 'error');
      process.exit(1);
    }
  }
}

// CLI setup
program
  .name('flashcore-status')
  .description('Show FlashCore backup system status')
  .version('1.0.0');

program
  .option('-d, --detailed', 'Show detailed information')
  .option('-f, --format <format>', 'Output format (text|json)', 'text')
  .option('--backup-dir <path>', 'Backup directory', 'backups');

program.parse();

const options = program.opts();

// Run status check
const status = new BackupStatus({
  detailed: options.detailed,
  format: options.format,
  backupDir: options.backupDir
});

status.run(); 