#!/usr/bin/env node

import fs from 'fs-extra';
import path from 'path';
import cron from 'node-cron';
import chalk from 'chalk';
import ora from 'ora';
import { program } from 'commander';
import { z } from 'zod';
import dotenv from 'dotenv';
import FlashCoreBackup from './backup.js';

// Load environment variables
dotenv.config();

// Schedule configuration schema
const ScheduleConfigSchema = z.object({
  schedule: z.string().default('0 2 * * *'), // Daily at 2 AM
  backupDir: z.string().default('backups'),
  maxBackups: z.number().default(10),
  compressionLevel: z.number().min(1).max(9).default(9),
  enabled: z.boolean().default(true),
  logFile: z.string().default('backup-scheduler.log')
});

class BackupScheduler {
  constructor(options = {}) {
    this.config = ScheduleConfigSchema.parse({
      ...this.loadScheduleConfig(),
      ...options
    });
    
    this.isRunning = false;
    this.lastBackup = null;
    this.nextBackup = null;
    this.backupCount = 0;
    this.errorCount = 0;
  }

  loadScheduleConfig() {
    try {
      const configPath = path.join(process.cwd(), 'backup-schedule.json');
      if (fs.existsSync(configPath)) {
        return JSON.parse(fs.readFileSync(configPath, 'utf8'));
      }
    } catch (error) {
      this.log(`Warning: Could not load schedule config: ${error.message}`, 'warning');
    }
    return {};
  }

  log(message, type = 'info') {
    const timestamp = new Date().toISOString();
    const colors = {
      info: chalk.blue,
      success: chalk.green,
      error: chalk.red,
      warning: chalk.yellow,
      debug: chalk.gray
    };
    
    const prefix = {
      info: 'ℹ️',
      success: '✅',
      error: '❌',
      warning: '⚠️',
      debug: '🔍'
    }[type];
    
    const logMessage = `${prefix} [${timestamp}] ${message}`;
    console.log(`${colors[type](logMessage)}`);
    
    // Write to log file
    this.writeToLog(logMessage);
  }

  writeToLog(message) {
    try {
      const logPath = path.join(process.cwd(), this.config.logFile);
      fs.appendFileSync(logPath, message + '\n');
    } catch (error) {
      console.error(`Failed to write to log file: ${error.message}`);
    }
  }

  async performBackup() {
    if (this.isRunning) {
      this.log('Backup already in progress, skipping...', 'warning');
      return;
    }

    this.isRunning = true;
    const startTime = Date.now();
    
    try {
      this.log('🚀 Starting scheduled backup...');
      
      const backup = new FlashCoreBackup({
        backupDir: this.config.backupDir,
        maxBackups: this.config.maxBackups,
        compressionLevel: this.config.compressionLevel
      });
      
      await backup.run();
      
      this.lastBackup = new Date();
      this.backupCount++;
      this.errorCount = 0;
      
      const duration = ((Date.now() - startTime) / 1000).toFixed(2);
      this.log(`✅ Scheduled backup completed successfully in ${duration} seconds`);
      
    } catch (error) {
      this.errorCount++;
      this.log(`❌ Scheduled backup failed: ${error.message}`, 'error');
    } finally {
      this.isRunning = false;
    }
  }

  startScheduler() {
    if (!this.config.enabled) {
      this.log('Scheduler is disabled in configuration', 'warning');
      return;
    }

    this.log('🕐 Starting backup scheduler...');
    this.log(`📅 Schedule: ${this.config.schedule}`);
    this.log(`📁 Backup directory: ${this.config.backupDir}`);
    this.log(`🗜️ Compression level: ${this.config.compressionLevel}`);
    this.log(`📊 Max backups: ${this.config.maxBackups}`);
    
    // Validate cron expression
    if (!cron.validate(this.config.schedule)) {
      this.log(`Invalid cron expression: ${this.config.schedule}`, 'error');
      process.exit(1);
    }

    // Schedule the backup task
    cron.schedule(this.config.schedule, async () => {
      await this.performBackup();
    }, {
      scheduled: true,
      timezone: "UTC"
    });

    // Calculate next backup time
    this.calculateNextBackup();
    
    this.log('✅ Backup scheduler started successfully');
    this.log(`⏰ Next backup: ${this.nextBackup}`);
    
    // Keep the process running
    this.keepAlive();
  }

  calculateNextBackup() {
    try {
      const now = new Date();
      const next = cron.getNextDate(this.config.schedule, now);
      this.nextBackup = next.toISOString();
    } catch (error) {
      this.log(`Warning: Could not calculate next backup time: ${error.message}`, 'warning');
    }
  }

  keepAlive() {
    // Log status every hour
    setInterval(() => {
      this.log(`📊 Scheduler Status - Backups: ${this.backupCount}, Errors: ${this.errorCount}`);
      if (this.nextBackup) {
        this.log(`⏰ Next scheduled backup: ${this.nextBackup}`);
      }
    }, 60 * 60 * 1000); // Every hour

    // Handle graceful shutdown
    process.on('SIGINT', () => {
      this.log('🛑 Received SIGINT, shutting down scheduler...');
      this.stopScheduler();
      process.exit(0);
    });

    process.on('SIGTERM', () => {
      this.log('🛑 Received SIGTERM, shutting down scheduler...');
      this.stopScheduler();
      process.exit(0);
    });
  }

  stopScheduler() {
    this.log('🛑 Stopping backup scheduler...');
    // Cleanup any running tasks
    if (this.isRunning) {
      this.log('⚠️  Backup in progress, waiting for completion...');
    }
  }

  async getStatus() {
    const status = {
      enabled: this.config.enabled,
      schedule: this.config.schedule,
      lastBackup: this.lastBackup,
      nextBackup: this.nextBackup,
      backupCount: this.backupCount,
      errorCount: this.errorCount,
      isRunning: this.isRunning,
      config: this.config
    };

    return status;
  }

  async showStatus() {
    const status = await this.getStatus();
    
    console.log('\n📊 Backup Scheduler Status');
    console.log('==========================');
    console.log(`Enabled: ${status.enabled ? '✅ Yes' : '❌ No'}`);
    console.log(`Schedule: ${status.schedule}`);
    console.log(`Last Backup: ${status.lastBackup || 'Never'}`);
    console.log(`Next Backup: ${status.nextBackup || 'Unknown'}`);
    console.log(`Total Backups: ${status.backupCount}`);
    console.log(`Errors: ${status.errorCount}`);
    console.log(`Currently Running: ${status.isRunning ? '✅ Yes' : '❌ No'}`);
    console.log(`Backup Directory: ${status.config.backupDir}`);
    console.log(`Max Backups: ${status.config.maxBackups}`);
    console.log(`Compression Level: ${status.config.compressionLevel}`);
    console.log('');
  }
}

// CLI setup
program
  .name('flashcore-scheduler')
  .description('Automated backup scheduler for FlashCore Pulse Dashboard')
  .version('1.0.0');

program
  .command('start')
  .description('Start the backup scheduler')
  .option('-s, --schedule <cron>', 'Cron schedule expression', '0 2 * * *')
  .option('-d, --backup-dir <path>', 'Backup directory', 'backups')
  .option('-m, --max-backups <number>', 'Maximum number of backups', '10')
  .option('-c, --compression <level>', 'Compression level (1-9)', '9')
  .option('--log-file <path>', 'Log file path', 'backup-scheduler.log')
  .action(async (options) => {
    const scheduler = new BackupScheduler({
      schedule: options.schedule,
      backupDir: options.backupDir,
      maxBackups: parseInt(options.maxBackups),
      compressionLevel: parseInt(options.compression),
      logFile: options.logFile
    });
    
    scheduler.startScheduler();
  });

program
  .command('status')
  .description('Show scheduler status')
  .action(async () => {
    const scheduler = new BackupScheduler();
    await scheduler.showStatus();
  });

program
  .command('backup-now')
  .description('Trigger an immediate backup')
  .action(async () => {
    const scheduler = new BackupScheduler();
    await scheduler.performBackup();
  });

program
  .command('install')
  .description('Install scheduler as a system service')
  .option('--user <username>', 'User to run the service as')
  .action(async (options) => {
    console.log('🔧 Installing backup scheduler as system service...');
    console.log('This feature is not yet implemented.');
    console.log('For now, you can run the scheduler manually or use a process manager like PM2.');
  });

program.parse();

// Run if no command specified
if (!process.argv.slice(2).length) {
  program.outputHelp();
}

export default BackupScheduler; 