#!/usr/bin/env node

import fs from 'fs-extra';
import path from 'path';
import archiver from 'archiver';
import chalk from 'chalk';
import ora from 'ora';
import { program } from 'commander';
import { glob } from 'glob';
import { z } from 'zod';
import dotenv from 'dotenv';
import readline from 'readline';

// Load environment variables
dotenv.config();

// Configuration schema
const RestoreConfigSchema = z.object({
  backupFile: z.string(),
  restoreDir: z.string().default('restored-project'),
  overwrite: z.boolean().default(false),
  interactive: z.boolean().default(false),
  skipDependencies: z.boolean().default(false),
  skipDatabase: z.boolean().default(false)
});

class FlashCoreRestore {
  constructor(options = {}) {
    this.config = RestoreConfigSchema.parse(options);
    this.rl = readline.createInterface({
      input: process.stdin,
      output: process.stdout
    });
  }

  log(message, type = 'info') {
    const timestamp = new Date().toISOString();
    const colors = {
      info: chalk.blue,
      success: chalk.green,
      error: chalk.red,
      warning: chalk.yellow,
      debug: chalk.gray
    };
    
    const prefix = {
      info: 'ℹ️',
      success: '✅',
      error: '❌',
      warning: '⚠️',
      debug: '🔍'
    }[type];
    
    console.log(`${colors[type](`${prefix} [${timestamp}] ${message}`)}`);
  }

  async question(prompt) {
    return new Promise((resolve) => {
      this.rl.question(prompt, resolve);
    });
  }

  async closeReadline() {
    this.rl.close();
  }

  async findLatestBackup() {
    try {
      const backupFiles = await glob('backups/*.tar.gz');
      
      if (backupFiles.length === 0) {
        throw new Error('No backup files found in backups/ directory');
      }
      
      // Sort by modification time (newest first)
      backupFiles.sort((a, b) => {
        const statA = fs.statSync(a);
        const statB = fs.statSync(b);
        return statB.mtime.getTime() - statA.mtime.getTime();
      });
      
      return backupFiles[0];
    } catch (error) {
      throw new Error(`Failed to find latest backup: ${error.message}`);
    }
  }

  async extractBackup() {
    const spinner = ora('📦 Extracting backup...').start();
    
    return new Promise((resolve, reject) => {
      const input = fs.createReadStream(this.config.backupFile);
      const extract = archiver('tar', { gzip: true });
      
      extract.on('error', (err) => {
        spinner.fail(`Failed to extract backup: ${err.message}`);
        reject(err);
      });
      
      extract.on('end', () => {
        spinner.succeed('Backup extracted successfully');
        resolve();
      });
      
      input.pipe(extract);
      extract.extractTo(this.config.restoreDir);
    });
  }

  async findBackupDirectory() {
    const entries = await fs.readdir(this.config.restoreDir);
    const backupDirs = entries.filter(entry => 
      entry.startsWith('flashcore-pulse-dashboard_') && 
      (await fs.stat(path.join(this.config.restoreDir, entry))).isDirectory()
    );
    
    if (backupDirs.length === 0) {
      throw new Error('No backup directory found');
    }
    
    return path.join(this.config.restoreDir, backupDirs[0]);
  }

  async readManifest(backupDir) {
    try {
      const manifestPath = path.join(backupDir, 'backup-manifest.json');
      if (await fs.pathExists(manifestPath)) {
        const manifest = JSON.parse(await fs.readFile(manifestPath, 'utf8'));
        return manifest;
      }
    } catch (error) {
      this.log(`Warning: Could not read backup manifest: ${error.message}`, 'warning');
    }
    return null;
  }

  async showBackupInfo(manifest) {
    if (manifest) {
      this.log('📋 Backup Information:');
      this.log(`   Project: ${manifest.project}`);
      this.log(`   Backup Date: ${manifest.backupDate}`);
      this.log(`   Version: ${manifest.version}`);
      this.log(`   Files: ${manifest.files?.length || 'Unknown'}`);
      this.log(`   System: ${manifest.system?.platform} ${manifest.system?.arch}`);
      this.log('');
    }
  }

  async checkExistingFiles() {
    const criticalFiles = ['package.json', 'src', 'public'];
    const existingFiles = [];
    
    for (const file of criticalFiles) {
      if (await fs.pathExists(file)) {
        existingFiles.push(file);
      }
    }
    
    if (existingFiles.length > 0 && !this.config.overwrite) {
      this.log(`⚠️  Found existing files: ${existingFiles.join(', ')}`, 'warning');
      
      if (this.config.interactive) {
        const answer = await this.question('Do you want to overwrite existing files? (y/N): ');
        if (answer.toLowerCase() !== 'y' && answer.toLowerCase() !== 'yes') {
          throw new Error('Restore cancelled by user');
        }
      } else {
        throw new Error('Existing files found. Use --overwrite flag or --interactive mode');
      }
    }
  }

  async restoreFiles(backupDir) {
    const spinner = ora('🔄 Restoring files...').start();
    
    try {
      // Copy all files from backup to current directory
      const copyRecursive = async (src, dest) => {
        const stat = await fs.stat(src);
        if (stat.isDirectory()) {
          await fs.ensureDir(dest);
          const files = await fs.readdir(src);
          for (const file of files) {
            await copyRecursive(path.join(src, file), path.join(dest, file));
          }
        } else {
          await fs.copy(src, dest);
        }
      };
      
      const entries = await fs.readdir(backupDir);
      let copiedCount = 0;
      
      for (const entry of entries) {
        if (entry !== 'restore.sh' && entry !== 'backup-manifest.json') {
          const srcPath = path.join(backupDir, entry);
          const destPath = path.join('.', entry);
          await copyRecursive(srcPath, destPath);
          copiedCount++;
        }
      }
      
      spinner.succeed(`Files restored: ${copiedCount} items`);
      return copiedCount;
    } catch (error) {
      spinner.fail(`Failed to restore files: ${error.message}`);
      throw error;
    }
  }

  async installDependencies() {
    if (this.config.skipDependencies) {
      this.log('⏭️  Skipping dependency installation', 'warning');
      return;
    }
    
    const spinner = ora('📦 Installing dependencies...').start();
    
    try {
      const { execSync } = await import('child_process');
      execSync('npm install', { stdio: 'inherit' });
      spinner.succeed('Dependencies installed successfully');
    } catch (error) {
      spinner.fail(`Failed to install dependencies: ${error.message}`);
      throw error;
    }
  }

  async setupEnvironment() {
    const spinner = ora('🔐 Setting up environment...').start();
    
    try {
      // Check for environment files
      const envFiles = await glob('.env*');
      
      if (envFiles.length > 0) {
        this.log('📝 Environment files found:');
        for (const envFile of envFiles) {
          this.log(`   - ${envFile}`);
        }
        
        if (this.config.interactive) {
          this.log('⚠️  Please review and update your environment files with correct values');
        }
        
        spinner.succeed('Environment files restored');
      } else {
        spinner.succeed('No environment files found (this is normal for new installations)');
      }
    } catch (error) {
      spinner.fail(`Environment setup failed: ${error.message}`);
    }
  }

  async setupDatabase() {
    if (this.config.skipDatabase) {
      this.log('⏭️  Skipping database setup', 'warning');
      return;
    }
    
    const spinner = ora('🗄️ Setting up database...').start();
    
    try {
      if (await fs.pathExists('database-backup.sh')) {
        this.log('📋 Database backup script found');
        this.log('   Run: bash database-backup.sh for instructions');
        spinner.succeed('Database setup instructions available');
      } else {
        spinner.succeed('No database backup found (this is normal for new installations)');
      }
    } catch (error) {
      spinner.fail(`Database setup failed: ${error.message}`);
    }
  }

  async verifyRestore() {
    const spinner = ora('🔍 Verifying restore...').start();
    
    try {
      const criticalFiles = [
        'package.json',
        'src/App.tsx',
        'src/main.tsx',
        'index.html'
      ];
      
      const missingFiles = [];
      
      for (const file of criticalFiles) {
        if (!await fs.pathExists(file)) {
          missingFiles.push(file);
        }
      }
      
      if (missingFiles.length > 0) {
        spinner.fail(`Missing critical files: ${missingFiles.join(', ')}`);
        throw new Error('Restore verification failed');
      }
      
      spinner.succeed('Restore verification passed');
      return true;
    } catch (error) {
      spinner.fail(`Verification failed: ${error.message}`);
      throw error;
    }
  }

  async showRestoreInstructions(manifest) {
    this.log('🎉 Restore completed successfully!');
    this.log('');
    this.log('📋 Next steps:');
    this.log('   1. Update your .env file with correct values');
    this.log('   2. Configure your database connection');
    this.log('   3. Run: npm run dev');
    this.log('   4. Test all functionality');
    this.log('');
    
    if (manifest) {
      this.log('📊 Restore Summary:');
      this.log(`   Original backup date: ${manifest.backupDate}`);
      this.log(`   Project version: ${manifest.version}`);
      this.log(`   Files restored: ${manifest.files?.length || 'Unknown'}`);
    }
  }

  async cleanup() {
    const spinner = ora('🧹 Cleaning up temporary files...').start();
    
    try {
      await fs.remove(this.config.restoreDir);
      spinner.succeed('Cleanup completed');
    } catch (error) {
      spinner.fail(`Warning: Could not clean up temporary files: ${error.message}`);
    }
  }

  async run() {
    const startTime = Date.now();
    
    try {
      // Determine backup file
      if (!this.config.backupFile) {
        if (this.config.interactive) {
          const answer = await this.question('No backup file specified. Use latest backup? (Y/n): ');
          if (answer.toLowerCase() !== 'n' && answer.toLowerCase() !== 'no') {
            this.config.backupFile = await this.findLatestBackup();
          } else {
            throw new Error('Please specify a backup file');
          }
        } else {
          this.config.backupFile = await this.findLatestBackup();
        }
      }

      if (!await fs.pathExists(this.config.backupFile)) {
        throw new Error(`Backup file not found: ${this.config.backupFile}`);
      }

      this.log('🔄 Starting FlashCore restore...');
      this.log(`📦 Backup file: ${this.config.backupFile}`);
      
      // Check existing files
      await this.checkExistingFiles();
      
      // Extract backup
      await this.extractBackup();
      
      // Find backup directory
      const backupDir = await this.findBackupDirectory();
      this.log(`📁 Found backup directory: ${backupDir}`);
      
      // Read manifest
      const manifest = await this.readManifest(backupDir);
      await this.showBackupInfo(manifest);
      
      // Restore files
      await this.restoreFiles(backupDir);
      
      // Install dependencies
      await this.installDependencies();
      
      // Setup environment
      await this.setupEnvironment();
      
      // Setup database
      await this.setupDatabase();
      
      // Verify restore
      await this.verifyRestore();
      
      // Show instructions
      await this.showRestoreInstructions(manifest);
      
      // Cleanup
      await this.cleanup();
      
      const duration = ((Date.now() - startTime) / 1000).toFixed(2);
      this.log(`⏱️  Restore completed in ${duration} seconds`);
      
    } catch (error) {
      this.log(`Restore failed: ${error.message}`, 'error');
      process.exit(1);
    } finally {
      await this.closeReadline();
    }
  }
}

// CLI setup
program
  .name('flashcore-restore')
  .description('Restore FlashCore Pulse Dashboard from backup')
  .version('1.0.0');

program
  .argument('[backup-file]', 'Backup file to restore from')
  .option('-i, --interactive', 'Interactive mode')
  .option('-o, --overwrite', 'Overwrite existing files')
  .option('--skip-dependencies', 'Skip dependency installation')
  .option('--skip-database', 'Skip database setup')
  .option('--restore-dir <path>', 'Temporary restore directory', 'restored-project');

program.parse();

const options = program.opts();
const backupFile = program.args[0];

// Run restore if this script is executed directly
if (import.meta.url === `file://${process.argv[1]}`) {
  const restore = new FlashCoreRestore({
    backupFile,
    interactive: options.interactive,
    overwrite: options.overwrite,
    skipDependencies: options.skipDependencies,
    skipDatabase: options.skipDatabase,
    restoreDir: options.restoreDir
  });
  restore.run();
}

export default FlashCoreRestore; 