#!/usr/bin/env node

const fs = require('fs-extra');
const path = require('path');
const { execSync } = require('child_process');
const chalk = require('chalk');
const ora = require('ora');
const inquirer = require('inquirer');

class BackupSystemInstaller {
  constructor() {
    this.projectRoot = path.resolve(__dirname, '../..');
    this.backupSystemDir = __dirname;
    this.spinner = null;
  }

  log(message, type = 'info') {
    const timestamp = new Date().toISOString();
    const colors = {
      info: chalk.blue,
      success: chalk.green,
      warning: chalk.yellow,
      error: chalk.red
    };
    console.log(`${colors[type](`[${timestamp}]`)} ${message}`);
  }

  async checkPrerequisites() {
    this.log('Checking prerequisites...', 'info');
    
    // Check Node.js version
    try {
      const nodeVersion = process.version;
      const majorVersion = parseInt(nodeVersion.slice(1).split('.')[0]);
      if (majorVersion < 16) {
        throw new Error(`Node.js version ${nodeVersion} is not supported. Please use Node.js 16 or higher.`);
      }
      this.log(`✓ Node.js ${nodeVersion} detected`, 'success');
    } catch (error) {
      this.log(`✗ Node.js check failed: ${error.message}`, 'error');
      return false;
    }

    // Check npm
    try {
      const npmVersion = execSync('npm --version', { encoding: 'utf8' }).trim();
      this.log(`✓ npm ${npmVersion} detected`, 'success');
    } catch (error) {
      this.log('✗ npm not found', 'error');
      return false;
    }

    // Check if backup-system directory exists
    if (!fs.existsSync(this.backupSystemDir)) {
      this.log('✗ Backup system directory not found', 'error');
      return false;
    }

    this.log('✓ All prerequisites met', 'success');
    return true;
  }

  async installDependencies() {
    this.log('Installing backup system dependencies...', 'info');
    
    try {
      const packageJsonPath = path.join(this.backupSystemDir, 'package.json');
      if (!fs.existsSync(packageJsonPath)) {
        throw new Error('package.json not found in backup system directory');
      }

      this.spinner = ora('Installing dependencies...').start();
      execSync('npm install', { 
        cwd: this.backupSystemDir, 
        stdio: 'pipe' 
      });
      this.spinner.succeed('Dependencies installed successfully');
      return true;
    } catch (error) {
      if (this.spinner) this.spinner.fail('Failed to install dependencies');
      this.log(`Error installing dependencies: ${error.message}`, 'error');
      return false;
    }
  }

  async setupConfiguration() {
    this.log('Setting up configuration files...', 'info');

    try {
      // Copy example environment file
      const envExamplePath = path.join(this.backupSystemDir, '.env.example');
      const envPath = path.join(this.backupSystemDir, '.env');
      
      if (!fs.existsSync(envPath) && fs.existsSync(envExamplePath)) {
        fs.copyFileSync(envExamplePath, envPath);
        this.log('✓ Created .env file from template', 'success');
      }

      // Create backup directory
      const backupDir = path.join(this.projectRoot, 'backups');
      if (!fs.existsSync(backupDir)) {
        fs.mkdirSync(backupDir, { recursive: true });
        this.log('✓ Created backups directory', 'success');
      }

      // Create logs directory
      const logsDir = path.join(this.backupSystemDir, 'logs');
      if (!fs.existsSync(logsDir)) {
        fs.mkdirSync(logsDir, { recursive: true });
        this.log('✓ Created logs directory', 'success');
      }

      return true;
    } catch (error) {
      this.log(`Error setting up configuration: ${error.message}`, 'error');
      return false;
    }
  }

  async setupPermissions() {
    this.log('Setting up file permissions...', 'info');

    try {
      // Make scripts executable on Unix-like systems
      if (process.platform !== 'win32') {
        const scripts = [
          'backup.js',
          'restore.js',
          'scheduler.js',
          'status.js',
          'cleanup.js',
          'verify.js',
          'test-backup.js',
          'help.js'
        ];

        for (const script of scripts) {
          const scriptPath = path.join(this.backupSystemDir, script);
          if (fs.existsSync(scriptPath)) {
            fs.chmodSync(scriptPath, '755');
          }
        }
        this.log('✓ Set executable permissions on scripts', 'success');
      }

      return true;
    } catch (error) {
      this.log(`Error setting up permissions: ${error.message}`, 'error');
      return false;
    }
  }

  async testInstallation() {
    this.log('Testing installation...', 'info');

    try {
      this.spinner = ora('Running installation test...').start();
      
      // Test backup system help command
      execSync('node help.js', { 
        cwd: this.backupSystemDir, 
        stdio: 'pipe',
        timeout: 10000 
      });
      
      this.spinner.succeed('Installation test passed');
      return true;
    } catch (error) {
      if (this.spinner) this.spinner.fail('Installation test failed');
      this.log(`Error testing installation: ${error.message}`, 'error');
      return false;
    }
  }

  async showPostInstallInstructions() {
    console.log('\n' + chalk.green.bold('🎉 Backup System Installation Complete!'));
    console.log('\n' + chalk.cyan.bold('Next Steps:'));
    console.log(chalk.white('1. Configure your environment variables:'));
    console.log(chalk.gray(`   cd ${this.backupSystemDir}`));
    console.log(chalk.gray('   cp .env.example .env'));
    console.log(chalk.gray('   # Edit .env with your settings'));
    
    console.log(chalk.white('\n2. Test the backup system:'));
    console.log(chalk.gray('   npm run backup:test'));
    
    console.log(chalk.white('\n3. Create your first backup:'));
    console.log(chalk.gray('   npm run backup'));
    
    console.log(chalk.white('\n4. View available commands:'));
    console.log(chalk.gray('   npm run backup:help'));
    
    console.log(chalk.white('\n5. Set up scheduled backups:'));
    console.log(chalk.gray('   npm run backup:scheduler'));
    
    console.log('\n' + chalk.yellow.bold('📚 Documentation:'));
    console.log(chalk.gray('   Read README.md for detailed usage instructions'));
    
    console.log('\n' + chalk.blue.bold('🆘 Support:'));
    console.log(chalk.gray('   Run "npm run backup:help" for command reference'));
  }

  async run() {
    console.log(chalk.cyan.bold('\n🚀 FlashCore Backup System Installer\n'));
    
    try {
      // Check prerequisites
      if (!(await this.checkPrerequisites())) {
        process.exit(1);
      }

      // Install dependencies
      if (!(await this.installDependencies())) {
        process.exit(1);
      }

      // Setup configuration
      if (!(await this.setupConfiguration())) {
        process.exit(1);
      }

      // Setup permissions
      if (!(await this.setupPermissions())) {
        process.exit(1);
      }

      // Test installation
      if (!(await this.testInstallation())) {
        process.exit(1);
      }

      // Show post-install instructions
      await this.showPostInstallInstructions();

    } catch (error) {
      this.log(`Installation failed: ${error.message}`, 'error');
      process.exit(1);
    }
  }
}

// Run installer if this script is executed directly
if (require.main === module) {
  const installer = new BackupSystemInstaller();
  installer.run();
}

module.exports = BackupSystemInstaller; 