#!/usr/bin/env node

import chalk from 'chalk';
import { program } from 'commander';

class BackupHelp {
  constructor() {
    this.setupCommands();
  }

  setupCommands() {
    program
      .name('flashcore-backup-help')
      .description('FlashCore Backup System - Complete Help Guide')
      .version('1.0.0');

    program
      .command('overview')
      .description('Show system overview')
      .action(() => this.showOverview());

    program
      .command('backup')
      .description('Backup commands help')
      .action(() => this.showBackupHelp());

    program
      .command('restore')
      .description('Restore commands help')
      .action(() => this.showRestoreHelp());

    program
      .command('scheduler')
      .description('Scheduler commands help')
      .action(() => this.showSchedulerHelp());

    program
      .command('maintenance')
      .description('Maintenance commands help')
      .action(() => this.showMaintenanceHelp());

    program
      .command('examples')
      .description('Show usage examples')
      .action(() => this.showExamples());

    program
      .command('config')
      .description('Configuration help')
      .action(() => this.showConfigHelp());

    program.parse();

    // Show default help if no command specified
    if (!process.argv.slice(2).length) {
      this.showMainHelp();
    }
  }

  showMainHelp() {
    console.log(chalk.blue.bold('\n🚀 FlashCore Backup System - Complete Help Guide'));
    console.log(chalk.gray('==================================================\n'));

    console.log(chalk.yellow('📋 Available Commands:'));
    console.log('  overview     - System overview and features');
    console.log('  backup       - Backup commands and options');
    console.log('  restore      - Restore commands and options');
    console.log('  scheduler    - Automated scheduling commands');
    console.log('  maintenance  - Status, cleanup, and verification');
    console.log('  examples     - Usage examples and scenarios');
    console.log('  config       - Configuration options');
    console.log('');

    console.log(chalk.yellow('💡 Quick Start:'));
    console.log('  npm run backup:quick          # Quick backup');
    console.log('  npm run restore:latest        # Restore latest backup');
    console.log('  npm run backup:status         # Check backup status');
    console.log('');

    console.log(chalk.yellow('🔧 Direct Script Usage:'));
    console.log('  node scripts/backup-system/backup.js');
    console.log('  node scripts/backup-system/restore.js');
    console.log('  node scripts/backup-system/status.js');
    console.log('');

    console.log(chalk.cyan('For detailed help on any command:'));
    console.log('  node scripts/backup-system/help.js <command>');
    console.log('');
  }

  showOverview() {
    console.log(chalk.blue.bold('\n📊 FlashCore Backup System Overview'));
    console.log(chalk.gray('==========================================\n'));

    console.log(chalk.green('✅ Features:'));
    console.log('  • Complete project backup (source, config, dependencies)');
    console.log('  • Automated compression and archiving');
    console.log('  • Intelligent file filtering and exclusion');
    console.log('  • Backup verification and integrity checks');
    console.log('  • Automated cleanup and retention policies');
    console.log('  • Scheduled backups with cron expressions');
    console.log('  • Interactive restore with conflict resolution');
    console.log('  • Detailed logging and status reporting');
    console.log('  • Cross-platform compatibility');
    console.log('');

    console.log(chalk.green('📦 What Gets Backed Up:'));
    console.log('  • Source code (src/, components/, pages/, etc.)');
    console.log('  • Configuration files (package.json, tsconfig.json, etc.)');
    console.log('  • Environment files (.env*)');
    console.log('  • Dependencies information (exact versions)');
    console.log('  • Supabase configuration and database scripts');
    console.log('  • Documentation and README files');
    console.log('  • Build configuration (Vite, Tailwind, ESLint)');
    console.log('');

    console.log(chalk.green('🛡️ Security & Reliability:'));
    console.log('  • SHA256 checksums for integrity verification');
    console.log('  • Manifest files with detailed metadata');
    console.log('  • Automatic backup rotation and cleanup');
    console.log('  • Error handling and recovery mechanisms');
    console.log('  • Progress indicators and detailed logging');
    console.log('  • Backup validation and testing tools');
    console.log('');
  }

  showBackupHelp() {
    console.log(chalk.blue.bold('\n📦 Backup Commands & Options'));
    console.log(chalk.gray('==============================\n'));

    console.log(chalk.yellow('🔧 Main Backup Script:'));
    console.log('  node scripts/backup-system/backup.js [options]');
    console.log('');

    console.log(chalk.yellow('📋 Available Options:'));
    console.log('  -q, --quick              Quick backup (skip some files)');
    console.log('  -f, --full               Full backup (include all files)');
    console.log('  -c, --config <path>      Custom config file path');
    console.log('  --compression <level>    Compression level (1-9, default: 9)');
    console.log('  --max-backups <number>   Maximum backups to keep (default: 10)');
    console.log('');

    console.log(chalk.yellow('📁 Backup Patterns:'));
    console.log('  Source Code:     src/, components/, pages/, hooks/, utils/');
    console.log('  Configuration:   package.json, tsconfig.json, vite.config.ts');
    console.log('  Environment:     .env, .env.local, .env.production');
    console.log('  Documentation:   README.md, docs/, *.md');
    console.log('  Database:        supabase/, database scripts');
    console.log('');

    console.log(chalk.yellow('🚫 Excluded Files:'));
    console.log('  node_modules/, dist/, .git/, *.log, backups/');
    console.log('  Temporary files, build artifacts, logs');
    console.log('');

    console.log(chalk.yellow('📊 Backup Output:'));
    console.log('  • Compressed .tar.gz archive');
    console.log('  • Backup manifest with metadata');
    console.log('  • Restore script for easy recovery');
    console.log('  • Database backup instructions');
    console.log('  • Dependencies information');
    console.log('');
  }

  showRestoreHelp() {
    console.log(chalk.blue.bold('\n🔄 Restore Commands & Options'));
    console.log(chalk.gray('================================\n'));

    console.log(chalk.yellow('🔧 Main Restore Script:'));
    console.log('  node scripts/backup-system/restore.js [backup-file] [options]');
    console.log('');

    console.log(chalk.yellow('📋 Available Options:'));
    console.log('  -i, --interactive        Interactive mode with prompts');
    console.log('  -o, --overwrite          Overwrite existing files');
    console.log('  --skip-dependencies      Skip npm install');
    console.log('  --skip-database          Skip database setup');
    console.log('  --restore-dir <path>     Temporary restore directory');
    console.log('');

    console.log(chalk.yellow('🔄 Restore Process:'));
    console.log('  1. Extract backup archive');
    console.log('  2. Validate backup integrity');
    console.log('  3. Check for existing files');
    console.log('  4. Restore project files');
    console.log('  5. Install dependencies');
    console.log('  6. Setup environment');
    console.log('  7. Verify restore completion');
    console.log('  8. Cleanup temporary files');
    console.log('');

    console.log(chalk.yellow('⚠️ Safety Features:'));
    console.log('  • Backup verification before restore');
    console.log('  • Conflict detection and resolution');
    console.log('  • Interactive confirmation prompts');
    console.log('  • Rollback capability on failure');
    console.log('  • Detailed restore logging');
    console.log('');

    console.log(chalk.yellow('📋 Post-Restore Steps:'));
    console.log('  1. Update .env file with correct values');
    console.log('  2. Configure database connection');
    console.log('  3. Run: npm run dev');
    console.log('  4. Test all functionality');
    console.log('');
  }

  showSchedulerHelp() {
    console.log(chalk.blue.bold('\n⏰ Scheduler Commands & Options'));
    console.log(chalk.gray('==================================\n'));

    console.log(chalk.yellow('🔧 Scheduler Script:'));
    console.log('  node scripts/backup-system/scheduler.js <command> [options]');
    console.log('');

    console.log(chalk.yellow('📋 Available Commands:'));
    console.log('  start         Start the backup scheduler');
    console.log('  status        Show scheduler status');
    console.log('  backup-now    Trigger immediate backup');
    console.log('  install       Install as system service');
    console.log('');

    console.log(chalk.yellow('⏰ Start Options:'));
    console.log('  -s, --schedule <cron>     Cron expression (default: "0 2 * * *")');
    console.log('  -d, --backup-dir <path>   Backup directory');
    console.log('  -m, --max-backups <num>   Maximum backups to keep');
    console.log('  -c, --compression <level> Compression level (1-9)');
    console.log('  --log-file <path>         Log file path');
    console.log('');

    console.log(chalk.yellow('📅 Cron Examples:'));
    console.log('  "0 2 * * *"     Daily at 2 AM');
    console.log('  "0 */6 * * *"   Every 6 hours');
    console.log('  "0 2 * * 0"     Weekly on Sunday at 2 AM');
    console.log('  "0 2 1 * *"     Monthly on 1st at 2 AM');
    console.log('');

    console.log(chalk.yellow('📊 Scheduler Features:'));
    console.log('  • Automatic backup scheduling');
    console.log('  • Configurable retention policies');
    console.log('  • Error handling and retry logic');
    console.log('  • Detailed logging and monitoring');
    console.log('  • Graceful shutdown handling');
    console.log('  • Status reporting and alerts');
    console.log('');
  }

  showMaintenanceHelp() {
    console.log(chalk.blue.bold('\n🔧 Maintenance Commands & Options'));
    console.log(chalk.gray('=====================================\n'));

    console.log(chalk.yellow('📊 Status Command:'));
    console.log('  node scripts/backup-system/status.js [options]');
    console.log('  Options:');
    console.log('    -d, --detailed          Show detailed information');
    console.log('    -f, --format <format>   Output format (text|json)');
    console.log('    --backup-dir <path>     Backup directory');
    console.log('');

    console.log(chalk.yellow('🧹 Cleanup Command:'));
    console.log('  node scripts/backup-system/cleanup.js [options]');
    console.log('  Options:');
    console.log('    -d, --backup-dir <path>   Backup directory');
    console.log('    -m, --max-backups <num>   Maximum backups to keep');
    console.log('    -a, --max-age <days>      Maximum age in days');
    console.log('    -s, --min-size <bytes>    Minimum size to keep');
    console.log('    --dry-run                 Show what would be deleted');
    console.log('    -f, --force               Force cleanup');
    console.log('    -i, --interactive         Interactive mode');
    console.log('');

    console.log(chalk.yellow('🔍 Verification Command:'));
    console.log('  node scripts/backup-system/verify.js [backup-file] [options]');
    console.log('  Options:');
    console.log('    -d, --backup-dir <path>   Backup directory');
    console.log('    -e, --extract             Extract and verify files');
    console.log('    --no-checksum             Skip checksum calculation');
    console.log('    --detailed                Show detailed information');
    console.log('    --fix                     Attempt to fix issues');
    console.log('');

    console.log(chalk.yellow('🛠️ Maintenance Features:'));
    console.log('  • Backup integrity verification');
    console.log('  • Automated cleanup and retention');
    console.log('  • Disk space monitoring');
    console.log('  • Backup health checks');
    console.log('  • Performance optimization');
    console.log('  • Error detection and reporting');
    console.log('');
  }

  showExamples() {
    console.log(chalk.blue.bold('\n💡 Usage Examples & Scenarios'));
    console.log(chalk.gray('=================================\n'));

    console.log(chalk.yellow('🚀 Quick Start Examples:'));
    console.log(chalk.cyan('  # Create a backup'));
    console.log('  npm run backup');
    console.log('  node scripts/backup-system/backup.js');
    console.log('');

    console.log(chalk.cyan('  # Restore from latest backup'));
    console.log('  npm run restore:latest');
    console.log('  node scripts/backup-system/restore.js --interactive');
    console.log('');

    console.log(chalk.cyan('  # Check backup status'));
    console.log('  npm run backup:status');
    console.log('  node scripts/backup-system/status.js --detailed');
    console.log('');

    console.log(chalk.yellow('⚙️ Advanced Examples:'));
    console.log(chalk.cyan('  # Custom backup with specific options'));
    console.log('  node scripts/backup-system/backup.js --compression 6 --max-backups 5');
    console.log('');

    console.log(chalk.cyan('  # Restore specific backup file'));
    console.log('  node scripts/backup-system/restore.js backups/flashcore_20241201.tar.gz');
    console.log('');

    console.log(chalk.cyan('  # Start automated scheduler'));
    console.log('  node scripts/backup-system/scheduler.js start --schedule "0 2 * * *"');
    console.log('');

    console.log(chalk.cyan('  # Clean up old backups'));
    console.log('  node scripts/backup-system/cleanup.js --max-age 30 --dry-run');
    console.log('  node scripts/backup-system/cleanup.js --max-age 30 --force');
    console.log('');

    console.log(chalk.cyan('  # Verify backup integrity'));
    console.log('  node scripts/backup-system/verify.js --extract --detailed');
    console.log('');

    console.log(chalk.yellow('🔧 Production Scenarios:'));
    console.log(chalk.cyan('  # Daily automated backup'));
    console.log('  # Add to crontab: 0 2 * * * cd /path/to/project && npm run backup');
    console.log('');

    console.log(chalk.cyan('  # Disaster recovery'));
    console.log('  1. node scripts/backup-system/status.js');
    console.log('  2. node scripts/backup-system/verify.js latest-backup.tar.gz');
    console.log('  3. node scripts/backup-system/restore.js latest-backup.tar.gz --overwrite');
    console.log('');

    console.log(chalk.cyan('  # Migration to new server'));
    console.log('  1. Create backup on old server');
    console.log('  2. Transfer backup file to new server');
    console.log('  3. Restore on new server');
    console.log('  4. Update environment configuration');
    console.log('');
  }

  showConfigHelp() {
    console.log(chalk.blue.bold('\n⚙️ Configuration Options'));
    console.log(chalk.gray('==========================\n'));

    console.log(chalk.yellow('📁 Configuration Files:'));
    console.log(chalk.cyan('  backup-config.json'));
    console.log('  {');
    console.log('    "projectName": "flashcore-pulse-dashboard",');
    console.log('    "backupDir": "backups",');
    console.log('    "compressionLevel": 9,');
    console.log('    "maxBackups": 10,');
    console.log('    "includeNodeModules": false,');
    console.log('    "includeDist": false');
    console.log('  }');
    console.log('');

    console.log(chalk.cyan('  backup-schedule.json'));
    console.log('  {');
    console.log('    "schedule": "0 2 * * *",');
    console.log('    "backupDir": "backups",');
    console.log('    "maxBackups": 10,');
    console.log('    "compressionLevel": 9,');
    console.log('    "enabled": true,');
    console.log('    "logFile": "backup-scheduler.log"');
    console.log('  }');
    console.log('');

    console.log(chalk.yellow('🔧 Environment Variables:'));
    console.log('  VITE_SUPABASE_URL          Supabase project URL');
    console.log('  VITE_SUPABASE_ANON_KEY     Supabase anonymous key');
    console.log('  NODE_ENV                   Environment (development/production)');
    console.log('');

    console.log(chalk.yellow('📋 Package.json Scripts:'));
    console.log('  "backup": "node scripts/backup-system/backup.js"');
    console.log('  "restore": "node scripts/backup-system/restore.js"');
    console.log('  "backup:quick": "node scripts/backup-system/backup.js --quick"');
    console.log('  "backup:full": "node scripts/backup-system/backup.js --full"');
    console.log('  "restore:latest": "node scripts/backup-system/restore.js --latest"');
    console.log('  "backup:status": "node scripts/backup-system/status.js"');
    console.log('  "backup:cleanup": "node scripts/backup-system/cleanup.js"');
    console.log('  "backup:verify": "node scripts/backup-system/verify.js"');
    console.log('');

    console.log(chalk.yellow('🛡️ Security Considerations:'));
    console.log('  • Store .env files securely');
    console.log('  • Use environment variables for sensitive data');
    console.log('  • Regularly rotate backup locations');
    console.log('  • Monitor backup access logs');
    console.log('  • Test restore procedures regularly');
    console.log('');
  }
}

// Initialize help system
new BackupHelp(); 