#!/usr/bin/env node

import fs from 'fs-extra';
import path from 'path';
import chalk from 'chalk';
import ora from 'ora';
import { program } from 'commander';
import { glob } from 'glob';
import { z } from 'zod';
import dotenv from 'dotenv';

// Load environment variables
dotenv.config();

// Cleanup configuration schema
const CleanupConfigSchema = z.object({
  backupDir: z.string().default('backups'),
  maxBackups: z.number().default(10),
  maxAge: z.number().default(30), // days
  minSize: z.number().default(0), // bytes
  dryRun: z.boolean().default(false),
  force: z.boolean().default(false),
  interactive: z.boolean().default(false)
});

class BackupCleanup {
  constructor(options = {}) {
    this.config = CleanupConfigSchema.parse(options);
  }

  log(message, type = 'info') {
    const timestamp = new Date().toISOString();
    const colors = {
      info: chalk.blue,
      success: chalk.green,
      error: chalk.red,
      warning: chalk.yellow,
      debug: chalk.gray
    };
    
    const prefix = {
      info: 'ℹ️',
      success: '✅',
      error: '❌',
      warning: '⚠️',
      debug: '🔍'
    }[type];
    
    console.log(`${colors[type](`${prefix} [${timestamp}] ${message}`)}`);
  }

  async getBackupFiles() {
    try {
      const backupFiles = await glob(path.join(this.config.backupDir, '*.tar.gz'));
      
      const backups = [];
      for (const file of backupFiles) {
        const stat = await fs.stat(file);
        const ageInDays = (Date.now() - stat.mtime.getTime()) / (1000 * 60 * 60 * 24);
        
        backups.push({
          name: path.basename(file),
          path: file,
          size: stat.size,
          modified: stat.mtime,
          ageInDays: Math.floor(ageInDays)
        });
      }
      
      // Sort by modification time (oldest first)
      backups.sort((a, b) => a.modified.getTime() - b.modified.getTime());
      
      return backups;
    } catch (error) {
      throw new Error(`Failed to get backup files: ${error.message}`);
    }
  }

  async analyzeBackups() {
    const backups = await this.getBackupFiles();
    const now = new Date();
    const cutoffDate = new Date(now.getTime() - (this.config.maxAge * 24 * 60 * 60 * 1000));
    
    const candidates = {
      byCount: [],
      byAge: [],
      bySize: [],
      total: []
    };
    
    // Candidates by count (keep only the newest maxBackups)
    if (backups.length > this.config.maxBackups) {
      candidates.byCount = backups.slice(0, backups.length - this.config.maxBackups);
    }
    
    // Candidates by age (older than maxAge days)
    candidates.byAge = backups.filter(backup => backup.modified < cutoffDate);
    
    // Candidates by size (smaller than minSize)
    if (this.config.minSize > 0) {
      candidates.bySize = backups.filter(backup => backup.size < this.config.minSize);
    }
    
    // Combine all candidates
    const allCandidates = new Set();
    candidates.byCount.forEach(b => allCandidates.add(b.path));
    candidates.byAge.forEach(b => allCandidates.add(b.path));
    candidates.bySize.forEach(b => allCandidates.add(b.path));
    
    candidates.total = Array.from(allCandidates).map(path => 
      backups.find(b => b.path === path)
    );
    
    // Calculate space to be freed
    const spaceToFree = candidates.total.reduce((sum, backup) => sum + backup.size, 0);
    
    return {
      backups,
      candidates,
      spaceToFree,
      totalBackups: backups.length,
      totalSize: backups.reduce((sum, backup) => sum + backup.size, 0)
    };
  }

  formatBytes(bytes) {
    if (bytes === 0) return '0 Bytes';
    const k = 1024;
    const sizes = ['Bytes', 'KB', 'MB', 'GB', 'TB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
  }

  formatDate(date) {
    return date.toLocaleDateString();
  }

  async showAnalysis(analysis) {
    console.log('\n🧹 FlashCore Backup Cleanup Analysis');
    console.log('====================================');
    
    console.log(`\n📊 Current Status:`);
    console.log(`   Total backups: ${analysis.totalBackups}`);
    console.log(`   Total size: ${this.formatBytes(analysis.totalSize)}`);
    console.log(`   Space to free: ${this.formatBytes(analysis.spaceToFree)}`);
    
    if (analysis.candidates.byCount.length > 0) {
      console.log(`\n📈 By Count (keeping ${this.config.maxBackups} newest):`);
      console.log(`   ${analysis.candidates.byCount.length} backups to remove`);
      for (const backup of analysis.candidates.byCount.slice(0, 5)) {
        console.log(`   - ${backup.name} (${this.formatBytes(backup.size)}, ${backup.ageInDays} days old)`);
      }
      if (analysis.candidates.byCount.length > 5) {
        console.log(`   ... and ${analysis.candidates.byCount.length - 5} more`);
      }
    }
    
    if (analysis.candidates.byAge.length > 0) {
      console.log(`\n⏰ By Age (older than ${this.config.maxAge} days):`);
      console.log(`   ${analysis.candidates.byAge.length} backups to remove`);
      for (const backup of analysis.candidates.byAge.slice(0, 5)) {
        console.log(`   - ${backup.name} (${this.formatBytes(backup.size)}, ${backup.ageInDays} days old)`);
      }
      if (analysis.candidates.byAge.length > 5) {
        console.log(`   ... and ${analysis.candidates.byAge.length - 5} more`);
      }
    }
    
    if (analysis.candidates.bySize.length > 0) {
      console.log(`\n📏 By Size (smaller than ${this.formatBytes(this.config.minSize)}):`);
      console.log(`   ${analysis.candidates.bySize.length} backups to remove`);
      for (const backup of analysis.candidates.bySize.slice(0, 5)) {
        console.log(`   - ${backup.name} (${this.formatBytes(backup.size)}, ${backup.ageInDays} days old)`);
      }
      if (analysis.candidates.bySize.length > 5) {
        console.log(`   ... and ${analysis.candidates.bySize.length - 5} more`);
      }
    }
    
    if (analysis.candidates.total.length === 0) {
      console.log(`\n✅ No backups to clean up!`);
    } else {
      console.log(`\n🗑️  Total backups to remove: ${analysis.candidates.total.length}`);
      console.log(`💾 Space to be freed: ${this.formatBytes(analysis.spaceToFree)}`);
    }
    
    console.log('');
  }

  async performCleanup(analysis) {
    if (analysis.candidates.total.length === 0) {
      this.log('No backups to clean up', 'info');
      return { deleted: 0, spaceFreed: 0 };
    }
    
    const spinner = ora('🧹 Cleaning up old backups...').start();
    
    let deletedCount = 0;
    let spaceFreed = 0;
    
    try {
      for (const backup of analysis.candidates.total) {
        if (this.config.dryRun) {
          this.log(`[DRY RUN] Would delete: ${backup.name}`, 'debug');
          deletedCount++;
          spaceFreed += backup.size;
        } else {
          await fs.remove(backup.path);
          this.log(`Deleted: ${backup.name}`, 'success');
          deletedCount++;
          spaceFreed += backup.size;
        }
      }
      
      if (this.config.dryRun) {
        spinner.succeed(`[DRY RUN] Would delete ${deletedCount} backups (${this.formatBytes(spaceFreed)})`);
      } else {
        spinner.succeed(`Cleaned up ${deletedCount} backups, freed ${this.formatBytes(spaceFreed)}`);
      }
      
      return { deleted: deletedCount, spaceFreed };
    } catch (error) {
      spinner.fail(`Cleanup failed: ${error.message}`);
      throw error;
    }
  }

  async run() {
    try {
      this.log('🔍 Analyzing backup files...');
      
      const analysis = await this.analyzeBackups();
      await this.showAnalysis(analysis);
      
      if (analysis.candidates.total.length === 0) {
        this.log('No cleanup needed', 'success');
        return;
      }
      
      // Check if we should proceed
      if (!this.config.force && !this.config.dryRun) {
        if (this.config.interactive) {
          const readline = (await import('readline')).createInterface({
            input: process.stdin,
            output: process.stdout
          });
          
          const question = (prompt) => new Promise((resolve) => {
            readline.question(prompt, resolve);
          });
          
          const answer = await question(`Proceed with cleanup? (y/N): `);
          readline.close();
          
          if (answer.toLowerCase() !== 'y' && answer.toLowerCase() !== 'yes') {
            this.log('Cleanup cancelled', 'warning');
            return;
          }
        } else {
          this.log('Use --force flag to proceed with cleanup, or --dry-run to see what would be deleted', 'warning');
          return;
        }
      }
      
      const result = await this.performCleanup(analysis);
      
      // Show final status
      const finalAnalysis = await this.analyzeBackups();
      console.log(`\n📊 Final Status:`);
      console.log(`   Remaining backups: ${finalAnalysis.totalBackups}`);
      console.log(`   Remaining size: ${this.formatBytes(finalAnalysis.totalSize)}`);
      
    } catch (error) {
      this.log(`Cleanup failed: ${error.message}`, 'error');
      process.exit(1);
    }
  }
}

// CLI setup
program
  .name('flashcore-cleanup')
  .description('Clean up old FlashCore backup files')
  .version('1.0.0');

program
  .option('-d, --backup-dir <path>', 'Backup directory', 'backups')
  .option('-m, --max-backups <number>', 'Maximum number of backups to keep', '10')
  .option('-a, --max-age <days>', 'Maximum age of backups in days', '30')
  .option('-s, --min-size <bytes>', 'Minimum size of backups to keep', '0')
  .option('--dry-run', 'Show what would be deleted without actually deleting')
  .option('-f, --force', 'Force cleanup without confirmation')
  .option('-i, --interactive', 'Interactive mode with confirmation prompts');

program.parse();

const options = program.opts();

// Run cleanup
const cleanup = new BackupCleanup({
  backupDir: options.backupDir,
  maxBackups: parseInt(options.maxBackups),
  maxAge: parseInt(options.maxAge),
  minSize: parseInt(options.minSize),
  dryRun: options.dryRun,
  force: options.force,
  interactive: options.interactive
});

cleanup.run(); 