#!/usr/bin/env node

import fs from 'fs-extra';
import path from 'path';
import archiver from 'archiver';
import chalk from 'chalk';
import ora from 'ora';
import { program } from 'commander';
import { glob } from 'glob';
import { z } from 'zod';
import dotenv from 'dotenv';

// Load environment variables
dotenv.config();

// Configuration schema
const ConfigSchema = z.object({
  projectName: z.string().default('flashcore-pulse-dashboard'),
  backupDir: z.string().default('backups'),
  compressionLevel: z.number().min(1).max(9).default(9),
  maxBackups: z.number().default(10),
  includeNodeModules: z.boolean().default(false),
  includeDist: z.boolean().default(false),
  includeLogs: z.boolean().default(false)
});

class FlashCoreBackup {
  constructor(options = {}) {
    this.config = ConfigSchema.parse({
      ...this.loadConfig(),
      ...options
    });
    
    this.date = new Date().toISOString().replace(/[-:T]/g, '').slice(0, 15);
    this.backupName = `${this.config.projectName}_${this.date}`;
    this.backupPath = path.join(this.config.backupDir, this.backupName);
    this.backupArchive = `${this.backupPath}.tar.gz`;
    
    // Define backup patterns
    this.patterns = {
      source: [
        'src/**/*',
        'public/**/*',
        'components/**/*',
        'hooks/**/*',
        'pages/**/*',
        'utils/**/*',
        'types/**/*',
        'lib/**/*',
        'data/**/*',
        'integrations/**/*'
      ],
      config: [
        'package.json',
        'package-lock.json',
        'bun.lockb',
        'tsconfig*.json',
        'tailwind.config.ts',
        'vite.config.ts',
        'eslint.config.js',
        'postcss.config.js',
        'components.json',
        'index.html',
        'README.md',
        '.gitignore',
        '.env*'
      ],
      docs: [
        'docs/**/*',
        '*.md',
        'LICENSE'
      ],
      supabase: [
        'supabase/**/*'
      ]
    };
    
    this.excludePatterns = [
      'node_modules/**/*',
      'dist/**/*',
      '.git/**/*',
      '*.log',
      '*.tmp',
      '.DS_Store',
      'Thumbs.db',
      'backups/**/*',
      'scripts/backup-system/**/*'
    ];
  }

  loadConfig() {
    try {
      const configPath = path.join(process.cwd(), 'backup-config.json');
      if (fs.existsSync(configPath)) {
        return JSON.parse(fs.readFileSync(configPath, 'utf8'));
      }
    } catch (error) {
      this.log(`Warning: Could not load config file: ${error.message}`, 'warning');
    }
    return {};
  }

  log(message, type = 'info') {
    const timestamp = new Date().toISOString();
    const colors = {
      info: chalk.blue,
      success: chalk.green,
      error: chalk.red,
      warning: chalk.yellow,
      debug: chalk.gray
    };
    
    const prefix = {
      info: 'ℹ️',
      success: '✅',
      error: '❌',
      warning: '⚠️',
      debug: '🔍'
    }[type];
    
    console.log(`${colors[type](`${prefix} [${timestamp}] ${message}`)}`);
  }

  async ensureDirectoryExists(dirPath) {
    try {
      await fs.ensureDir(dirPath);
      return true;
    } catch (error) {
      this.log(`Failed to create directory ${dirPath}: ${error.message}`, 'error');
      return false;
    }
  }

  async copyFile(src, dest) {
    try {
      if (await fs.pathExists(src)) {
        await fs.ensureDir(path.dirname(dest));
        await fs.copy(src, dest);
        return true;
      }
      return false;
    } catch (error) {
      this.log(`Failed to copy ${src}: ${error.message}`, 'error');
      return false;
    }
  }

  async copyDirectory(src, dest) {
    try {
      if (!await fs.pathExists(src)) {
        return false;
      }

      const stat = await fs.stat(src);
      if (stat.isDirectory()) {
        await fs.ensureDir(dest);
        await fs.copy(src, dest);
        return true;
      } else {
        return await this.copyFile(src, dest);
      }
    } catch (error) {
      this.log(`Failed to copy directory ${src}: ${error.message}`, 'error');
      return false;
    }
  }

  async backupSourceCode() {
    const spinner = ora('📁 Backing up source code...').start();
    
    try {
      let copiedCount = 0;
      const totalFiles = this.patterns.source.length;
      
      for (const pattern of this.patterns.source) {
        const files = await glob(pattern, { 
          ignore: this.excludePatterns,
          nodir: true 
        });
        
        for (const file of files) {
          const relativePath = path.relative(process.cwd(), file);
          const destPath = path.join(this.backupPath, relativePath);
          
          if (await this.copyFile(file, destPath)) {
            copiedCount++;
          }
        }
      }
      
      spinner.succeed(`Source code backup completed: ${copiedCount} files copied`);
      return copiedCount;
    } catch (error) {
      spinner.fail(`Source code backup failed: ${error.message}`);
      throw error;
    }
  }

  async backupConfigurationFiles() {
    const spinner = ora('⚙️ Backing up configuration files...').start();
    
    try {
      let copiedCount = 0;
      
      for (const pattern of this.patterns.config) {
        const files = await glob(pattern, { nodir: true });
        
        for (const file of files) {
          const relativePath = path.relative(process.cwd(), file);
          const destPath = path.join(this.backupPath, relativePath);
          
          if (await this.copyFile(file, destPath)) {
            copiedCount++;
          }
        }
      }
      
      spinner.succeed(`Configuration backup completed: ${copiedCount} files copied`);
      return copiedCount;
    } catch (error) {
      spinner.fail(`Configuration backup failed: ${error.message}`);
      throw error;
    }
  }

  async backupDocumentation() {
    const spinner = ora('📚 Backing up documentation...').start();
    
    try {
      let copiedCount = 0;
      
      for (const pattern of this.patterns.docs) {
        const files = await glob(pattern, { nodir: true });
        
        for (const file of files) {
          const relativePath = path.relative(process.cwd(), file);
          const destPath = path.join(this.backupPath, relativePath);
          
          if (await this.copyFile(file, destPath)) {
            copiedCount++;
          }
        }
      }
      
      spinner.succeed(`Documentation backup completed: ${copiedCount} files copied`);
      return copiedCount;
    } catch (error) {
      spinner.fail(`Documentation backup failed: ${error.message}`);
      throw error;
    }
  }

  async backupSupabase() {
    const spinner = ora('🗄️ Backing up Supabase configuration...').start();
    
    try {
      let copiedCount = 0;
      
      for (const pattern of this.patterns.supabase) {
        const files = await glob(pattern, { nodir: true });
        
        for (const file of files) {
          const relativePath = path.relative(process.cwd(), file);
          const destPath = path.join(this.backupPath, relativePath);
          
          if (await this.copyFile(file, destPath)) {
            copiedCount++;
          }
        }
      }
      
      // Create database backup script
      const dbBackupScript = `#!/bin/bash
echo "🗄️ Database backup script for FlashCore"
echo "====================================="
echo ""
echo "📋 To backup your Supabase database:"
echo "1. Go to your Supabase dashboard"
echo "2. Navigate to Settings > Database"
echo "3. Click 'Download backup'"
echo ""
echo "📋 To restore your database:"
echo "1. Go to your Supabase dashboard"
echo "2. Navigate to Settings > Database"
echo "3. Click 'Restore from backup'"
echo "4. Upload your backup file"
echo ""
echo "🔧 Alternative using Supabase CLI:"
echo "npx supabase db dump --db-url 'your-supabase-url' > database_backup.sql"
echo "npx supabase db reset --db-url 'your-supabase-url' < database_backup.sql"
`;
      
      await fs.writeFile(path.join(this.backupPath, 'database-backup.sh'), dbBackupScript);
      await fs.chmod(path.join(this.backupPath, 'database-backup.sh'), 0o755);
      
      spinner.succeed(`Supabase backup completed: ${copiedCount} files + backup script`);
      return copiedCount;
    } catch (error) {
      spinner.fail(`Supabase backup failed: ${error.message}`);
      throw error;
    }
  }

  async backupDependencies() {
    const spinner = ora('📦 Backing up dependencies information...').start();
    
    try {
      // Get exact dependency versions
      const packageJson = JSON.parse(await fs.readFile('package.json', 'utf8'));
      const dependencies = {
        dependencies: packageJson.dependencies || {},
        devDependencies: packageJson.devDependencies || {},
        peerDependencies: packageJson.peerDependencies || {},
        scripts: packageJson.scripts || {}
      };
      
      await fs.writeFile(
        path.join(this.backupPath, 'dependencies.json'),
        JSON.stringify(dependencies, null, 2)
      );
      
      // Create npm list output
      try {
        const { execSync } = await import('child_process');
        const npmList = execSync('npm list --depth=0 --json', { encoding: 'utf8' });
        await fs.writeFile(path.join(this.backupPath, 'npm-list.json'), npmList);
      } catch (error) {
        this.log(`Warning: Could not generate npm list: ${error.message}`, 'warning');
      }
      
      spinner.succeed('Dependencies information backed up');
      return 1;
    } catch (error) {
      spinner.fail(`Dependencies backup failed: ${error.message}`);
      throw error;
    }
  }

  async createRestoreScript() {
    const spinner = ora('🔧 Creating restore script...').start();
    
    try {
      const restoreScript = `#!/bin/bash

echo "🔄 FlashCore Dashboard Restore Script"
echo "====================================="
echo ""

# Check if we're in the right directory
if [ ! -f "package.json" ]; then
    echo "❌ Error: package.json not found. Please run this script from the project root."
    exit 1
fi

echo "📦 Installing dependencies..."
npm install

if [ $? -eq 0 ]; then
    echo "✅ Dependencies installed successfully"
else
    echo "❌ Failed to install dependencies"
    exit 1
fi

echo ""
echo "🔐 Environment setup..."
if [ -f ".env" ]; then
    echo "✅ .env file found"
    echo "⚠️  Please review and update any sensitive information in .env"
else
    echo "⚠️  No .env file found"
    echo "📝 Please create a .env file with your configuration:"
    echo "   - Copy from .env.backup if available"
    echo "   - Or create a new one based on .env.template"
fi

echo ""
echo "🗄️ Database setup..."
if [ -f "database-backup.sh" ]; then
    echo "📋 Database backup script found"
    echo "   Run: bash database-backup.sh for instructions"
else
    echo "⚠️  No database backup found"
    echo "   Please configure your database connection manually"
fi

echo ""
echo "🚀 Starting development server..."
echo "   Run: npm run dev"
echo ""
echo "✅ Restore completed successfully!"
echo ""
echo "📋 Next steps:"
echo "   1. Update your .env file with correct values"
echo "   2. Configure your database connection"
echo "   3. Run: npm run dev"
echo "   4. Test all functionality"
`;

      await fs.writeFile(path.join(this.backupPath, 'restore.sh'), restoreScript);
      await fs.chmod(path.join(this.backupPath, 'restore.sh'), 0o755);
      
      spinner.succeed('Restore script created');
      return true;
    } catch (error) {
      spinner.fail(`Failed to create restore script: ${error.message}`);
      throw error;
    }
  }

  async createManifest() {
    const spinner = ora('📋 Creating backup manifest...').start();
    
    try {
      const manifest = {
        project: this.config.projectName,
        backupDate: new Date().toISOString(),
        version: await this.getPackageVersion(),
        files: await this.getFileList(this.backupPath),
        system: {
          nodeVersion: process.version,
          platform: process.platform,
          arch: process.arch,
          cwd: process.cwd()
        },
        backupInfo: {
          totalSize: await this.getDirectorySize(this.backupPath),
          compressionLevel: this.config.compressionLevel,
          backupType: 'full'
        },
        config: this.config
      };
      
      await fs.writeFile(
        path.join(this.backupPath, 'backup-manifest.json'),
        JSON.stringify(manifest, null, 2)
      );
      
      spinner.succeed('Backup manifest created');
      return manifest;
    } catch (error) {
      spinner.fail(`Failed to create manifest: ${error.message}`);
      throw error;
    }
  }

  async getPackageVersion() {
    try {
      const packageJson = JSON.parse(await fs.readFile('package.json', 'utf8'));
      return packageJson.version || '1.0.0';
    } catch {
      return '1.0.0';
    }
  }

  async getFileList(dir) {
    const files = [];
    
    async function scanDirectory(currentDir, relativePath = '') {
      try {
        const entries = await fs.readdir(currentDir);
        
        for (const entry of entries) {
          const fullPath = path.join(currentDir, entry);
          const relativeFilePath = path.join(relativePath, entry);
          
          try {
            const stat = await fs.stat(fullPath);
            if (stat.isDirectory()) {
              await scanDirectory(fullPath, relativeFilePath);
            } else {
              files.push({
                path: relativeFilePath,
                size: stat.size,
                modified: stat.mtime.toISOString()
              });
            }
          } catch (error) {
            // Skip files we can't access
          }
        }
      } catch (error) {
        // Skip directories we can't access
      }
    }
    
    await scanDirectory(dir);
    return files;
  }

  async getDirectorySize(dir) {
    let totalSize = 0;
    
    async function calculateSize(currentDir) {
      try {
        const entries = await fs.readdir(currentDir);
        
        for (const entry of entries) {
          const fullPath = path.join(currentDir, entry);
          
          try {
            const stat = await fs.stat(fullPath);
            if (stat.isDirectory()) {
              await calculateSize(fullPath);
            } else {
              totalSize += stat.size;
            }
          } catch (error) {
            // Skip files we can't access
          }
        }
      } catch (error) {
        // Skip directories we can't access
      }
    }
    
    await calculateSize(dir);
    return totalSize;
  }

  async compressBackup() {
    const spinner = ora('🗜️ Compressing backup...').start();
    
    return new Promise((resolve, reject) => {
      const output = fs.createWriteStream(this.backupArchive);
      const archive = archiver('tar', { 
        gzip: true,
        gzipOptions: { level: this.config.compressionLevel }
      });
      
      output.on('close', () => {
        const size = (archive.pointer() / 1024 / 1024).toFixed(2);
        spinner.succeed(`Backup compressed: ${this.backupArchive} (${size} MB)`);
        
        // Clean up uncompressed directory
        fs.remove(this.backupPath).then(() => {
          this.log('🧹 Cleaned up uncompressed backup directory');
          resolve();
        }).catch(error => {
          this.log(`Warning: Could not clean up directory: ${error.message}`, 'warning');
          resolve();
        });
      });
      
      archive.on('error', (err) => {
        spinner.fail(`Failed to compress backup: ${err.message}`);
        reject(err);
      });
      
      archive.pipe(output);
      archive.directory(this.backupPath, false);
      archive.finalize();
    });
  }

  async cleanupOldBackups() {
    const spinner = ora('🧹 Cleaning up old backups...').start();
    
    try {
      const backupFiles = await glob(path.join(this.config.backupDir, '*.tar.gz'));
      
      if (backupFiles.length > this.config.maxBackups) {
        // Sort by modification time (oldest first)
        backupFiles.sort((a, b) => {
          const statA = fs.statSync(a);
          const statB = fs.statSync(b);
          return statA.mtime.getTime() - statB.mtime.getTime();
        });
        
        const filesToDelete = backupFiles.slice(0, backupFiles.length - this.config.maxBackups);
        
        for (const file of filesToDelete) {
          await fs.remove(file);
          this.log(`Deleted old backup: ${path.basename(file)}`);
        }
        
        spinner.succeed(`Cleaned up ${filesToDelete.length} old backups`);
      } else {
        spinner.succeed('No old backups to clean up');
      }
    } catch (error) {
      spinner.fail(`Cleanup failed: ${error.message}`);
    }
  }

  async run() {
    const startTime = Date.now();
    
    try {
      this.log('🚀 Starting FlashCore backup...');
      this.log(`📁 Backup directory: ${this.backupPath}`);
      this.log(`🗜️ Compression level: ${this.config.compressionLevel}`);
      
      // Create backup directory
      await this.ensureDirectoryExists(this.backupPath);
      
      // Perform backups
      const sourceCount = await this.backupSourceCode();
      const configCount = await this.backupConfigurationFiles();
      const docsCount = await this.backupDocumentation();
      const supabaseCount = await this.backupSupabase();
      const depsCount = await this.backupDependencies();
      
      // Create restore script and manifest
      await this.createRestoreScript();
      const manifest = await this.createManifest();
      
      // Compress backup
      await this.compressBackup();
      
      // Cleanup old backups
      await this.cleanupOldBackups();
      
      const duration = ((Date.now() - startTime) / 1000).toFixed(2);
      
      this.log('🎉 Backup completed successfully!');
      this.log(`📦 Backup file: ${this.backupArchive}`);
      this.log(`⏱️  Duration: ${duration} seconds`);
      this.log(`📊 Files backed up: ${manifest.files.length}`);
      this.log('');
      this.log('📋 To restore:');
      this.log(`   npm run restore ${this.backupArchive}`);
      this.log('   or');
      this.log(`   node scripts/backup-system/restore.js ${this.backupArchive}`);
      
    } catch (error) {
      this.log(`Backup failed: ${error.message}`, 'error');
      process.exit(1);
    }
  }
}

// CLI setup
program
  .name('flashcore-backup')
  .description('Comprehensive backup system for FlashCore Pulse Dashboard')
  .version('1.0.0');

program
  .option('-q, --quick', 'Quick backup (skip some files)')
  .option('-f, --full', 'Full backup (include all files)')
  .option('-c, --config <path>', 'Custom config file path')
  .option('--compression <level>', 'Compression level (1-9)', '9')
  .option('--max-backups <number>', 'Maximum number of backups to keep', '10');

program.parse();

const options = program.opts();

// Run backup if this script is executed directly
if (import.meta.url === `file://${process.argv[1]}`) {
  const backup = new FlashCoreBackup({
    compressionLevel: parseInt(options.compression),
    maxBackups: parseInt(options.maxBackups)
  });
  backup.run();
}

export default FlashCoreBackup; 