/**
 * environment-detector.js
 * Frontend utility for environment-aware configuration.
 */

const EnvironmentDetector = {
    /**
     * Detects the current environment.
     * @returns {'development' | 'production'}
     */
    getEnvironment() {
        const hostname = window.location.hostname;
        if (
            hostname === 'localhost' ||
            hostname === '127.0.0.1' ||
            hostname.startsWith('192.168.') ||
            hostname.endsWith('.local')
        ) {
            return 'development';
        }
        return 'production';
    },

    /**
     * Gets the base API URL based on the current environment.
     * @param {string} localPath - Path to use in development.
     * @param {string} prodPath - Path to use in production.
     * @returns {string}
     */
    getApiUrl(localPath = 'http://localhost:8080', prodPath = '') {
        const env = this.getEnvironment();
        if (env === 'development') {
            return localPath;
        }
        // In production, usually same-origin or specific domain
        return prodPath || window.location.origin;
    },

    /**
     * Checks if running in a mobile/native context (Capacitor).
     * @returns {boolean}
     */
    isNative() {
        return window.location.protocol === 'capacitor:';
    }
};

export default EnvironmentDetector;
// For non-module environments
if (typeof window !== 'undefined') {
    window.EnvironmentDetector = EnvironmentDetector;
}
