<?php
/**
 * Test SMTP Connection and Email Sending
 * Tests the configured SMTP settings with Titan Email
 */

require_once __DIR__ . '/config/database.php';
require_once __DIR__ . '/utils/helpers.php';
require_once __DIR__ . '/vendor/autoload.php';

use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;

echo "========================================\n";
echo "SMTP Connection Test\n";
echo "========================================\n\n";

// Get test email recipient
$testEmail = getenv('TEST_EMAIL') ?: 'test@example.com';
echo "Test Email Recipient: $testEmail\n";
echo "Set TEST_EMAIL environment variable to use a different email.\n\n";

try {
    // Get active SMTP settings
    $stmt = $pdo->prepare("SELECT * FROM smtp_settings WHERE is_active = TRUE LIMIT 1");
    $stmt->execute();
    $settings = $stmt->fetch();
    
    if (!$settings) {
        echo "❌ ERROR: No active SMTP settings found!\n";
        echo "   Run: php backend/configure_smtp.php\n\n";
        exit(1);
    }
    
    echo "Active SMTP Settings:\n";
    echo "  Provider: {$settings['provider']}\n";
    echo "  Host: {$settings['host']}\n";
    echo "  Port: {$settings['port']}\n";
    echo "  Username: {$settings['username']}\n";
    echo "  From Email: {$settings['from_email']}\n";
    echo "  From Name: {$settings['from_name']}\n\n";
    
    // Test SMTP connection
    echo "Testing SMTP connection...\n";
    
    $mail = new PHPMailer(true);
    
    try {
        // Server settings
        $mail->isSMTP();
        $mail->Host = $settings['host'];
        $mail->SMTPAuth = true;
        $mail->Username = $settings['username'];
        $mail->Password = $settings['password'];
        
        // Port 465 uses SSL, port 587 uses STARTTLS
        $port = $settings['port'] ?? 587;
        if ($port == 465) {
            $mail->SMTPSecure = PHPMailer::ENCRYPTION_SMTPS; // SSL for port 465
            echo "  Using SSL encryption (port 465)\n";
        } else {
            $mail->SMTPSecure = PHPMailer::ENCRYPTION_STARTTLS; // STARTTLS for port 587
            echo "  Using STARTTLS encryption (port 587)\n";
        }
        
        $mail->Port = $port;
        $mail->SMTPDebug = 2; // Enable verbose debug output
        $mail->Debugoutput = function($str, $level) {
            echo "  [DEBUG] $str";
        };
        
        // Recipients
        $mail->setFrom($settings['from_email'], $settings['from_name']);
        $mail->addAddress($testEmail);
        
        // Content
        $mail->isHTML(true);
        $mail->Subject = 'FlashCore SMTP Test - ' . date('Y-m-d H:i:s');
        $mail->Body = '<h1>SMTP Test Email</h1><p>This is a test email from FlashCore to verify SMTP configuration.</p><p>Sent at: ' . date('Y-m-d H:i:s') . '</p>';
        $mail->AltBody = 'SMTP Test Email - This is a test email from FlashCore to verify SMTP configuration. Sent at: ' . date('Y-m-d H:i:s');
        
        echo "\nAttempting to send test email...\n";
        echo "========================================\n";
        
        $mail->send();
        
        echo "========================================\n";
        echo "✅ SUCCESS: Test email sent successfully!\n\n";
        echo "Email Details:\n";
        echo "  To: $testEmail\n";
        echo "  From: {$settings['from_email']} ({$settings['from_name']})\n";
        echo "  Subject: {$mail->Subject}\n";
        echo "  Check your inbox (and spam folder) for the test email.\n";
        
    } catch (Exception $e) {
        echo "========================================\n";
        echo "❌ ERROR: Failed to send email\n";
        echo "  Error: {$mail->ErrorInfo}\n";
        echo "  Exception: {$e->getMessage()}\n\n";
        
        echo "Troubleshooting Tips:\n";
        echo "  1. Verify SMTP credentials are correct\n";
        echo "  2. Check if port 465 is open in firewall\n";
        echo "  3. Verify SSL/TLS is properly configured\n";
        echo "  4. Check if the email account allows SMTP access\n";
        echo "  5. Review the debug output above for specific errors\n";
        
        exit(1);
    }
    
} catch (Exception $e) {
    echo "❌ ERROR: " . $e->getMessage() . "\n";
    exit(1);
}

echo "\n========================================\n";
echo "SMTP Test Complete\n";
echo "========================================\n";

