<?php
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../utils/helpers.php';

try {
    // Create exchanges table
    $pdo->exec("CREATE TABLE IF NOT EXISTS exchanges (
        id VARCHAR(36) PRIMARY KEY,
        name VARCHAR(100) NOT NULL UNIQUE,
        logo_url VARCHAR(255),
        logo_path VARCHAR(255),
        gradient_start VARCHAR(50),
        gradient_end VARCHAR(50),
        description TEXT,
        volume_24h DECIMAL(20, 2) DEFAULT 0,
        rating DECIMAL(3, 2) DEFAULT 0,
        features TEXT,
        is_active BOOLEAN DEFAULT 1,
        is_hot BOOLEAN DEFAULT 0,
        has_ai BOOLEAN DEFAULT 0,
        website_url VARCHAR(255),
        api_supported BOOLEAN DEFAULT 0,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        updated_at DATETIME DEFAULT CURRENT_TIMESTAMP
    )");
    echo "Created table: exchanges\n";

    // Insert exchange data
    $exchanges = [
        [
            'id' => generateUUID(),
            'name' => 'Binance',
            'logo_path' => 'images/binance.png',
            'gradient_start' => '#f3ba2f',
            'gradient_end' => '#f0b90b',
            'description' => 'World\'s largest cryptocurrency exchange by trading volume',
            'volume_24h' => 2100000000,
            'rating' => 4.9,
            'features' => json_encode(['Low Fees', 'High Volume', 'AI Trading', 'Futures', 'Staking']),
            'is_hot' => 1,
            'has_ai' => 1,
            'website_url' => 'https://binance.com',
            'api_supported' => 1
        ],
        [
            'id' => generateUUID(),
            'name' => 'Coinbase',
            'logo_path' => 'images/logo.png', // Will use default
            'gradient_start' => '#0052ff',
            'gradient_end' => '#0052ff',
            'description' => 'Trusted US-based cryptocurrency exchange platform',
            'volume_24h' => 1800000000,
            'rating' => 4.8,
            'features' => json_encode(['Regulated', 'Secure', 'User-Friendly', 'Insurance']),
            'is_hot' => 0,
            'has_ai' => 1,
            'website_url' => 'https://coinbase.com',
            'api_supported' => 1
        ],
        [
            'id' => generateUUID(),
            'name' => 'Kraken',
            'logo_path' => 'images/logo.png',
            'gradient_start' => '#584fbb',
            'gradient_end' => '#584fbb',
            'description' => 'Advanced trading platform with professional features',
            'volume_24h' => 890000000,
            'rating' => 4.7,
            'features' => json_encode(['Futures', 'Margin Trading', 'Staking', 'OTC']),
            'is_hot' => 0,
            'has_ai' => 0,
            'website_url' => 'https://kraken.com',
            'api_supported' => 1
        ],
        [
            'id' => generateUUID(),
            'name' => 'Bybit',
            'logo_path' => 'images/logo.png',
            'gradient_start' => '#f0b90b',
            'gradient_end' => '#f0b90b',
            'description' => 'Derivatives trading specialist with high leverage',
            'volume_24h' => 1200000000,
            'rating' => 4.6,
            'features' => json_encode(['Futures', 'Options', 'High Leverage', 'Copy Trading']),
            'is_hot' => 1,
            'has_ai' => 1,
            'website_url' => 'https://bybit.com',
            'api_supported' => 1
        ],
        [
            'id' => generateUUID(),
            'name' => 'KuCoin',
            'logo_path' => 'images/kucoin.png',
            'gradient_start' => '#26a17b',
            'gradient_end' => '#26a17b',
            'description' => 'Global cryptocurrency exchange with wide altcoin selection',
            'volume_24h' => 650000000,
            'rating' => 4.5,
            'features' => json_encode(['Altcoins', 'Trading Bot', 'Staking', 'Lending']),
            'is_hot' => 0,
            'has_ai' => 0,
            'website_url' => 'https://kucoin.com',
            'api_supported' => 1
        ],
        [
            'id' => generateUUID(),
            'name' => 'OKX',
            'logo_path' => 'images/okb.png',
            'gradient_start' => '#000000',
            'gradient_end' => '#1a1a1a',
            'description' => 'Leading crypto exchange with advanced trading tools',
            'volume_24h' => 950000000,
            'rating' => 4.7,
            'features' => json_encode(['Futures', 'Options', 'DeFi', 'Earn']),
            'is_hot' => 1,
            'has_ai' => 1,
            'website_url' => 'https://okx.com',
            'api_supported' => 1
        ],
        [
            'id' => generateUUID(),
            'name' => 'MetaMask',
            'logo_path' => 'images/logo.png',
            'gradient_start' => '#f6851b',
            'gradient_end' => '#f6851b',
            'description' => 'Leading Web3 wallet and DeFi gateway',
            'volume_24h' => 450000000,
            'rating' => 4.8,
            'features' => json_encode(['DeFi', 'NFTs', 'DApp Browser', 'Swap']),
            'is_hot' => 1,
            'has_ai' => 0,
            'website_url' => 'https://metamask.io',
            'api_supported' => 0
        ],
        [
            'id' => generateUUID(),
            'name' => 'Trust Wallet',
            'logo_path' => 'images/trust.png',
            'gradient_start' => '#3375bb',
            'gradient_end' => '#3375bb',
            'description' => 'Multi-chain mobile cryptocurrency wallet',
            'volume_24h' => 320000000,
            'rating' => 4.7,
            'features' => json_encode(['Mobile', 'Staking', 'DApps', 'Multi-chain']),
            'is_hot' => 0,
            'has_ai' => 0,
            'website_url' => 'https://trustwallet.com',
            'api_supported' => 0
        ],
        [
            'id' => generateUUID(),
            'name' => 'Phantom',
            'logo_path' => 'images/logo.png',
            'gradient_start' => '#ab9ff2',
            'gradient_end' => '#ab9ff2',
            'description' => 'Solana ecosystem wallet with NFT support',
            'volume_24h' => 280000000,
            'rating' => 4.9,
            'features' => json_encode(['Solana', 'NFTs', 'Fast', 'Staking']),
            'is_hot' => 1,
            'has_ai' => 0,
            'website_url' => 'https://phantom.app',
            'api_supported' => 0
        ],
        [
            'id' => generateUUID(),
            'name' => 'Huobi',
            'logo_path' => 'images/huobi.png',
            'gradient_start' => '#1890ff',
            'gradient_end' => '#1890ff',
            'description' => 'Global digital asset exchange platform',
            'volume_24h' => 580000000,
            'rating' => 4.4,
            'features' => json_encode(['Spot', 'Futures', 'Staking', 'Earn']),
            'is_hot' => 0,
            'has_ai' => 0,
            'website_url' => 'https://huobi.com',
            'api_supported' => 1
        ],
        [
            'id' => generateUUID(),
            'name' => 'Gate.io',
            'logo_path' => 'images/logo.png',
            'gradient_start' => '#c99400',
            'gradient_end' => '#c99400',
            'description' => 'Professional cryptocurrency exchange',
            'volume_24h' => 420000000,
            'rating' => 4.5,
            'features' => json_encode(['Spot', 'Futures', 'Options', 'Lending']),
            'is_hot' => 0,
            'has_ai' => 0,
            'website_url' => 'https://gate.io',
            'api_supported' => 1
        ],
        [
            'id' => generateUUID(),
            'name' => 'Bitfinex',
            'logo_path' => 'images/logo.png',
            'gradient_start' => '#4fa8f0',
            'gradient_end' => '#4fa8f0',
            'description' => 'Advanced cryptocurrency trading platform',
            'volume_24h' => 380000000,
            'rating' => 4.3,
            'features' => json_encode(['Margin', 'Lending', 'Derivatives', 'OTC']),
            'is_hot' => 0,
            'has_ai' => 0,
            'website_url' => 'https://bitfinex.com',
            'api_supported' => 1
        ],
        [
            'id' => generateUUID(),
            'name' => 'Crypto.com',
            'logo_path' => 'images/logo.png',
            'gradient_start' => '#103f68',
            'gradient_end' => '#103f68',
            'description' => 'Complete crypto ecosystem with exchange and card',
            'volume_24h' => 720000000,
            'rating' => 4.6,
            'features' => json_encode(['Exchange', 'Card', 'DeFi', 'NFT']),
            'is_hot' => 1,
            'has_ai' => 1,
            'website_url' => 'https://crypto.com',
            'api_supported' => 1
        ],
        [
            'id' => generateUUID(),
            'name' => 'Gemini',
            'logo_path' => 'images/logo.png',
            'gradient_start' => '#00d9ff',
            'gradient_end' => '#00d9ff',
            'description' => 'Regulated US cryptocurrency exchange',
            'volume_24h' => 290000000,
            'rating' => 4.5,
            'features' => json_encode(['Regulated', 'Secure', 'Institutional', 'Insurance']),
            'is_hot' => 0,
            'has_ai' => 0,
            'website_url' => 'https://gemini.com',
            'api_supported' => 1
        ],
        [
            'id' => generateUUID(),
            'name' => 'FTX',
            'logo_path' => 'images/logo.png',
            'gradient_start' => '#02d6ed',
            'gradient_end' => '#02d6ed',
            'description' => 'Advanced derivatives and spot trading',
            'volume_24h' => 0,
            'rating' => 3.5,
            'features' => json_encode(['Derivatives', 'Spot', 'Options']),
            'is_hot' => 0,
            'has_ai' => 0,
            'is_active' => 0,
            'website_url' => 'https://ftx.com',
            'api_supported' => 0
        ],
        [
            'id' => generateUUID(),
            'name' => 'Exodus',
            'logo_path' => 'images/logo.png',
            'gradient_start' => '#7d3cf8',
            'gradient_end' => '#7d3cf8',
            'description' => 'Multi-asset cryptocurrency wallet',
            'volume_24h' => 150000000,
            'rating' => 4.6,
            'features' => json_encode(['Multi-asset', 'Staking', 'Exchange', 'Portfolio']),
            'is_hot' => 0,
            'has_ai' => 0,
            'website_url' => 'https://exodus.com',
            'api_supported' => 0
        ],
        [
            'id' => generateUUID(),
            'name' => 'Ledger',
            'logo_path' => 'images/logo.png',
            'gradient_start' => '#000000',
            'gradient_end' => '#1a1a1a',
            'description' => 'Hardware wallet for maximum security',
            'volume_24h' => 0,
            'rating' => 4.9,
            'features' => json_encode(['Hardware', 'Secure', 'Multi-chain', 'NFT']),
            'is_hot' => 1,
            'has_ai' => 0,
            'website_url' => 'https://ledger.com',
            'api_supported' => 0
        ],
        [
            'id' => generateUUID(),
            'name' => 'Coinbase Pro',
            'logo_path' => 'images/logo.png',
            'gradient_start' => '#0052ff',
            'gradient_end' => '#0052ff',
            'description' => 'Advanced trading platform by Coinbase',
            'volume_24h' => 850000000,
            'rating' => 4.7,
            'features' => json_encode(['Advanced', 'API', 'Institutional', 'Low Fees']),
            'is_hot' => 0,
            'has_ai' => 1,
            'website_url' => 'https://pro.coinbase.com',
            'api_supported' => 1
        ],
        [
            'id' => generateUUID(),
            'name' => 'Uniswap',
            'logo_path' => 'images/unus.png',
            'gradient_start' => '#ff007a',
            'gradient_end' => '#ff007a',
            'description' => 'Decentralized exchange protocol',
            'volume_24h' => 1100000000,
            'rating' => 4.8,
            'features' => json_encode(['DEX', 'DeFi', 'Liquidity', 'Governance']),
            'is_hot' => 1,
            'has_ai' => 0,
            'website_url' => 'https://uniswap.org',
            'api_supported' => 0
        ],
        [
            'id' => generateUUID(),
            'name' => 'PancakeSwap',
            'logo_path' => 'images/logo.png',
            'gradient_start' => '#1fc7d4',
            'gradient_end' => '#1fc7d4',
            'description' => 'BSC-based decentralized exchange',
            'volume_24h' => 680000000,
            'rating' => 4.6,
            'features' => json_encode(['DEX', 'BSC', 'Farming', 'Lottery']),
            'is_hot' => 1,
            'has_ai' => 0,
            'website_url' => 'https://pancakeswap.finance',
            'api_supported' => 0
        ],
    ];

    foreach ($exchanges as $exchange) {
        try {
            // Check if exchange already exists
            $checkStmt = $pdo->prepare("SELECT id FROM exchanges WHERE name = ?");
            $checkStmt->execute([$exchange['name']]);
            if ($checkStmt->fetch()) {
                echo "Exchange {$exchange['name']} already exists, skipping...\n";
                continue;
            }

            $stmt = $pdo->prepare("
                INSERT INTO exchanges 
                (id, name, logo_path, gradient_start, gradient_end, description, volume_24h, rating, features, is_active, is_hot, has_ai, website_url, api_supported)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            ");

            $stmt->execute([
                $exchange['id'],
                $exchange['name'],
                $exchange['logo_path'],
                $exchange['gradient_start'],
                $exchange['gradient_end'],
                $exchange['description'],
                $exchange['volume_24h'],
                $exchange['rating'],
                $exchange['features'],
                isset($exchange['is_active']) ? (int)$exchange['is_active'] : 1,
                isset($exchange['is_hot']) ? (int)$exchange['is_hot'] : 0,
                isset($exchange['has_ai']) ? (int)$exchange['has_ai'] : 0,
                $exchange['website_url'],
                isset($exchange['api_supported']) ? (int)$exchange['api_supported'] : 0
            ]);
            echo "Inserted exchange: {$exchange['name']}\n";
        } catch (PDOException $e) {
            if (strpos($e->getMessage(), 'UNIQUE constraint') !== false) {
                echo "Exchange {$exchange['name']} already exists, skipping...\n";
            } else {
                echo "Error inserting {$exchange['name']}: " . $e->getMessage() . "\n";
            }
        }
    }

    echo "\n✓ Successfully created exchanges table and seeded data.\n";

} catch (PDOException $e) {
    echo "Error: " . $e->getMessage() . "\n";
    exit(1);
}
?>
