<?php
require_once __DIR__ . '/../config/cors.php';
require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../utils/helpers.php';
require_once __DIR__ . '/../utils/email-templates.php';
require_once __DIR__ . '/../utils/mailer.php';

$token = getAuthToken();
$user = getUserFromToken($token);

if (!$user) {
    errorResponse('Unauthorized', 401);
}

$method = $_SERVER['REQUEST_METHOD'];

if ($method === 'POST') {
    $data = getJsonInput();

    // Validate inputs
    if (!isset($data['address']) || !isset($data['amount']) || !isset($data['symbol'])) {
        errorResponse('Missing required fields: address, amount, symbol');
    }

    $recipientAddress = trim($data['address']);
    $amount = floatval($data['amount']);
    $symbol = strtoupper($data['symbol']);
    $senderId = $user['id'];

    if ($amount <= 0) {
        errorResponse('Amount must be positive');
    }

    try {
        $pdo->beginTransaction();

        // 1. Debit Sender (Atomic)
        $stmt = $pdo->prepare("UPDATE user_assets SET balance = balance - ? WHERE user_id = ? AND symbol = ? AND balance >= ?");
        $stmt->execute([$amount, $senderId, $symbol, $amount]);
        
        if ($stmt->rowCount() === 0) {
            throw new Exception("Insufficient balance or asset not found");
        }

        // 2. Find Recipient
        $stmt = $pdo->prepare("SELECT id, email, wallet_address FROM users WHERE wallet_address = ?");
        $stmt->execute([$recipientAddress]);
        $recipient = $stmt->fetch();

        if (!$recipient) {
            throw new Exception("Recipient wallet address not found");
        }

        if ($recipient['id'] == $senderId) {
            throw new Exception("Cannot send to yourself");
        }

        $recipientId = $recipient['id'];

        // 3. Credit Recipient (Upsert)
        $stmt = $pdo->prepare("SELECT balance FROM user_assets WHERE user_id = ? AND symbol = ?");
        $stmt->execute([$recipientId, $symbol]);
        $recipientAsset = $stmt->fetch();

        if ($recipientAsset) {
            $stmt = $pdo->prepare("UPDATE user_assets SET balance = balance + ? WHERE user_id = ? AND symbol = ?");
            $stmt->execute([$amount, $recipientId, $symbol]);
        } else {
            $stmt = $pdo->prepare("INSERT INTO user_assets (user_id, symbol, name, balance, color, icon) VALUES (?, ?, ?, ?, ?, ?)");
            $stmt->execute([$recipientId, $symbol, $symbol, $amount, 'from-gray-500 to-gray-600', 'Coins']);
        }

        // 4. Record Transaction
        $txId = generateUUID();
        $txHash = '0x' . bin2hex(random_bytes(32));
        $stmt = $pdo->prepare("INSERT INTO transactions (id, sender_id, receiver_id, amount, symbol, status, tx_hash) VALUES (?, ?, ?, ?, ?, 'completed', ?)");
        $stmt->execute([$txId, $senderId, $recipientId, $amount, $symbol, $txHash]);

        // 5. Get New Balance for Response
        $stmt = $pdo->prepare("SELECT balance FROM user_assets WHERE user_id = ? AND symbol = ?");
        $stmt->execute([$senderId, $symbol]);
        $newSenderBalance = floatval($stmt->fetch()['balance']);

        $pdo->commit();

        // 6. Send Emails
        $date = date('M d, Y');
        
        // Email to Sender
        $senderContent = "
            <div style='text-align: center; margin-bottom: 30px;'>
                 <div style='width: 60px; height: 60px; background-color: #dcfce7; border-radius: 50%; display: flex; align-items: center; justify-content: center; margin: 0 auto 15px auto;'>
                    <span style='color: #16a34a; font-size: 30px;'>✓</span>
                 </div>
                 <h2 style='color: #16a34a; margin: 0;'>Transaction Successful</h2>
            </div>
            <p>You successfully sent <strong>$amount $symbol</strong>.</p>
            <div style='background-color: #f4f4f5; padding: 20px; border-radius: 12px; margin: 20px 0;'>
                <table style='width: 100%; border-collapse: collapse;'>
                    <tr>
                        <td style='color: #71717a; padding: 8px 0;'>Recipient:</td>
                        <td style='text-align: right; font-family: monospace;'>User " . substr($recipientAddress, 0, 6) . "..." . substr($recipientAddress, -4) . "</td>
                    </tr>
                    <tr>
                        <td style='color: #71717a; padding: 8px 0;'>Transaction ID:</td>
                        <td style='text-align: right; font-family: monospace;'>" . substr($txId, 0, 8) . "</td>
                    </tr>
                    <tr>
                        <td style='color: #71717a; padding: 8px 0;'>Date:</td>
                        <td style='text-align: right;'>$date</td>
                    </tr>
                    <tr style='border-top: 1px solid #e4e4e7;'>
                        <td style='padding-top: 15px; font-weight: bold;'>New Balance:</td>
                        <td style='padding-top: 15px; text-align: right; font-weight: bold;'>$newSenderBalance $symbol</td>
                    </tr>
                </table>
            </div>
        ";
        
        $senderEmailBody = getBrandedTemplate($senderContent, 'Transaction Sent');
        sendEmail($user['email'], "Transaction Sent: $amount $symbol", $senderEmailBody);

        // Email to Recipient
        $recipientContent = "
            <div style='text-align: center; margin-bottom: 30px;'>
                 <h2 style='color: #8b5cf6; margin: 0;'>Payment Received!</h2>
                 <p style='color: #71717a;'>Good news! You just received crypto.</p>
            </div>
            <div style='text-align: center; margin: 30px 0;'>
                <div style='font-size: 36px; font-weight: 800; color: #16a34a;'>+$amount $symbol</div>
            </div>
            <div style='background-color: #f4f4f5; padding: 20px; border-radius: 12px; margin: 20px 0;'>
                <p style='margin: 0; color: #71717a; font-size: 14px;'>From Wallet:</p>
                <p style='margin: 5px 0 0 0; font-family: monospace; font-weight: 600;'>User " . substr($user['wallet_address'] ?? 'Unknown', 0, 6) . "...</p>
            </div>
            <div style='text-align: center; margin-top: 30px;'>
                <a href='#' style='display: inline-block; padding: 12px 24px; background-color: #8b5cf6; color: white; text-decoration: none; border-radius: 8px; font-weight: 600;'>View Dashboard</a>
            </div>
        ";
        $recipientEmailBody = getBrandedTemplate($recipientContent, "You Received $amount $symbol!");
        sendEmail($recipient['email'], "You Received $amount $symbol!", $recipientEmailBody);

        jsonResponse([
            'message' => 'Transaction successful',
            'tx_hash' => $txHash,
            'new_balance' => $newSenderBalance
        ]);

    } catch (Exception $e) {
        $pdo->rollBack();
        errorResponse($e->getMessage(), 400);
    }
} else {
    errorResponse('Method not allowed', 405);
}
?>
