<?php
require_once '../config/cors.php';
require_once '../config/database.php';
require_once '../utils/helpers.php';

// Get JSON input
$input = json_decode(file_get_contents('php://input'), true);

// Extract authorization token
$token = getAuthToken();

// Validate token
$user = getUserFromToken($token);
if (!$user) {
    errorResponse('User not authenticated', 401);
}

// Check if user is admin
if (!isAdmin($user['id'])) {
    errorResponse('Unauthorized: Admin access required', 403);
}

// Validate action
if (!isset($input['action'])) {
    errorResponse('Action is required', 400);
}

$action = $input['action'];

try {
    switch ($action) {
        case 'list':
            // Get orders with user names
            $stmt = $pdo->query("
                SELECT o.*, u.email as user_email 
                FROM orders o 
                LEFT JOIN users u ON o.user_id = u.id 
                ORDER BY o.created_at DESC
            ");
            $orders = $stmt->fetchAll();
            
            // For each order, get its items
            foreach ($orders as &$order) {
                $itemStmt = $pdo->prepare("SELECT * FROM order_items WHERE order_id = ?");
                $itemStmt->execute([$order['id']]);
                $order['items'] = $itemStmt->fetchAll();
                
                // Decode address and notes if needed
                if ($order['billing_address']) {
                    $order['billing_address'] = json_decode($order['billing_address'], true);
                }
            }
            
            jsonResponse(['orders' => $orders], 200);
            break;
            
        case 'get':
            if (!isset($input['id'])) {
                errorResponse('Order ID is required', 400);
            }
            
            $stmt = $pdo->prepare("
                SELECT o.*, u.email as user_email 
                FROM orders o 
                LEFT JOIN users u ON o.user_id = u.id 
                WHERE o.id = ?
            ");
            $stmt->execute([$input['id']]);
            $order = $stmt->fetch();
            
            if (!$order) {
                errorResponse('Order not found', 404);
            }
            
            // Get items
            $itemStmt = $pdo->prepare("SELECT * FROM order_items WHERE order_id = ?");
            $itemStmt->execute([$order['id']]);
            $order['items'] = $itemStmt->fetchAll();
            
            if ($order['billing_address']) {
                $order['billing_address'] = json_decode($order['billing_address'], true);
            }
            
            jsonResponse(['order' => $order], 200);
            break;
            
        case 'update_status':
            if (!isset($input['id']) || !isset($input['status'])) {
                errorResponse('ID and status are required', 400);
            }
            
            $id = $input['id'];
            $status = $input['status'];
            $payment_status = isset($input['payment_status']) ? $input['payment_status'] : null;
            
            $query = "UPDATE orders SET status = ?, updated_at = datetime('now')";
            $params = [$status];
            
            if ($payment_status) {
                $query .= ", payment_status = ?";
                $params[] = $payment_status;
            }
            
            $query .= " WHERE id = ?";
            $params[] = $id;
            
            $stmt = $pdo->prepare($query);
            $stmt->execute($params);
            
            // Log activity
            debugLog("Order $id status updated to $status by admin " . $user['id']);
            
            jsonResponse(['message' => 'Order updated successfully'], 200);
            break;
            
        case 'delete':
            if (!isset($input['id'])) {
                errorResponse('Order ID is required', 400);
            }
            
            $id = $input['id'];
            
            // Table order_items has ON DELETE CASCADE in SQLite schema if defined, 
            // but let's be explicit if needed or trust the schema.
            $stmt = $pdo->prepare("DELETE FROM orders WHERE id = ?");
            $stmt->execute([$id]);
            
            jsonResponse(['message' => 'Order deleted successfully'], 200);
            break;
            
        default:
            errorResponse('Invalid action', 400);
    }
} catch (Exception $e) {
    errorResponse('Error managing orders: ' . $e->getMessage(), 500);
}
?>
