<?php
require_once __DIR__ . '/../config/cors.php';
require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../utils/helpers.php';
require_once __DIR__ . '/../utils/email-templates.php';
require_once __DIR__ . '/../utils/mailer.php';

$token = getAuthToken();
$user = getUserFromToken($token);

if (!$user) {
    errorResponse('Unauthorized', 401);
}

$input = json_decode(file_get_contents('php://input'), true);

// Validate required fields
$requiredFields = ['exchange', 'amount', 'currency', 'receiverEmail'];
validateInput($input, $requiredFields);

// Extract Data
$exchange = $input['exchange'];
$amount = $input['amount'];
$currency = $input['currency'];
$receiverEmail = $input['receiverEmail'];
$senderEmail = isset($input['senderEmail']) ? $input['senderEmail'] : $user['email'];
$txHash = isset($input['transactionHash']) && !empty($input['transactionHash']) ? $input['transactionHash'] : '0x' . bin2hex(random_bytes(32));
$walletAddress = isset($input['walletAddress']) ? $input['walletAddress'] : '';
$network = isset($input['network']) ? $input['network'] : '';
$notes = isset($input['notes']) ? $input['notes'] : '';
$txId = generateUUID();

try {
    $pdo->beginTransaction();

    // 1. Try to find receiver user ID
    $stmt = $pdo->prepare("SELECT id FROM users WHERE email = ?");
    $stmt->execute([$receiverEmail]);
    $receiver = $stmt->fetch();
    // If receiver not found (external email), fallback to sender_id to satisfy NOT NULL constraint
    // This allows users to "send" to external emails while keeping a record linked to their account
    $receiverId = $receiver ? $receiver['id'] : $user['id']; 

    // 2. Insert into transactions table
    $insertStmt = $pdo->prepare("INSERT INTO transactions (id, sender_id, receiver_id, amount, symbol, status, tx_hash, exchange) VALUES (?, ?, ?, ?, ?, ?, ?, ?)");
    $insertStmt->execute([
        $txId,
        $user['id'],
        $receiverId,
        $amount,
        $currency,
        'completed',
        $txHash,
        $exchange
    ]);

    // 3. Prepare Email Content
    $receiptContent = "
        <h2 style='text-align: center; color: #8b5cf6;'>Transaction Receipt</h2>
        <div style='background-color: #f9fafb; padding: 20px; border-radius: 8px; margin: 20px 0;'>
            <table style='width: 100%; border-collapse: collapse;'>
                <tr>
                    <td style='padding: 10px; color: #6b7280;'>Transaction ID:</td>
                    <td style='padding: 10px; font-family: monospace; text-align: right;'>$txId</td>
                </tr>
                <tr>
                    <td style='padding: 10px; color: #6b7280;'>Amount:</td>
                    <td style='padding: 10px; font-weight: bold; text-align: right; font-size: 18px;'>$amount $currency</td>
                </tr>
                <tr>
                    <td style='padding: 10px; color: #6b7280;'>Exchange:</td>
                    <td style='padding: 10px; text-align: right;'>$exchange</td>
                </tr>
                <tr>
                    <td style='padding: 10px; color: #6b7280;'>Status:</td>
                    <td style='padding: 10px; text-align: right; color: #10b981; font-weight: bold;'>Completed</td>
                </tr>
                <tr>
                    <td style='padding: 10px; color: #6b7280;'>Date:</td>
                    <td style='padding: 10px; text-align: right;'>" . date('M j, Y H:i') . "</td>
                </tr>
            </table>
        </div>
        <div style='text-align: center; margin-top: 20px;'>
            <p style='color: #6b7280; font-size: 14px;'>Transaction Hash:</p>
            <p style='font-family: monospace; background: #f3f4f6; padding: 10px; border-radius: 6px; word-break: break-all;'>$txHash</p>
        </div>
    ";

    // 4. Send Emails
    $subject = "Flash Transaction Receipt: $amount $currency";
    $htmlBody = getBrandedTemplate($receiptContent, $subject);
    
    // Send to Receiver
    sendRobustEmail(
        $receiverEmail,
        $subject,
        $htmlBody
    );

    // Send to Sender (Receipt)
    if ($senderEmail && $senderEmail !== $receiverEmail) {
        sendRobustEmail(
            $senderEmail,
            $subject,
            $htmlBody
        );
    }

    $pdo->commit();

    jsonResponse([
        'success' => true, 
        'message' => 'Flash transaction processed successfully',
        'transaction' => [
            'id' => $txId,
            'status' => 'completed',
            'hash' => $txHash,
            'timestamp' => date('c')
        ]
    ]);

} catch (Exception $e) {
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    errorResponse('Transaction failed: ' . $e->getMessage(), 500);
}
?>
